<?php
/**
 * AutoLoginEnterprise.php
 *
 * PHP 5.4
 *
 * Usage:
 * require_once 'Secioss/autoload.php';
 * use Secioss;
 * $login = new AutoLoginEnterprise('LDAP', []);
 * $login->auth($password);
 *
 * @package Auth
 *
 * @author Kaoru Sekiguchi <sekiguchi.kaoru@secioss.co.jp>
 * @copyright 2020 SECIOSS, INC.
 */
namespace Secioss;

use PEAR;

define('ALCOOKIE', 'secioss_autologin');
define('PWD_LEN', 12);
if (strtoupper(substr(PHP_OS, 0, 3)) !== 'WIN') {
    define('WINPWDCMD', '/opt/secioss/sbin/chwinpasswd');
} else {
    define('WINPWDCMD', 'perl /secioss/sbin/chwinpasswd');
}
define('AUTO_LOGIN_ERROR', -1);
define('AUTO_LOGIN_NO_USER', -2);
define('AUTO_LOGIN_NO_PASSWD', -3);
define('AUTO_LOGIN_INVALID_VALUE', -4);
define('AUTO_LOGIN_MIN_AGE', -5);
define('AUTO_LOGIN_IN_HISTORY', -6);
define('AUTO_LOGIN_BAD_PASSWD', -7);
define('AUTO_LOGIN_WEAK_PASSWD', -8);
define('AUTO_LOGIN_WEAK_PASSWD_WARN', -9);
define('AUTO_LOGIN_MIN_PASSWD', -10);
define('AUTO_LOGIN_MAX_PASSWD', -11);
define('AUTO_LOGIN_ALREADY_EXISTS', -12);
define('AUTO_LOGIN_PROHIBIT_ATTR', -13);
define('AUTO_LOGIN_DEVICE_INACTIVE', -100);
define('AUTO_LOGIN_DEVICE_NOEXIST', -101);
define('SECRETKEY_DIRECTIVE', 'TKTAuthSecret');

/**
 * Secioss\AutoLoginEnterprise
 */
class AutoLoginEnterprise extends AutoLoginSimple
{
    /**
     * AutoLoginEnterpriseクラスのコンストラクタ
     *
     * @access public
     *
     * @param string      $driver   ストレージドライバの型
     * @param mixed       $options  ストレージドライバの設定
     * @param null|string $app      アプリケーション名
     * @param null|array  $url      ログイン時にリダイレクトするURL
     * @param null|string $postUser POSTするユーザー名のname
     * @param null|string $postPass POSTするパスワードのname
     * @param null|mixed  $postData POSTするデータ
     * @param null|mixed  $cookie   送信するcookie
     * @param null|mixed  $content  受信したコンテンツに対する設定
     * @param null|mixed  $debug    デバッグ情報を出力するファイル
     *
     * @return mixed 0:正常終了 PEAR_Error:エラー
     */
    public function __construct($driver, $options, $app = null, $url = null, $postUser = null, $postPass = null, $postData = null, $cookie = null, $content = null, $debug = null)
    {
        $this->_setDefaults();
        $this->_parseOptions($options);

        if (isset($options['libs'])) {
            $libs = preg_split('/ /D', $this->options['libs']);
            foreach ($libs as $lib) {
                require_once $lib;
            }
        }
        if (isset($options['domain'])) {
            $this->domain = $options['domain'];
        }
        if (isset($options['sessioncookie'])) {
            $this->sessioncookie = $options['sessioncookie'];
        }
        if (isset($options['redirect'])) {
            $this->redirect = $options['redirect'];
        } else {
            $this->redirect = 'header';
        }
        if (isset($options['otp'])) {
            $this->otp = $options['otp'];
        } else {
            $this->otp = false;
        }

        $this->app = $app;

        if (isset($_GET['url_back'])) {
            $url['back'] = $_GET['url_back'];
        }
        if (!isset($url['fatal'])) {
            $url['fatal'] = '/error/HTTP_INTERNAL_SERVER_ERROR.html.var';
        }
        if (!isset($url['password'])) {
            $url['password'] = '/user/password.php';
        }
        $this->url = $url;

        $this->postUser = $postUser;

        $this->postPass = $postPass;

        if (is_array($postData)) {
            $this->postData = $postData;
        } else {
            $this->postData = [];
        }

        if (is_array($cookie)) {
            $this->cookie = $cookie;
        } else {
            $this->cookie = [];
        }

        $this->content = $content;
        if (isset($this->content['check'])) {
            $this->content['check'] = preg_quote($this->content['check'], '/');
        }

        $this->debug = $debug;

        $storage = $this->_factory($driver);
        if (PEAR::isError($storage)) {
            return $storage;
        }
        $this->storage = $storage;

        return 0;
    }

    /**
     * 自動ログインを行う
     *
     * @access public
     *
     * @return bool true:正常終了 PEAR_Error:エラー
     */
    public function start()
    {
        if (isset($_SERVER['REMOTE_USER_LOGGEDIN'])) {
            $loggedin = true;
            if (isset($this->url['session'])) {
                $loggedin = false;
                $req = new HTTP_Request($this->url['session']);
                $req->addHeader('User-Agent', $_SERVER['HTTP_USER_AGENT']);
                $req->setMethod(HTTP_REQUEST_METHOD_GET);
                foreach ($_COOKIE as $key => $value) {
                    $req->addCookie($key, $value);
                }
                $res = $req->sendRequest();
                if (!PEAR::isError($res)) {
                    $response = $req->getResponseCode();
                    if ($response < 300) {
                        $body = $req->getResponseBody();
                        if (!isset($this->content['sessionfail']) || !preg_match('/'.$this->content['sessionfail'].'/', $body)) {
                            $loggedin = true;
                        }
                    }
                }
            }
            if ($loggedin) {
                $back = isset($this->url['back']) ? $this->url['back'] : $this->url['login'];
                header('Location: '.$back);
                return true;
            }
        }

        if (PEAR::isError($rc = $this->login())) {
            $querystr = null;
            if ($this->app) {
                $querystr = 'ssoapp='.$this->app;
            }
            $errcode = $rc->getCode();
            switch ($errcode) {
                case AUTO_LOGIN_NO_PASSWD:
                    if ($querystr) {
                        $querystr = "$querystr&msg=USS_NML_002";
                    } else {
                        $querystr = 'msg=USS_NML_002';
                    }
                    $this->_redirect('password', $querystr);
                    break;
                case AUTO_LOGIN_BAD_PASSWD:
                    if ($querystr) {
                        $querystr = "$querystr&msg=USS_NML_003";
                    } else {
                        $querystr = 'msg=USS_NML_003';
                    }
                    $this->_redirect('password', $querystr);
                    break;
                default:
                    $this->_redirect('fatal', $querystr);
                    break;
            }

            return $rc;
        }

        return true;
    }

    /**
     * ログインを実行後、リダイレクトする
     *
     * @access public
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function login()
    {
        if (isset($_GET['url_back']) && isset($this->url['check_url_back']) && preg_match('/^https?:\/\//', $_GET['url_back']) && !preg_match('#'.$this->url['check_url_back'].'#', $_GET['url_back'])) {
            return PEAR::raiseError('Bad back url', AUTO_LOGIN_ERROR);
        }
        $loginid = $this->storage->getLoginId();
        if (!$loginid) {
            return PEAR::raiseError('No login id', AUTO_LOGIN_ERROR);
        }
        $this->loginid = $loginid;

        if ($this->otp) {
            $password = Util::random(PWD_LEN);
        } else {
            $password = $this->storage->getPassword($this->app);
            if (PEAR::isError($password)) {
                return $password;
            } elseif (!$password) {
                return PEAR::raiseError('No password', AUTO_LOGIN_NO_PASSWD);
            }
            if (isset($this->options['pwsalt']) && $this->options['pwsalt']) {
                $password .= $this->options['pwsalt'];
                if (isset($this->options['pwmaxlen']) && $this->options['pwmaxlen']) {
                    $password = substr($password, 0, $this->options['pwmaxlen']);
                }
            }
        }
        $this->password = $password;

        if ($this->otp || (isset($this->options['sync']) && $this->options['sync'])) {
            $rc = $this->storage->setPassword($password);
            if (PEAR::isError($rc)) {
                return $rc;
            }
        }

        if (isset($_SERVER['HTTP_X_FORWARDED_FOR']) && $_SERVER['HTTP_X_FORWARDED_FOR']) {
            $ipaddr = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            $ipaddr = $_SERVER['REMOTE_ADDR'];
        }

        $querystr = '';
        foreach ($_GET as $param => $value) {
            if (!isset($this->postData[$param])) {
                $querystr = $querystr ? $querystr.'&'."$param=$value" : "$param=$value";
            }
        }

        $domain = $this->domain;

        if ($this->sessioncookie) {
            $now = time();
            $hmac = hash_hmac('sha1', $now, Crypt::getSecretKey($this->options['keyfile']));
            if (!setcookie($this->sessioncookie, "$now\t$hmac", 0, '/', $domain)) {
                return PEAR::raiseError("Can't set cookie ".$this->sessioncookie, AUTO_LOGIN_ERROR);
            }
        }

        if ($this->redirect == 'post') {
            $this->_postRedirect($this->url['login'], $loginid, $password);
            return true;
        }

        $login_url = preg_split('/, */', $this->url['login']);
        $url = array_shift($login_url);

        $login = false;
        $authenticated = false;
        $oldurl = $url;
        $body = '';
        $cookies = [];
        foreach ($_COOKIE as $key => $value) {
            if ($key == ALCOOKIE) {
                continue;
            }
            $cookies[$key] = ['name' => $key, 'value' => $value];
        }
        foreach ($this->cookie as $key => $value) {
            $value = Util::parsePostval($value);
            $cookies[$key] = ['name' => $key, 'value' => $value];
        }

        $send_cookies = [];
        for ($i = 0; $i < 10; $i++) {
            $req = new HTTP_Request($url);
            $req->addHeader('User-Agent', $_SERVER['HTTP_USER_AGENT']);
            if ($login && !$authenticated) {
                $req->setMethod(HTTP_REQUEST_METHOD_POST);

                $req->addHeader('Referer', isset($this->url['referer']) ? $this->url['referer'] : $oldurl);
                $req->addHeader('X-Forwarded-For', $ipaddr);
                if ($this->postUser) {
                    $req->addPostData($this->postUser, $loginid);
                }
                if ($this->postPass) {
                    $req->addPostData($this->postPass, $password);
                }
                if ($body && preg_match_all('/<input[^>]*type="hidden"[^>]*name="([^">]+)"[^>]*value="([^">]*)"/i', $body, $matches)) {
                    for ($i = 0; $i < count($matches[0]); $i++) {
                        if ($matches[1][$i] == $this->postUser || $matches[1][$i] == $this->postPass) {
                            continue;
                        }
                        $req->addPostData($matches[1][$i], $matches[2][$i]);
                    }
                }
                if ($body && preg_match_all('/<input[^>]*name="([^">]+)"[^>]*type="hidden"[^>]*value="([^">]*)"/i', $body, $matches)) {
                    for ($i = 0; $i < count($matches[0]); $i++) {
                        if ($matches[1][$i] == $this->postUser || $matches[1][$i] == $this->postPass) {
                            continue;
                        }
                        $req->addPostData($matches[1][$i], $matches[2][$i]);
                    }
                }
                foreach ($this->postData as $key => $value) {
                    if ($key == 'query_string' && $value == 'all') {
                        foreach ($_GET as $query_key => $query_val) {
                            $req->addPostData($query_key, $query_val);
                        }
                        continue;
                    }

                    $value = Util::hextostr($value);
                    $value = Util::parsePostval($value, ['loginid' => $this->loginid, 'password' => $this->password]);
                    $key = Util::hextostr($key);
                    $req->addPostData($key, $value);
                }
                $authenticated = true;
            } else {
                $req->setMethod(HTTP_REQUEST_METHOD_GET);
                $req->addHeader('Referer', isset($this->url['referer']) ? $this->url['referer'] : $oldurl);
                $req->addHeader('X-Forwarded-For', $ipaddr);
            }

            foreach ($cookies as $key => $cookie) {
                $req->addCookie($cookie['name'], $cookie['value']);
            }

            for ($i = 0; $i < 5; $i++) {
                $res = $req->sendRequest();
                if (PEAR::isError($res)) {
                    return PEAR::raiseError('Send request failure('.$res->getMessage().')', AUTO_LOGIN_ERROR);
                }

                $response = $req->getResponseCode();
                if ($response >= 500) {
                    if (isset($this->content['return_error']) && $this->content['return_error']) {
                        if (isset($this->url['back'])) {
                            $this->url['back'] = null;
                        }
                    } else {
                        return PEAR::raiseError("Bad response code($response)", AUTO_LOGIN_ERROR);
                    }
                } elseif ($response) {
                    break;
                }
            }

            $rescookies = $req->getResponseCookies();
            if (is_array($rescookies)) {
                foreach ($rescookies as $rescookie) {
                    $send_cookies[$rescookie['name']] = $rescookie;
                    $cookies[$rescookie['name']] = $rescookie;
                }
            }

            $body = $req->getResponseBody();
            $headers = $req->getResponseHeader();

            if ($this->debug) {
                file_put_contents($this->debug, date('M d H:i:s Y')." $url $response\n", FILE_APPEND);
                file_put_contents($this->debug, $headers, FILE_APPEND);
                file_put_contents($this->debug, "\n$body\n\n", FILE_APPEND);
            }

            $oldurl = $url;
            if (!$login && isset($headers['location'])) {
                $url = $this->_constructUrl($headers['location'], $url);
            } elseif (!$login && preg_match('/window\.location\.href = "([^"]+)"/', $body, $matches)) {
                $url = $this->_constructUrl($matches[1], $url);
            } elseif ($login) {
                if (isset($this->content['authfail']) &&
                    preg_match('/'.$this->content['authfail'].'/', $body)) {
                    return PEAR::raiseError('Authentication failure', AUTO_LOGIN_BAD_PASSWD);
                }
                if (isset($this->content['check']) &&
                    !preg_match('/'.$this->content['check'].'/', $body)) {
                    if (isset($this->content['return_error']) && $this->content['return_error']) {
                        if (isset($this->url['back'])) {
                            $this->url['back'] = null;
                        }
                    } else {
                        return PEAR::raiseError('Bad content', AUTO_LOGIN_ERROR);
                    }
                }
                break;
            } elseif (preg_match('/<form .*action="([^"]+)".*>.*name="'.$this->postUser.($this->postPass ? '".*name="'.$this->postPass.'"' : '').'/si', $body, $matches)) {
                $url = $this->_constructUrl($matches[1], $url);
                if (preg_match('/\?/', $url)) {
                    $url = $url.'&'.$querystr;
                } else {
                    $url = $url.'?'.$querystr;
                }
                $login = true;
            } else {
                if (count($login_url)) {
                    $url = array_shift($login_url);
                } else {
                    $login = true;
                }
            }
        }

        $appcookies = isset($_COOKIE[ALCOOKIE]) ? preg_split('/#/', $_COOKIE[ALCOOKIE]) : [];
        if (is_array($send_cookies)) {
            foreach ($send_cookies as $key => $cookie) {
                if ($cookie['name'] == 'auth_tkt' || $cookie['name'] == 'SimpleSAMLSessionID') {
                    continue;
                }
                if (preg_match('/[,; \t\r\n]/', $cookie['value'])) {
                    $cookie['value'] = base64_encode($cookie['value']);
                }
                if (isset($cookie['path'])) {
                    if (isset($this->options['rewrite_cookiepath']) && $this->options['rewrite_cookiepath']) {
                        list($match, $substitute) = preg_split('/, */', $this->options['rewrite_cookiepath']);
                        $cookiepath = preg_replace('#^'.$match.'$#', $substitute, $cookie['path']);
                    } elseif (!isset($this->url['back']) || preg_match('#^'.$cookie['path'].'#', preg_replace('#^https?://[^/]+#', '', $this->url['back']))) {
                        $cookiepath = $cookie['path'];
                    } else {
                        $cookiepath = '/';
                    }
                } else {
                    $cookiepath = '/';
                }
                if (!setrawcookie($cookie['name'], $cookie['value'], isset($cookie['expires']) ? strtotime($cookie['expires']) : 0, $cookiepath, $domain, isset($cookie['secure']) ? $cookie['secure'] : false)) {
                    return PEAR::raiseError("Can't set cookie ".$cookie['name'], AUTO_LOGIN_ERROR);
                }
                if (isset($this->options['cookiename']) && $this->options['cookiename'] == $cookie['name']) {
                    array_push($appcookies, $cookie['name']."=$cookiepath");
                }
            }
        }
        if (count($appcookies)) {
            setcookie(ALCOOKIE, join('#', $appcookies), 0, null, $domain);
        }

        if (isset($this->url['back'])) {
            $back = $this->url['back'];
            if (preg_match('/%{([^}]+)}/', $back, $matches) && isset($_GET[$matches[1]])) {
                $back = preg_replace('/%{'.$matches[1].'}/', $_GET[$matches[1]], $back);
            }
            header('Location: '.$back);
        } elseif (isset($this->url['webdav'])) {
            $req->reset($this->url['webdav']);
            $req->setMethod(HTTP_REQUEST_METHOD_OPTIONS);
            $req->sendRequest();
            echo $req->getResponseBody();
        } elseif (isset($headers['location'])) {
            header('Location: '.$this->_constructUrl($headers['location'], $url));
        } else {
            echo $body;
        }
        return true;
    }

    // ========================================================================
    //
    // ========================================================================

    /**
     * ユーザーの認証を行う
     *
     * appcookieid.php
     * appradiusid.php
     * cookieid.php
     * user/index.php
     * mobileid.php
     * password.php
     * secret.php
     *
     * @access public
     *
     * @param string パスワード
     * @param mixed $password
     *
     * @return string 0:成功 1:失敗
     */
    public function auth($password)
    {
        $rc = $this->storage->auth($password);
        if (!$rc) {
            $status = $this->storage->getStatus();
            if ($status && $status != 'active') {
                $rc = 2;
            }
        }
        if (!$rc) {
            $lockout = $this->storage->getPwdLockout();
            if ($lockout) {
                $rc = 3;
            }
        }

        return $rc;
    }

    /**
     * パスワードの妥当性を確認する
     *
     * storage->validatePasswd
     * Secioss_User.php
     *
     * @access public
     *
     * @param string パスワード
     * @param mixed $password
     *
     * @return mixed 0:正常終了
     */
    public function validatePasswd($password)
    {
        $pwlen = strlen($password);

        if ($pwlen < $this->options['pwminlen']) {
            return AUTO_LOGIN_MIN_PASSWD;
        }

        if ($pwlen > $this->options['pwmaxlen']) {
            return AUTO_LOGIN_MAX_PASSWD;
        }

        if ($this->options['pwallow']) {
            $pwallows = Util::to_array($this->options['pwallow']);
            $pwallowlimit = count($pwallows);
            if ($this->options['pwallowlimit']) {
                $pwallowlimit = $this->options['pwallowlimit'];
            }
            $match = 0;
            foreach ($pwallows as $pwallow) {
                if (preg_match("/$pwallow/", $password)) {
                    $match++;
                }
            }
            if ($match < $pwallowlimit) {
                return AUTO_LOGIN_INVALID_VALUE;
            }
        }

        if ($this->options['pwdeny']) {
            $pwdenys = Util::to_array($this->options['pwdeny']);
            foreach ($pwdenys as $pwdeny) {
                if (preg_match("/$pwdeny/", $password)) {
                    return AUTO_LOGIN_INVALID_VALUE;
                }
            }
        }

        if ($this->options['pwminage'] && !$this->isPwdMustChange()) {
            $pwdtime = $this->getPwdChangedTime();
            if ($pwdtime && time() < $pwdtime + $this->options['pwminage']) {
                return AUTO_LOGIN_MIN_AGE;
            }
        }

        if ($this->options['pwinhistory']) {
            $pwdhistory = $this->getPwdHistory();

            for ($i = 0; $i < $this->options['pwinhistory'] && $i < count($pwdhistory); $i++) {
                $pwhash = null;
                $oldpasswd = $pwdhistory[$i];
                if (preg_match('/^{([^}]+)}(.+)$/', $pwdhistory[$i], $matches)) {
                    $pwhash = $matches[1];
                    $oldpasswd = '{'.$pwhash.'}'.$matches[2];
                }
                if (!$pwhash) {
                    $pwhash = $this->options['pwhash'];
                }
                if (Util::cmpPasswd($password, $oldpasswd, $pwhash)) {
                    return AUTO_LOGIN_IN_HISTORY;
                }
            }
        }

        if ($this->options['pwcheckfuncs']) {
            foreach (explode(' ', $this->options['pwcheckfuncs']) as $func) {
                $rc = $func($password, $this->id);
                if ($rc) {
                    return $rc;
                }
            }
        }

        if ($this->options['pwprohibitattr']) {
            $phoneignore = [' ', '-', '+', '#', '*'];
            foreach (explode(',', $this->options['pwprohibitattr']) as $attr) {
                switch ($attr) {
                    case 'uid':
                        $uid = strstr($this->prop[$attr], '@', true);
                        $uid = $uid ? $uid : $this->prop[$attr];
                        if ($this->_pwdInAttr($password, $uid, "$attr ユーザーID", 0)) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'employeenumber':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 社員番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'cn':
                        if ($this->_pwdInAttr($password, $this->prop['sn'], 'sn 氏名')) { //姓
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        if ($this->_pwdInAttr($password, $this->prop['givenname'], 'gn 氏名')) { //名
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'displayname':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 別名")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'mail':
                        $mail = strstr($this->prop['mail'], '@', true);
                        if ($this->_pwdInAttr($password, $mail, "$attr メールアドレス")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        // TODO seciossmailalias
                        break;
                    case 'seciossnotificationmail':
                        $seciossnotificationmail = strstr($this->prop['seciossnotificationmail'], '@', true);
                        if ($this->_pwdInAttr($password, $seciossnotificationmail, "$attr 通知用メールアドレス")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'seciosstelephonenumber':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 電話番号") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr 電話番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'seciossfax':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr FAX") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr FAX")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'seciossmobile':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 携帯電話番号") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr 携帯電話番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'seciosshomephone':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 自宅電話番号") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr 自宅電話番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'pager':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr ポケベル番号") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr ポケベル番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    default:
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr $attr")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                }
            }
        }

        if ($this->options['pwstrength']) {
            $level = $this->options['pwstrength'];

            if (preg_match('/^5\.[1|4]\./', PHP_VERSION)) {
                $dict = crack_opendict('/usr/share/cracklib/pw_dict');
                if (!$dict) {
                    return AUTO_LOGIN_ERROR;
                }

                $check = crack_check($dict, $password);
                $message = crack_getlastmessage();
            } else {
                exec('echo '.escapeshellarg($password).' | /usr/sbin/cracklib-check 2>/dev/null', $output, $rc);
                if ($rc || !preg_match('/^.*: (.+)$/', $output[0], $matches)) {
                    return AUTO_LOGIN_ERROR;
                }
                $message = $matches[1];
            }
            if ($message != 'strong password' && $message != 'OK') {
                if (preg_match('/^5\.[1|4]\./', PHP_VERSION)) {
                    crack_closedict($dict);
                }
                $this->error = $message;
                switch ($level) {
                    case 1:
                        return AUTO_LOGIN_WEAK_PASSWD_WARN;
                    default:
                        return AUTO_LOGIN_WEAK_PASSWD;
                }
            }
            if (preg_match('/^5\.[1|4]\./', PHP_VERSION)) {
                crack_closedict($dict);
            }
        }

        return 0;
    }

    /**
     * パスワードを変更する
     *
     * @access public
     *
     * @param string     $password  パスワード
     * @param mixed      $init
     * @param null|mixed $oldpasswd
     * @param null|mixed $euser
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function changePasswd($password = null, $init = false, $oldpasswd = null, $euser = null)
    {
        $rc = 0;
        $pwstrength = true;
        $random = Util::random(PWD_LEN);
        $winpasswd = isset($this->options['winpasswd']) && $this->options['winpasswd'] ? true : false;

        if ($euser) {
            $this->storage->euser = $euser;
        }
        $object = $this->storage;
        $prop = $object->getProp();
        if ($winpasswd) {
            exec(WINPWDCMD.' '.$object->id.' '.base64_encode(Util::hashPasswd($password)).' '.base64_encode(Util::hashPasswd($oldpasswd)), $output, $errcode);
            if ($errcode) {
                if ($errcode == 53) {
                    return PEAR::raiseError('Password is invalid', AUTO_LOGIN_INVALID_VALUE);
                } else {
                    return PEAR::raiseError(WINPWDCMD, AUTO_LOGIN_ERROR);
                }
            }
        } else {
            if (!$this->app && $init == false) {
                $rc = $this->storage->validatePasswd($password);
            }
            if ($rc == AUTO_LOGIN_WEAK_PASSWD_WARN) {
                $pwstrength = false;
            } elseif ($rc) {
                switch ($rc) {
                    case AUTO_LOGIN_INVALID_VALUE:
                        $message = 'Invalid password';
                        break;
                    case AUTO_LOGIN_MIN_AGE:
                        $message = "Minimum time doesn't elapse between modification to the password";
                        break;
                    case AUTO_LOGIN_IN_HISTORY:
                        $message = 'Password is in th history';
                        break;
                    case AUTO_LOGIN_WEAK_PASSWD:
                        $message = $this->storage->error;
                        break;
                    case AUTO_LOGIN_PROHIBIT_ATTR:
                        $message = $this->storage->error;
                        break;
                    default:
                        $message = 'Internal error';
                }
                return PEAR::raiseError($message, $rc);
            }

            $rc = $this->storage->setPassword($password, $this->app, $init, $random);
            if (PEAR::isError($rc)) {
                return $rc;
            }
        }

        if (isset($this->options['sync']) && $this->options['sync'] && !$this->app) {
            $passwdfile = '/opt/secioss/var/lib/passwd/passwd.log';
            $lock_create = file_exists("$passwdfile.lock") ? false : true;
            $lp = fopen("$passwdfile.lock", 'w');
            if (!$lp) {
                return PEAR::raiseError("Can't open $passwdfile.lock", AUTO_LOGIN_ERROR);
            }
            flock($lp, LOCK_EX);

            $file_create = file_exists($passwdfile) ? false : true;
            $epassword = str_replace(',', '\\2C', $password);
            if (file_put_contents($passwdfile, $this->username.",$epassword,$random,".($euser ? urlencode($euser) : '')."\n", FILE_APPEND) === false) {
                return PEAR::raiseError("Can't write $passwdfile", AUTO_LOGIN_ERROR);
            }
            if ($lock_create) {
                chmod("$passwdfile.lock", 0660);
            }
            if ($file_create) {
                chmod($passwdfile, 0660);
            }

            fclose($lp);
        } elseif (isset($this->options['command']) && $this->options['command']) {
            $commands = $this->options['command'];
            if (!is_array($commands)) {
                $commands = [$commands];
            }

            $complete = 0;
            for ($i = 0; $i < count($commands); $i++) {
                $command = $this->_parseCommand($commands[$i], $password, $oldpasswd);
                exec($command, $output, $errcode);
                if ($errcode) {
                    if (!$winpasswd && isset($this->options['rollback']) && $this->options['rollback']) {
                        for ($j = 0; $j < $complete; $j++) {
                            $command = $this->_parseCommand($commands[$i], $oldpasswd, $password);
                            exec($command, $output, $errcode);
                        }
                        $pwdhistoryattr = $object->options['pwdhistoryattr'];
                        if ($pwdhistoryattr && !isset($prop[$pwdhistoryattr])) {
                            $prop[$pwdhistoryattr] = '';
                        }
                        $object->setProp($prop);
                    }
                    return PEAR::raiseError('Changing password failed: '.$commands[$i]." failed($errcode)", AUTO_LOGIN_ERROR);
                }
                $complete++;
            }
        }

        if ($rc && $pwstrength == false) {
            return PEAR::raiseError($this->storage->error, AUTO_LOGIN_WEAK_PASSWD_WARN);
        }

        return $rc;
    }

    public function deletePasswd()
    {
        return $this->storage->deletePassword($this->app);
    }

    /**
     * 端末IDを設定する
     *
     * @access public
     *
     * @param mixed $device   端末種類 computer/smartphone
     * @param mixed $deviceid 端末ID MD5/UUID/IMEI
     * @param mixed $status   端末状態 active/inactive
     * @param mixed $os       端末OS windows/linux/mac/ipad/iphone/android/computer/smartphone
     * @param mixed $ip       申請時IP 192.168.0.1/::1
     * @param mixed $note     備考 base64_encode
     * @param mixed $appver   アプリ syncux 2.0
     *
     * @return mixed true:正常終了 PEAR_Error:エラー
     */
    public function updateDevice($device, $deviceid, $status, $os, $ip, $note, $appver)
    {
        if (!$device) {
            return PEAR::raiseError('Invalid Device', AUTO_LOGIN_INVALID_VALUE);
        }
        if (!$deviceid) {
            return PEAR::raiseError('Invalid DeviceID', AUTO_LOGIN_INVALID_VALUE);
        }
        if (!$os) {
            if (preg_match('/iPad/i', $_SERVER['HTTP_USER_AGENT'])) {
                $os = 'ipad';
            } elseif (preg_match('/iPhone/i', $_SERVER['HTTP_USER_AGENT'])) {
                $os = 'iphone';
            } elseif (preg_match('/Android/i', $_SERVER['HTTP_USER_AGENT'])) {
                $os = 'android';
            } else {
                $os = $device;
            }
        }
        return $this->storage->setDevice($device, $deviceid, $status, $os, $ip, $note, $appver);
    }

    public function deleteDevice($device, $deviceid)
    {
        if (!$device) {
            return PEAR::raiseError('Invalid Device', AUTO_LOGIN_INVALID_VALUE);
        }
        if (!$deviceid) {
            return PEAR::raiseError('Invalid DeviceID', AUTO_LOGIN_INVALID_VALUE);
        }
        return $this->storage->deleteDevice($device, $deviceid);
    }

    public function getDeviceCurrentNum($ltype = 'device', $dtype = 'smartphone', $newid = null, $tenant = null)
    {
        return $this->storage->getDeviceCurrentNum($ltype, $dtype, $newid, $tenant);
    }

    /**
     * プロファイルのコンフィグ一覧を取得する
     *
     * @access public
     *
     * @param mixed $baseconf
     *
     * @return bool true:正常終了 PEAR_Error:エラー
     */
    public function getProfileConf($baseconf = [])
    {
        $profileconf = $baseconf;

        if (!$this->username) {
            return $profileconf;
        }

        $tenant = null;
        if (preg_match('/[^@]+@(.+)/i', $this->username, $matches)) {
            $tenant = $matches[1];
        }

        if (method_exists($this->storage, 'getProfileConf')) {
            $tmpconf = $this->storage->getProfileConf($tenant);
            if (isset($tmpconf['template'])) {
                foreach ((array) $tmpconf['template'] as $key => $value) {
                    if ($key == 'setaccount') {
                        foreach ($tmpconf['template'][$key] as $setaccountkey => $setaccountval) {
                            $profileconf['template'][$setaccountkey] = $setaccountval;
                        }
                    } else {
                        $profileconf['template'][$key] = $value;
                    }
                }
                if (isset($profileconf['template']['showlinks']['setaccount'])) {
                    $profileconf['template']['setaccount'] = $profileconf['template']['showlinks']['setaccount'];
                }
            }
        }

        return $profileconf;
    }

    /**
     * optionsにデフォルト値を設定する
     *
     * @access protected
     */
    protected function _setDefaults()
    {
        parent::_setDefaults();

        $this->options['pwhash'] = '';
        $this->options['pwminlen'] = 0;
        $this->options['pwmaxlen'] = 255;
        $this->options['pwallow'] = null;
        $this->options['pwallowlimit'] = 0;
        $this->options['pwdeny'] = null;
        $this->options['pwminage'] = 0;
        $this->options['pwinhistory'] = 0;
        $this->options['pwstrength'] = 0;
        $this->options['pwcheckfuncs'] = '';
        $this->options['keyfile'] = '';
        $this->options['oldkeyfile'] = '';
        $this->options['privatekey'] = '';
        $this->options['oldprivatekey'] = '';
        $this->options['publickey'] = '';
        $this->options['libs'] = '';
        $this->options['pwprohibitattr'] = null;
    }

    /**
     * ストレージドライバのオブジェクトを返す
     *
     * @access private
     * @static
     *
     * @param string $driver ストレージクラスの型
     *
     * @return object Object   Storageオブジェクト
     */
    private function _factory($driver)
    {
        $class = 'Secioss\Enterprise\\'.$driver;
        if (class_exists($class)) {
            return new $class($this->options);
        } else {
            return PEAR::raiseError('Driver not exist', AUTO_LOGIN_ERROR);
        }
    }

    /**
     * 指定した画面にリダイレクトする
     *
     * @access private
     *
     * @param mixed      $dst
     * @param null|mixed $querystr
     *
     * @return void
     */
    private function _redirect($dst, $querystr = null)
    {
        if (isset($this->url[$dst])) {
            $url = $this->url[$dst];
            if (preg_match('/%{([^}]+)}/', $url, $matches) && isset($_GET[$matches[1]])) {
                $url = preg_replace('/%{'.$matches[1].'}/', $_GET[$matches[1]], $url);
            }

            if (isset($this->url['back'])) {
                $back = $this->url['back'];
                if (preg_match('/%{([^}]+)}/', $back, $matches) && isset($_GET[$matches[1]])) {
                    $back = preg_replace('/%{'.$matches[1].'}/', $_GET[$matches[1]], $back);
                }
            } else {
                if (isset($_SERVER['REQUEST_URI'])) {
                    $back = $_SERVER['REQUEST_URI'];
                } else {
                    $back = $_SERVER['SCRIPT_NAME'];
                    if (isset($_SERVER['QUERY_STRING'])) {
                        $back .= '?'.$_SERVER['QUERY_STRING'];
                    }
                }
            }
            if (!preg_match('/^http/', $back)) {
                $port = $_SERVER['SERVER_PORT'];
                $back = ($port == 443 ? 'https' : 'http').'://'.$_SERVER['SERVER_NAME'].($port != 443 && $port != 80 ? ":$port" : '').$back;
            }
            $back = urlencode($back);
            if ($querystr) {
                $back = "back=$back&$querystr";
            } else {
                $back = "back=$back";
            }
            if (preg_match('/\?/', $url)) {
                $back = "&$back";
            } else {
                $back = "?$back";
            }

            header('Location: '.$url.$back);
        }
    }

    private function _postRedirect($url, $loginid = null, $password = null)
    {
        if (preg_match('/\?.+/', $url)) {
            $url .= '&post_redirect=on';
        } else {
            $url .= '?post_redirect=on';
        }

        header('Content-Type: text/html; charset=UTF-8');
        header('Pragma: no-cache');
        header('Cache-Control: no-cache');
        header('Expires: Thu, 01 Des 1994 16:00:00 GMT');

        echo '<html>
<body onLoad="document.redirect.submit()">
<form name="redirect" method="POST" action="'.htmlspecialchars($url).'">';

        if ($this->postUser) {
            echo '<input type="hidden" name="'.htmlspecialchars($this->postUser).'" value="'.htmlspecialchars($loginid).'">';
        }
        if ($this->postPass) {
            echo '<input type="hidden" name="'.htmlspecialchars($this->postPass).'" value="'.htmlspecialchars($password).'">';
        }
        $postdata = $this->postData;
        if (isset($postdata['postparam']) && isset($postdata['postval']) && is_array($postdata['postparam']) && is_array($postdata['postval'])) {
            for ($i = 0; $i < count($postdata['postparam']); $i++) {
                if ($postdata['postparam'][$i]) {
                    $value = Util::parsePostval($postdata['postval'][$i], ['loginid' => $this->loginid, 'password' => $this->password]);
                    echo '<input type="hidden" name="'.htmlspecialchars($postdata['postparam'][$i]).'" value="'.htmlspecialchars($value).'">';
                }
            }
        } else {
            foreach ($postdata as $key => $value) {
                $value = Util::parsePostval($value, ['loginid' => $this->loginid, 'password' => $this->password]);
                echo '<input type="hidden" name="'.htmlspecialchars($key).'" value="'.htmlspecialchars($value).'">';
            }
        }

        echo '</form>
</body>
</html>';

        return true;
    }

    private function _constructUrl($url, $oldurl)
    {
        if (preg_match('/^http/', $url)) {
            return $url;
        } elseif (preg_match('/^\//', $url)) {
            return preg_replace('/^(https?:\/\/[^\/]+).*$/', '$1', $oldurl).$url;
        } else {
            return preg_replace('/[^\/]+$/', '', $oldurl).$url;
        }
    }

    private function _parseCommand($command, $password, $oldpasswd)
    {
        $command = str_replace('\'', '"', $command);
        $command = preg_replace(['/%u/', '/%p/', '/%o/', '/%P/', '/%O/'],
            [$this->username, $password, $oldpasswd, Util::hashPasswd($password), Util::hashPasswd($oldpasswd)],
                $command);

        if (preg_match_all('/%{([^}]+)}/', $command, $matches)) {
            for ($i = 0; $i < count($matches[1]); $i++) {
                $values = $this->get($matches[1][$i]);
                if (is_array($values)) {
                    $values = join('+', $values);
                }
                $values = preg_replace('/(["`\\\\])/', '\\\\$1', $values);
                $values = str_replace('$', '\\\$', $values);
                $command = preg_replace('/%{'.$matches[1][$i].'}/', $values, $command);
            }
        }

        return $command;
    }

    /**
     * パスワードに属性値が含まれるかどうか
     *
     * @access private
     *
     * @param string パスワード
     * @param string 属性値
     * @param string 説明（ログエラーメッセージ 画面エラーメッセージ）
     * @param int 桁数
     * @param mixed $password
     * @param mixed $attrvalue
     * @param mixed $desc
     * @param mixed $num
     *
     * @return int 0:含まれない 1:含まれる
     */
    private function _pwdInAttr($password, $attrvalue, $desc, $num = 4)
    {
        if (!isset($attrvalue)) {
            return 0;
        }
        $attrvalue = trim($attrvalue);
        if ($attrvalue) {
            // 完全一致
            $tmp = preg_quote($attrvalue, '/');
            if (preg_match("/$tmp/i", $password)) {
                $this->error = $desc;
                return 1;
            }
            // 任意連続 $num 文字
            if ($num !== 0) {
                for ($i = 0; $i <= strlen($attrvalue) - $num; $i++) {
                    $tmp = substr($attrvalue, $i, $num);
                    $tmp = preg_quote($tmp, '/');
                    if (preg_match("/$tmp/i", $password)) {
                        $this->error = $desc;
                        return 1;
                    }
                }
            }
        }
        return 0;
    }
}
