<?php
/**
 *  AutoLogin_Container.php
 *
 *  PHP versions 4 and 5
 *
 *  @package    Auth
 *
 *  @author     Kaoru Sekiguchi <sekiguchi.kaoru@secioss.co.jp>
 *  @copyright  2016 SECIOSS, INC.
 *
 *  @version    CVS: $Id$
 */
require_once 'PEAR.php';
require_once 'Secioss/Crypt.php';

use Secioss\Crypt;

define('AUTO_LOGIN_ERROR', -1);
define('AUTO_LOGIN_NO_USER', -2);
define('AUTO_LOGIN_NO_PASSWD', -3);
define('AUTO_LOGIN_INVALID_VALUE', -4);
define('AUTO_LOGIN_MIN_AGE', -5);
define('AUTO_LOGIN_IN_HISTORY', -6);
define('AUTO_LOGIN_BAD_PASSWD', -7);
define('AUTO_LOGIN_WEAK_PASSWD', -8);
define('AUTO_LOGIN_WEAK_PASSWD_WARN', -9);
define('AUTO_LOGIN_MIN_PASSWD', -10);
define('AUTO_LOGIN_MAX_PASSWD', -11);
define('AUTO_LOGIN_ALREADY_EXISTS', -12);
define('AUTO_LOGIN_PROHIBIT_ATTR', -13);
define('AUTO_LOGIN_DEVICE_INACTIVE', -100);
define('AUTO_LOGIN_DEVICE_NOEXIST', -101);

define('SECRETKEY_DIRECTIVE', 'TKTAuthSecret');

/**
 *  AutoLogin_Container
 *
 *  @package    Auth
 *
 *  @author     Kaoru Sekiguchi <sekiguchi.kaoru@secioss.co.jp>
 *  @copyright  2016 SECIOSS, INC.
 *
 *  @version    CVS: $Id$
 */
class AutoLogin_Container
{
    /**
     * ストレージの設定
     *
     * @var array
     */
    public $options = [];

    /**
     * ストレージに対する接続
     *
     * @var mixed
     */
    public $conn;

    /**
     * ユーザーのID
     *
     * @var string
     */
    public $id;

    /**
     * ユーザーのログインID
     *
     * @var string
     */
    public $loginid;

    /**
     * ユーザーのステータス
     *
     * @var string
     */
    public $status;

    /**
     * ユーザーのプロパティ
     *
     * @var array
     */
    public $prop;

    public $pwdlockout;

    public $euser;

    /**
     * エラーメッセージ
     *
     * @var array
     */
    public $error;

    // {{{ AutoLogin_Container

    /**
     *  AutoLogin_Containerクラスのコンストラクタ
     *
     *  @access public
     *
     *  @param  mixed   $options        ストレージの設定
     *
     *  @return mixed   0:正常終了 PEAR_Error:エラー
     */
    public function __construct($options)
    {
        $this->_setDefaults();
        $this->_parseOptions($options);
        if ($this->options['libs']) {
            $libs = preg_split('/ /D', $this->options['libs']);
            foreach ($libs as $lib) {
                require_once $lib;
            }
        }
    }

    // }}}

    // {{{ auth()

    /**
     *  ユーザーの認証を行う
     *
     *  @access public
     *
     *  @param  string   パスワード
     * @param mixed $password
     *
     *  @return string   0:成功 1:失敗
     */
    public function auth($password)
    {
        return 1;
    }

    // }}}

    // {{{ get

    /**
     *  プロパティへのアクセサ(R)
     *
     *  @access public
     *
     *  @param  string  $key    プロパティ名
     *
     *  @return mixed   プロパティ
     */
    public function get($key)
    {
        if (isset($this->prop[$key])) {
            return $this->prop[$key];
        }
    }

    // }}}

    public function getProp($keys = null)
    {
        $prop = [];
        if ($keys) {
            foreach ($keys as $key) {
                if (isset($this->prop[$key])) {
                    $prop[$key] = $this->prop[$key];
                }
            }
        } else {
            $prop = $this->prop;
        }
        return $prop;
    }

    // {{{ getLoginId

    /**
     *  ログインIDを取得する。
     *
     *  @access public
     *
     *  @return string   ログインID
     */
    public function getLoginId()
    {
        return $this->loginid;
    }

    // }}}

    // {{{ getStatus

    /**
     *  ステータスを取得する。
     *
     *  @access public
     *
     *  @return string   ステータス
     */
    public function getStatus()
    {
        return $this->status;
    }

    // }}}

    // {{{ getPassword()

    /**
     *  暗号化されているパスワードを復号化して取得する
     *
     *  @access public
     *
     * @param null|mixed $app
     *
     *  @return string   パスワード
     */
    public function getPassword($app = null)
    {
    }

    // }}}

    public function getTenants($tenant = null)
    {
    }

    public function getDeviceCurrentNum($ltype, $dtype, $newid, $tenant)
    {
    }

    // {{{ set

    /**
     *  プロパティへのアクセサ(W)
     *
     *  @access public
     *
     *  @param  string  $key    プロパティ名
     *  @param  string  $value  値
     *
     *  @return mixed   プロパティ
     */
    public function set($key, $value)
    {
        return $this->setProp([$key => $value]);
    }

    // }}}

    public function setProp($prop)
    {
    }

    // {{{ setPassword()

    /**
     *  パスワードを暗号化してストレージに格納する
     *
     *  @access public
     *
     *  @param  string   パスワード
     *  @param  string   アプリケーション
     * @param mixed      $password
     * @param null|mixed $app
     * @param mixed      $init
     * @param null|mixed $random
     *
     *  @return mixed    true:正常終了 PEAR_Error:エラー
     */
    public function setPassword($password, $app = null, $init = false, $random = null)
    {
    }

    // }}}

    public function deletePassword($app)
    {
    }

    // {{{ getPwdChangedTime()

    /**
     *  パスワード変更時間を取得する
     *
     *  @access public
     *
     *  @return string   パスワード変更時間（秒）
     */
    public function getPwdChangedTime()
    {
        return 0;
    }

    // }}}

    // {{{ getPwdHistory()

    /**
     *  パスワード履歴を取得する
     *
     *  @access public
     *
     *  @return mixed   パスワード履歴
     */
    public function getPwdHistory()
    {
        return [];
    }

    // }}}

    // {{{ getPwdLockout

    /**
     *  ロックアウトを取得する。
     *
     *  @access public
     *
     *  @return string   ロックアウト
     */
    public function getPwdLockout()
    {
        return $this->pwdlockout;
    }

    // }}}

    public function isPwdMustChange()
    {
        return false;
    }

    // {{{ validatePasswd()

    /**
     *  パスワードの妥当性を確認する
     *
     *  @access public
     *
     *  @param  string   パスワード
     * @param mixed $password
     *
     *  @return mixed    0:正常終了
     */
    public function validatePasswd($password)
    {
        $pwlen = strlen($password);

        if ($pwlen < $this->options['pwminlen']) {
            return AUTO_LOGIN_MIN_PASSWD;
        }

        if ($pwlen > $this->options['pwmaxlen']) {
            return AUTO_LOGIN_MAX_PASSWD;
        }

        if ($this->options['pwallow']) {
            $pwallows = $this->_to_array($this->options['pwallow']);
            $pwallowlimit = count($pwallows);
            if ($this->options['pwallowlimit']) {
                $pwallowlimit = $this->options['pwallowlimit'];
            }
            $match = 0;
            foreach ($pwallows as $pwallow) {
                if (preg_match("/$pwallow/", $password)) {
                    $match++;
                }
            }
            if ($match < $pwallowlimit) {
                return AUTO_LOGIN_INVALID_VALUE;
            }
        }

        if ($this->options['pwdeny']) {
            $pwdenys = $this->_to_array($this->options['pwdeny']);
            foreach ($pwdenys as $pwdeny) {
                if (preg_match("/$pwdeny/", $password)) {
                    return AUTO_LOGIN_INVALID_VALUE;
                }
            }
        }

        if ($this->options['pwminage'] && !$this->isPwdMustChange()) {
            $pwdtime = $this->getPwdChangedTime();
            if ($pwdtime && time() < $pwdtime + $this->options['pwminage']) {
                return AUTO_LOGIN_MIN_AGE;
            }
        }

        if ($this->options['pwinhistory']) {
            $pwdhistory = $this->getPwdHistory();

            for ($i = 0; $i < $this->options['pwinhistory'] && $i < count($pwdhistory); $i++) {
                $pwhash = null;
                $oldpasswd = $pwdhistory[$i];
                if (preg_match('/^{([^}]+)}(.+)$/', $pwdhistory[$i], $matches)) {
                    $pwhash = $matches[1];
                    $oldpasswd = '{'.$pwhash.'}'.$matches[2];
                }
                if ($this->cmpPasswd($password, $oldpasswd, $pwhash)) {
                    return AUTO_LOGIN_IN_HISTORY;
                }
            }
        }

        if ($this->options['pwcheckfuncs']) {
            foreach (explode(' ', $this->options['pwcheckfuncs']) as $func) {
                $rc = $func($password, $this->id);
                if ($rc) {
                    return $rc;
                }
            }
        }

        if ($this->options['pwprohibitattr']) {
            $phoneignore = [' ', '-', '+', '#', '*'];
            foreach (explode(',', $this->options['pwprohibitattr']) as $attr) {
                switch ($attr) {
                    case 'uid':
                        $uid = strstr($this->prop[$attr], '@', true);
                        $uid = $uid ? $uid : $this->prop[$attr];
                        if ($this->_pwdInAttr($password, $uid, "$attr ユーザーID", 0)) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'employeenumber':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 社員番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'cn':
                        if ($this->_pwdInAttr($password, $this->prop['sn'], 'sn 氏名')) { //姓
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        if ($this->_pwdInAttr($password, $this->prop['givenname'], 'gn 氏名')) { //名
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'displayname':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 別名")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'mail':
                        $mail = strstr($this->prop['mail'], '@', true);
                        if ($this->_pwdInAttr($password, $mail, "$attr メールアドレス")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        // TODO seciossmailalias
                        break;
                    case 'seciossnotificationmail':
                        $seciossnotificationmail = strstr($this->prop['seciossnotificationmail'], '@', true);
                        if ($this->_pwdInAttr($password, $seciossnotificationmail, "$attr 通知用メールアドレス")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'seciosstelephonenumber':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 電話番号") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr 電話番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'seciossfax':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr FAX") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr FAX")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'seciossmobile':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 携帯電話番号") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr 携帯電話番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'seciosshomephone':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr 自宅電話番号") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr 自宅電話番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    case 'pager':
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr ポケベル番号") || $this->_pwdInAttr($password, str_replace($phoneignore, '', $this->prop[$attr]), "$attr ポケベル番号")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                        break;
                    default:
                        if ($this->_pwdInAttr($password, $this->prop[$attr], "$attr $attr")) {
                            return AUTO_LOGIN_PROHIBIT_ATTR;
                        }
                }
            }
        }

        if ($this->options['pwstrength']) {
            $level = $this->options['pwstrength'];

            if (preg_match('/^5\.[1|4]\./', PHP_VERSION)) {
                $dict = crack_opendict('/usr/share/cracklib/pw_dict');
                if (!$dict) {
                    return AUTO_LOGIN_ERROR;
                }

                $check = crack_check($dict, $password);
                $message = crack_getlastmessage();
            } else {
                exec('echo '.escapeshellarg($password).' | /usr/sbin/cracklib-check 2>/dev/null', $output, $rc);
                if ($rc || !preg_match('/^.*: (.+)$/', $output[0], $matches)) {
                    return AUTO_LOGIN_ERROR;
                }
                $message = $matches[1];
            }
            if ($message != 'strong password' && $message != 'OK') {
                if (preg_match('/^5\.[1|4]\./', PHP_VERSION)) {
                    crack_closedict($dict);
                }
                $this->error = $message;
                switch ($level) {
                    case 1:
                        return AUTO_LOGIN_WEAK_PASSWD_WARN;
                    default:
                        return AUTO_LOGIN_WEAK_PASSWD;
                }
            }
            if (preg_match('/^5\.[1|4]\./', PHP_VERSION)) {
                crack_closedict($dict);
            }
        }

        return 0;
    }

    // }}}

    // {{{ validatePasswd()

    /**
     *  パスワードに属性値が含まれるかどうか
     *
     *  @access private
     *
     *  @param  string   パスワード
     *  @param  string   属性値
     *  @param  string   説明（ログエラーメッセージ 画面エラーメッセージ）
     *  @param  int      桁数
     * @param mixed $password
     * @param mixed $attrvalue
     * @param mixed $desc
     * @param mixed $num
     *
     *  @return int      0:含まれない 1:含まれる
     */
    public function _pwdInAttr($password, $attrvalue, $desc, $num = 4)
    {
        if (!isset($attrvalue)) {
            return 0;
        }
        $attrvalue = trim($attrvalue);
        if ($attrvalue) {
            // 完全一致
            $tmp = preg_quote($attrvalue, '/');
            if (preg_match("/$tmp/i", $password)) {
                $this->error = $desc;
                return 1;
            }
            // 任意連続 $num 文字
            if ($num !== 0) {
                for ($i = 0; $i <= strlen($attrvalue) - $num; $i++) {
                    $tmp = substr($attrvalue, $i, $num);
                    $tmp = preg_quote($tmp, '/');
                    if (preg_match("/$tmp/i", $password)) {
                        $this->error = $desc;
                        return 1;
                    }
                }
            }
        }
        return 0;
    }

    // }}}

    public function cmpPasswd($passwd, $hashedpwd, $pwhash = null)
    {
        if (!$pwhash) {
            $pwhash = $this->options['pwhash'];
        }

        $tab = '';
        if ($pwhash == 'CRYPT' || $pwhash == 'MD5' || $pwhash == 'SHA' || $pwhash == 'SSHA') {
            $tab = '{'.$pwhash.'}';
        }

        if ($pwhash == 'SSHA') {
            $salt = substr(base64_decode(substr($hashedpwd, 6)), 20);
            return $hashedpwd == $tab.$this->hashPasswd($passwd, $pwhash, $salt);
        } elseif ($pwhash == 'CRYPT') {
            $salt = substr($hashedpwd, strlen($tab), 2);
            return $hashedpwd == $tab.$this->hashPasswd($passwd, $pwhash, $salt);
        } else {
            return $hashedpwd == $tab.$this->hashPasswd($passwd, $pwhash);
        }
    }

    // {{{ hashPasswd()

    /**
     * パスワードの暗号化
     *
     * パスワードを暗号化する。対応する暗号方式はCRYPT、MD5、SHA、SSHA。
     *
     * @access private
     *
     * @param string     $passwd パスワード
     * @param null|mixed $pwhash
     * @param null|mixed $salt
     *
     * @return string 暗号化されたパスワード
     */
    public function hashPasswd($passwd, $pwhash = null, $salt = null)
    {
        if (!$pwhash) {
            $pwhash = $this->options['pwhash'];
        }

        if (!$pwhash) {
            return $passwd;
        }

        $opts = preg_split('/:/D', $pwhash);
        $htype = $opts[0];
        $otype = isset($opts[1]) ? $opts[1] : '';
        $num = isset($opts[2]) ? $opts[2] : 1;

        for ($i = 0; $i < $num; $i++) {
            switch ($htype) {
                case 'CRYPT':
                    $token = array_merge(range('0', '9'), range('a', 'z'), range('A', 'Z'));
                    if (!$salt) {
                        $salt = $token[rand(0, count($token) - 1)].$token[rand(0, count($token) - 1)];
                    }
                    $passwd = crypt($passwd, $salt);
                    break;
                case 'MD5':
                    $passwd = md5($passwd);
                    $passwd = $otype == 'hex' ? $passwd : base64_encode(pack('H*', $passwd));
                    break;
                case 'SHA':
                    $passwd = sha1($passwd);
                    $passwd = $otype == 'hex' ? $passwd : base64_encode(pack('H*', $passwd));
                    break;
                case 'SSHA':
                    if (!$salt) {
                        $salt = openssl_random_pseudo_bytes(4);
                    }
                    $passwd = base64_encode(sha1($passwd.$salt, true).$salt);
                    break;
                case 'AD':
                    $passwd = '"'.$passwd.'"';
                    $adpasswd = '';
                    for ($j = 0; $j < strlen($passwd); $j++) {
                        $adpasswd .= "{$passwd[$j]}\000";
                    }
                    return $adpasswd;
                    break;
                case 'CLEARTEXT':
                    return $passwd;
                    break;
                default:
                    return $passwd;
            }
        }

        return $passwd;
    }

    // }}}

    // {{{ encrypt()

    /**
     *  文字列の暗号化を行う
     *
     *  @access public
     *
     *  @param  string   文字列
     * @param mixed $string
     *
     *  @return string   暗号化された文字列
     */
    public function encrypt($string)
    {
        if ($string === null || $string === '') {
            return $string;
        }

        if ($this->options['publickey']) {
            return $this->ssl_encrypt($string);
        } else {
            return Crypt::encrypt($string, Crypt::getSecretKey($this->options['keyfile']));
        }
    }

    // }}}

    public function ssl_encrypt($string)
    {
        $publickey = openssl_get_publickey('file://'.$this->options['publickey']);
        if (openssl_public_encrypt($string, $encrypted, $publickey, OPENSSL_PKCS1_OAEP_PADDING)) {
            return base64_encode($encrypted);
        }
    }

    // {{{ decrypt()

    /**
     *  文字列の復号化を行う
     *
     *  @access public
     *
     * @param mixed $string
     *
     *  @return string   復号化された文字列
     */
    public function decrypt($string)
    {
        if ($string === null || $string === '') {
            return $string;
        }

        if ($this->options['privatekey']) {
            return $this->ssl_decrypt($string);
        } else {
            return Crypt::decrypt($string, Crypt::getSecretKey($this->options['keyfile']));
        }
    }

    // }}}

    public function ssl_decrypt($string)
    {
        $keyfiles = [$this->options['privatekey']];
        if ($this->options['oldprivatekey']) {
            array_push($keyfiles, $this->options['oldprivatekey']);
        }

        $string = base64_decode($string);
        foreach ($keyfiles as $keyfile) {
            $privatekey = openssl_get_privatekey('file://'.$keyfile);
            if (openssl_private_decrypt($string, $decrypted, $privatekey, OPENSSL_PKCS1_OAEP_PADDING)) {
                return $decrypted;
            }
        }

        return PEAR::raiseError('Encrypted password is invalid', AUTO_LOGIN_ERROR);
    }

    public function getSecretKey()
    {
        return Crypt::getSecretKey($this->options['keyfile']);
    }

    // {{{ _to_array

    /**
     *  スカラー値を要素数1の配列として返す
     *
     *  @param  mixed   $v  配列として扱う値
     *
     *  @return array   配列に変換された値
     */
    public function _to_array($v)
    {
        if (is_array($v)) {
            return $v;
        } elseif ($v) {
            return [$v];
        } else {
            return [];
        }
    }

    // }}}

    // {{{ _parseOptions()

    /**
     * optionsに値を設定する
     *
     * @access private
     *
     * @param  array
     * @param mixed $array
     */
    public function _parseOptions($array)
    {
        if (is_array($array)) {
            foreach ($array as $key => $value) {
                if (array_key_exists($key, $this->options)) {
                    $this->options[$key] = $value;
                }
            }
        }
    }

    // }}}

    // {{{ _setDefaults()

    /**
     * optionsにデフォルト値を設定する
     *
     * @access private
     */
    public function _setDefaults()
    {
        $this->options['pwhash'] = '';
        $this->options['pwminlen'] = 0;
        $this->options['pwmaxlen'] = 255;
        $this->options['pwallow'] = null;
        $this->options['pwallowlimit'] = 0;
        $this->options['pwdeny'] = null;
        $this->options['pwminage'] = 0;
        $this->options['pwinhistory'] = 0;
        $this->options['pwstrength'] = 0;
        $this->options['pwcheckfuncs'] = '';
        $this->options['keyfile'] = '';
        $this->options['oldkeyfile'] = '';
        $this->options['privatekey'] = '';
        $this->options['oldprivatekey'] = '';
        $this->options['publickey'] = '';
        $this->options['libs'] = '';
        $this->options['pwprohibitattr'] = null;
    }

    // }}}

    public function setDevice($device, $deviceid, $status, $os, $ip, $note, $appver)
    {
    }

    public function deleteDevice($device, $deviceid)
    {
    }

    // {{{ validateSecret()

    /**
     *  added by okazaki 20090710
     *  シークレットの妥当性を確認する
     *
     *  @access public
     *
     *  @param  string   シークレット
     * @param mixed $secret
     *
     *  @return mixed    0:正常終了
     */
    public function validateSecret($secret)
    {
        /*
                $len = strlen($secret);

                if ($len != 16) {
                    return AUTO_LOGIN_INVALID_VALUE;
                }
        */
        return 0;
    }

    // }}}

    // {{{ setSecret()

    /**
     *  added by okazaki 20090710
     *  シークレットを暗号化してストレージに格納する
     *
     *  @access public
     *
     *  @param  string   シークレット
     *  @param  string   PIN
     * @param mixed $secret
     * @param mixed $pin
     * @param mixed $deviceid
     * @param mixed $device
     * @param mixed $otplen
     * @param mixed $timewindow
     * @param mixed $os
     * @param mixed $ip
     *
     *  @return mixed    true:正常終了 PEAR_Error:エラー
     */
    public function setSecret($secret, $pin, $deviceid, $device, $otplen, $timewindow, $os, $ip)
    {
    }

    // }}}

    // {{{ getSecret()

    /**
     *  added by okazaki 20090710
     *  暗号化されているシークレットを復号化して取得する
     *
     *  @access public
     *
     * @param null|mixed $app
     *
     *  @return string   シークレット
     */
    public function getSecret($app = null)
    {
    }

    // }}}

    public function random($len)
    {
        $token = array_merge(range('0', '9'), range('a', 'z'), range('A', 'Z'));

        $str = '';
        for ($i = 0; $i < $len; $i++) {
            $str = $str.$token[rand(0, count($token) - 1)];
        }

        return $str;
    }
}
