package org.seasar.framework.sel.tokenizer;

import java.math.BigDecimal;
import java.math.BigInteger;

public class CoreTokenizer {

	public static final int TT_EOF = -1;
	public static final int TT_INTEGER = -11;
	public static final int TT_LONG = -12;
	public static final int TT_DECIMAL = -13;
	public static final int TT_WORD = -3;

	private static final int TT_NOTHING = -4;
	private static final int NEED_CHAR = Integer.MAX_VALUE;
	private static final int QUOTE = '\'';

	private static final byte CT_WHITESPACE = 1;
	private static final byte CT_DIGIT = 2;
	private static final byte CT_ALPHA = 4;
	
	private static byte[] ctype = new byte[256];
	
	private String str_;
	private int colno_ = 0;
	private int ttype_ = TT_NOTHING;
	private String sval_;
	private Integer ival_;
	private Long lval_;
	private BigDecimal dval_;
	private char[] buf_ = new char[20];
	private int peekc_ = NEED_CHAR;
	private byte peekct_ = 0;

	static {
		setup();
	}

	private static void setup() {
		wordChars('a', 'z');
		wordChars('A', 'Z');
		wordChars('0', '9');
		wordChars('@', '@');
		wordChars('|', '|');
		wordChars('_', '_');
		wordChars('?', '?');
		wordChars('>', '>');
		wordChars('=', '=');
		wordChars('!', '!');
		wordChars('<', '<');
		wordChars(':', ':');
		wordChars('"', '"');
		wordChars('~', '~');
		wordChars('*', '*');
		ordinaryChar('(');
		ordinaryChar(')');
		ordinaryChar('+');
		ordinaryChar('-');
		ordinaryChar('/');
		ordinaryChar('%');
		ordinaryChar(',');
		ordinaryChar('[');
		ordinaryChar(']');
		ordinaryChar('{');
		ordinaryChar('}');
		whitespaceChars(0, ' ');
		numbers();
	}

	private static void wordChars(int low, int hi) {
		if (low < 0) {
			low = 0;
		}
		if (hi >= ctype.length) {
			hi = ctype.length - 1;
		}
		while (low <= hi) {
			ctype[low++] |= CT_ALPHA;
		}
	}

	private static void whitespaceChars(int low, int hi) {
		if (low < 0) {
			low = 0;
		}
		if (hi >= ctype.length) {
			hi = ctype.length - 1;
		}
		while (low <= hi) {
			ctype[low++] = CT_WHITESPACE;
		}
	}

	private static void ordinaryChar(int ch) {
		if (ch >= 0 && ch < ctype.length) {
			ctype[ch] = 0;
		}
	}

	private static void numbers() {
		for (int i = '0'; i <= '9'; i++) {
			ctype[i] |= CT_DIGIT;
		}
		ctype['.'] |= CT_DIGIT;
		ctype['-'] |= CT_DIGIT;
	}

	public CoreTokenizer(String str) {
		str_ = str;
	}
	
	public final int getTokenType() {
		return ttype_;
	}
	
	public final String getStringValue() {
		return sval_;
	}
	
	public final Integer getIntegerValue() {
		return ival_;
	}
	
	public final Long getLongValue() {
		return lval_;
	}
	
	public final BigDecimal getBigDecimalValue() {
		return dval_;
	}

	public int nextToken() {
		initVal();
		if (processEOF()) {
			return ttype_;
		}
		if (processWhitespace()) {
			return ttype_;
		}
		if (processDigit()) {
			return ttype_;
		}
		if (processWord()) {
			return ttype_;
		}
		if (processQuote()) {
			return ttype_;
		}
		if (processOrdinary()) {
			return ttype_;
		}
		return ttype_ = peekc_;
	}

	public final String getReadString() {
		return str_.substring(0, colno_ - 1);
	}

	private int read() {
		if (colno_ >= str_.length()) {
			return -1;
		}
		return str_.charAt(colno_++);
	}

	private void initVal() {
		sval_ = null;
		ival_ = null;
		lval_ = null;
		dval_ = null;
	}

	private boolean processEOF() {
		if (peekc_ < 0) {
			ttype_ = TT_EOF;
			return true;
		}
		if (peekc_ == NEED_CHAR) {
			peekc_ = read();
			if (peekc_ < 0) {
				ttype_ = TT_EOF;
				return true;
			}
		}
		return false;
	}

	private boolean processWhitespace() {
		peekct_ = peekc_ < 256 ? ctype[peekc_] : CT_ALPHA;
		while ((peekct_ & CT_WHITESPACE) != 0) {
			if (peekc_ == '\r') {
				peekc_ = read();
				if (peekc_ == '\n') {
					peekc_ = read();
				}
			} else {
				peekc_ = read();
			}
			if (peekc_ < 0) {
				ttype_ = TT_EOF;
				return true;
			}
			peekct_ = peekc_ < 256 ? ctype[peekc_] : CT_ALPHA;
		}
		return false;
	}

	private boolean processDigit() {
		if ((peekct_ & CT_DIGIT) != 0) {
			boolean neg = false;
			long v = 0;
			int decexp = 0;
			int seendot = 0;
			switch (peekc_) {
				case '-' :
					peekc_ = read();
					if (peekc_ != '.' && (peekc_ < '0' || peekc_ > '9')) {
						ttype_ = '-';
						return true;
					}
					neg = true;
					break;
				case '.' :
					peekc_ = read();
					if (peekc_ != '.' && (peekc_ < '0' || peekc_ > '9')) {
						ttype_ = '.';
						return true;
					} else {
						seendot = 1;
					}
					break;
			}
			while (true) {
				if (peekc_ == '.' && seendot == 0) {
					seendot = 1;
				} else if ('0' <= peekc_ && peekc_ <= '9') {
					v = v * 10 + (peekc_ - '0');
					decexp += seendot;
				} else {
					break;
				}
				peekc_ = read();
			}
			long v2 = neg ? -v : v;
			if (decexp != 0) {
				dval_ = new BigDecimal(BigInteger.valueOf(v2), decexp);
				ttype_ = TT_DECIMAL;
				return true;
			}
			if (v2 > Integer.MAX_VALUE || v2 < Integer.MIN_VALUE) {
				lval_ = new Long(v2);
				ttype_ = TT_LONG;
				return true;
			}
			int v3 = (int) v;
			v3 = neg ? -v3 : v3;
			ival_ = new Integer(v3);
			ttype_ = TT_INTEGER;
			return true;
		}
		return false;
	}

	private boolean processWord() {
		if ((peekct_ & CT_ALPHA) != 0) {
			int i = 0;
			do {
				if (i >= buf_.length) {
					char nb[] = new char[buf_.length * 2];
					System.arraycopy(buf_, 0, nb, 0, buf_.length);
					buf_ = nb;
				}
				buf_[i++] = (char) peekc_;
				peekc_ = read();
				peekct_ =
					peekc_ < 0
						? CT_WHITESPACE
						: (peekc_ < 256 ? ctype[peekc_] : CT_ALPHA);
			} while ((peekct_ & (CT_ALPHA | CT_DIGIT)) != 0);
			sval_ = String.copyValueOf(buf_, 0, i);
			ttype_ = TT_WORD;
			return true;
		}
		return false;
	}

	private boolean processQuote() {
		if (peekc_ == QUOTE) {
			ttype_ = QUOTE;
			int i = 0;
			int d = read();
			int c = d;
			while (d >= 0) {
				if (d == QUOTE) {
					int d2 = read();
					if (d2 == QUOTE) {
						c = QUOTE;
					} else {
						d = d2;
						break;
					}
				} else {
					c = d;
				}
				if (i >= buf_.length) {
					char nb[] = new char[buf_.length * 2];
					System.arraycopy(buf_, 0, nb, 0, buf_.length);
					buf_ = nb;
				}
				buf_[i++] = (char) c;
				d = read();
			}
			peekc_ = d;
			sval_ = String.copyValueOf(buf_, 0, i);
			return true;
		}
		return false;
	}

	private boolean processOrdinary() {
		if (peekct_ == 0) {
			ttype_ = peekc_;
			peekc_ = read();
			peekct_ =
				peekc_ < 0
					? CT_WHITESPACE
					: (peekc_ < 256 ? ctype[peekc_] : CT_ALPHA);
			return true;
		}
		return false;
	}
}
