package org.seasar.extension.unit;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

import javax.transaction.TransactionManager;

import org.seasar.framework.container.ComponentDef;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.S2ContainerFactory;
import org.seasar.framework.container.impl.S2ContainerImpl;
import org.seasar.framework.exception.NoSuchMethodRuntimeException;
import org.seasar.framework.util.ClassUtil;
import org.seasar.framework.util.FieldUtil;
import org.seasar.framework.util.MethodUtil;
import org.seasar.framework.util.StringUtil;

import junit.framework.TestCase;

public class S2TestCase extends TestCase {

	private S2Container container_;
	private List bindedFields_;

	public S2TestCase(String name) {
		super(name);
	}

	public S2Container getContainer() {
		return container_;
	}

	public Object getComponent(String componentName) {
		return container_.getComponent(componentName);
	}

	public Object getComponent(Class componentClass) {
		return container_.getComponent(componentClass);
	}

	public ComponentDef getComponentDef(String componentName) {
		return container_.getComponentDef(componentName);
	}

	public ComponentDef getComponentDef(Class componentClass) {
		return container_.getComponentDef(componentClass);
	}

	public void register(Class componentClass) {
		container_.register(componentClass);
	}

	public void register(Class componentClass, String componentName) {
		container_.register(componentClass, componentName);
	}

	public void register(ComponentDef componentDef) {
		container_.register(componentDef);
	}

	public void include(String path) {
		container_.include(S2ContainerFactory.create(path));
	}

	public void runBare() throws Throwable {
		container_ = new S2ContainerImpl();
		setUp();
		try {
			setUpForEachTestMethod();
			try {
				container_.init();
				try {
					bindFields();
					try {
						runTestTx();
					} finally {
						unbindFields();
					}
				} finally {
					container_.destroy();
				}
			} finally {
				tearDownForEachTestMethod();
			}
			
		} finally {
			tearDown();
		}
	}

	protected void setUpForEachTestMethod() throws Throwable {
		invoke("setUp" + getTargetName());
	}

	protected void tearDownForEachTestMethod() throws Throwable {
		invoke("tearDown" + getTargetName());
	}
	
	private String getTargetName() {
		return getName().substring(4);
	}

	private void invoke(String methodName) throws Throwable {
		try {
			Method method = ClassUtil.getMethod(getClass(), methodName, null);
			MethodUtil.invoke(method, this, null);
		} catch (NoSuchMethodRuntimeException ignore) {
		}
	}

	private void bindFields() throws Throwable {
		bindedFields_ = new ArrayList();
		for (Class clazz = getClass();
			clazz != S2TestCase.class && clazz != null;
			clazz = clazz.getSuperclass()) {

			Field[] fields = clazz.getDeclaredFields();
			for (int i = 0; i < fields.length; ++i) {
				bindField(fields[i]);
			}
		}
	}

	private void bindField(Field field) {
		if (isAutoBindable(field)) {
			field.setAccessible(true);
			if (FieldUtil.get(field, this) != null) {
				return;
			}
			String name = normalizeName(field.getName());
			Object component = null;
			if (getContainer().hasComponentDef(name)) {
				component = getComponent(name);
			} else if (Modifier.isInterface(field.getType().getModifiers())) {
				try {
					 component = getComponent(field.getType());
				} catch (Throwable t) {
					System.err.println(t);
					return;
				}
			}
			FieldUtil.set(field, this, component);
			bindedFields_.add(field);
		}
	}
	
	private String normalizeName(String name) {
		return StringUtil.replace(name, "_", "");
	}

	private boolean isAutoBindable(Field field) {
		int modifiers = field.getModifiers();
		return !Modifier.isStatic(modifiers)
			&& !Modifier.isFinal(modifiers)
			&& !field.getType().isPrimitive();
	}

	private void unbindFields() {
		for (int i = 0; i < bindedFields_.size(); ++i) {
			Field field = (Field) bindedFields_.get(i);
			try {
				field.set(this, null);
			} catch (IllegalArgumentException e) {
				System.err.println(e);
			} catch (IllegalAccessException e) {
				System.err.println(e);
			}
		}
	}
	
	private void runTestTx() throws Throwable {
		TransactionManager tm = null;
		if (needTransaction()) {
			try {
				tm = (TransactionManager)
					getComponent(TransactionManager.class);
				tm.begin();
			} catch (Throwable t) {
				System.err.println(t);
			}
		}
		try {
			runTest();
		} finally {
			if (tm != null) {
				tm.rollback();
			}
		}
	}
	
	private boolean needTransaction() {
		return getName().endsWith("Tx");
	}
}