package test.org.seasar.framework.sel.tokenizer;

import java.math.BigDecimal;

import org.seasar.framework.sel.tokenizer.CoreTokenizer;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

public class CoreTokenizerTest extends TestCase {

	public CoreTokenizerTest(String name) {
		super(name);
	}

	public void testEOF() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("");
		assertEquals("1", CoreTokenizer.TT_EOF, tokenizer.nextToken());
		assertEquals("2", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	public void testWhitespace() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("\t       \n");
		assertEquals("1", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	public void testHyphen() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("       - ");
		assertEquals("1", '-', tokenizer.nextToken());
		assertEquals("2", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	public void testInteger() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("       123");
		assertEquals("1", CoreTokenizer.TT_INTEGER, tokenizer.nextToken());
		assertEquals("2", new Integer(123), tokenizer.getIntegerValue());
		assertEquals("3", CoreTokenizer.TT_EOF, tokenizer.nextToken());

		CoreTokenizer tokenizer2 = new CoreTokenizer("    -1234567890");
		assertEquals("4", CoreTokenizer.TT_INTEGER, tokenizer2.nextToken());
		assertEquals(
			"5",
			new Integer(-1234567890),
			tokenizer2.getIntegerValue());

		CoreTokenizer tokenizer3 =
			new CoreTokenizer(String.valueOf(Integer.MAX_VALUE));
		assertEquals("6", CoreTokenizer.TT_INTEGER, tokenizer3.nextToken());
		assertEquals(
			"7",
			new Integer(Integer.MAX_VALUE),
			tokenizer3.getIntegerValue());

		CoreTokenizer tokenizer4 =
			new CoreTokenizer(String.valueOf(Integer.MIN_VALUE));
		assertEquals("8", CoreTokenizer.TT_INTEGER, tokenizer4.nextToken());
		assertEquals(
			"9",
			new Integer(Integer.MIN_VALUE),
			tokenizer4.getIntegerValue());
	}

	public void testLong() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("       12345678901");
		assertEquals("1", CoreTokenizer.TT_LONG, tokenizer.nextToken());
		assertEquals("2", new Long("12345678901"), tokenizer.getLongValue());
		assertEquals("3", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	public void testBigDecimal() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("       123.45 ");
		assertEquals("1", CoreTokenizer.TT_DECIMAL, tokenizer.nextToken());
		assertEquals("2", new BigDecimal("123.45"), tokenizer.getBigDecimalValue());
		assertEquals("3", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	public void testDot() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer(".hoge");
		assertEquals("1", '.', tokenizer.nextToken());
		assertEquals("2", '.', tokenizer.getTokenType());
		assertEquals("3", CoreTokenizer.TT_WORD, tokenizer.nextToken());
		assertEquals("4", "hoge", tokenizer.getStringValue());
	}

	public void testOrdinary() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("(1 + 2)");
		assertEquals("1", '(', tokenizer.nextToken());
		assertEquals("2", CoreTokenizer.TT_INTEGER, tokenizer.nextToken());
		assertEquals("3", new Integer(1), tokenizer.getIntegerValue());
		assertEquals("4", '+', tokenizer.nextToken());
		assertEquals("5", CoreTokenizer.TT_INTEGER, tokenizer.nextToken());
		assertEquals("6", new Integer(2), tokenizer.getIntegerValue());
		assertEquals("7", ')', tokenizer.nextToken());
		assertEquals("8", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	public void testWord() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("123aaa45");
		assertEquals("1", CoreTokenizer.TT_INTEGER, tokenizer.nextToken());
		assertEquals("2", new Integer(123), tokenizer.getIntegerValue());
		assertEquals("3", CoreTokenizer.TT_WORD, tokenizer.nextToken());
		assertEquals("4", "aaa45", tokenizer.getStringValue());
		assertEquals("5", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	public void testQuote() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("'\n123''aaa'aa");
		assertEquals("1", '\'', tokenizer.nextToken());
		assertEquals("2", "\n123'aaa", tokenizer.getStringValue());
		assertEquals("3", CoreTokenizer.TT_WORD, tokenizer.nextToken());
		assertEquals("4", "aa", tokenizer.getStringValue());
		assertEquals("5", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	public void testQuote2() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("out.println('a\b')");
		tokenizer.nextToken();
		tokenizer.nextToken();
		assertEquals("1", '\'', tokenizer.nextToken());
		assertEquals("2", "a\b", tokenizer.getStringValue());
		tokenizer.nextToken();
		assertEquals("3", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	public void testAtMark() throws Exception {
		CoreTokenizer tokenizer = new CoreTokenizer("@aaa");
		assertEquals("1", CoreTokenizer.TT_WORD, tokenizer.nextToken());
		assertEquals("2", "@aaa", tokenizer.getStringValue());
		assertEquals("3", CoreTokenizer.TT_EOF, tokenizer.nextToken());
	}

	protected void setUp() throws Exception {
	}

	protected void tearDown() throws Exception {
	}

	public static Test suite() {
		return new TestSuite(CoreTokenizerTest.class);
	}

	public static void main(String[] args) {
		junit.textui.TestRunner.main(
			new String[] { CoreTokenizerTest.class.getName()});
	}
}