package test.org.seasar.nazuna;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Map;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.seasar.nazuna.ArgType;
import org.seasar.nazuna.BooleanExpression;
import org.seasar.nazuna.Expression;
import org.seasar.nazuna.Flowlet;
import org.seasar.nazuna.InputType;
import org.seasar.nazuna.LocalType;
import org.seasar.nazuna.RuleContext;
import org.seasar.nazuna.RuleContextImpl;
import org.seasar.nazuna.RuleParser;
import org.seasar.nazuna.VarType;
import org.seasar.util.Conversion;
import org.seasar.util.EMap;

public class RuleParserTest extends TestCase {

	public RuleParserTest(String name) {
		super(name);
	}

	public void testOrExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("true or false").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testOrExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("(true or false) and false")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testOrExp3() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("true or false and false").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testAndExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("true and false").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testAndExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("not false and false").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testAndExp3() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("not (false and false)").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("not false").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("true").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testEqualExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 = 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testEqualExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 = 2").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testEqualExp3() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 eq 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotEqualExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 != 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotEqualExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 != 2").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotEqualExp3() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 ne 2").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 >= 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 >= 2").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp3() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 >= 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp4() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 ge 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterThanExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 > 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testGreaterThanExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 > 2").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testGreaterThanExp3() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 > 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterThanExp4() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 gt 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessEqualExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 <= 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessEqualExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 <= 2").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessEqualExp3() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 <= 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessEqualExp4() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 le 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessThanExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 < 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessThanExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 < 2").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessThanExp3() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 < 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessThanExp4() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 lt 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testInExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 in (1,2,3,null)").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testInExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("4 in (1,2,3,null)").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotInExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 not in (1,2,3,null)").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotInExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("4 not in (1,2,3,null)").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testBetweenExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 between 1 and 3").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testBetweenExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("4 between 1 and 3").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotBetweenExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("2 not between 1 and 3").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotBetweenExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("4 not between 1 and 3").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testIsNullExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("null is null").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testIsNullExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 is null").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testIsNotNullExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("null is not null").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testIsNotNullExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("1 is not null").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLikeExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("'SCOTT' LIKE '_CO__'").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotLikeExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("'SCOTT' NOT LIKE '_CO__'").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotLikeExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("'SCOTT' NOT LIKE ''").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testMatchExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("'abcAde' ~ '[a-zA-Z]*'").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testMatchExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("'aBcAde' ~* '[a-z]*'").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotMatchExp() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("'abcAde' !~ '[a-zA-Z]*'").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotMatchExp2() throws Exception {
		BooleanExpression boolExp =
			new RuleParser("'aBcAde' !~* '[a-z]*'").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testAddSubExp() throws Exception {
		Expression exp = new RuleParser("1 + 2").parseExpression();
		assertEquals("1", new Integer(3), exp.evaluateValue(null));
	}

	public void testAddSubExp2() throws Exception {
		Expression exp = new RuleParser("1 - 2").parseExpression();
		assertEquals("1", new Integer(-1), exp.evaluateValue(null));
	}

	public void testAddSubExp3() throws Exception {
		Expression exp = new RuleParser("1 + 2 * 3").parseExpression();
		assertEquals("1", new Integer(7), exp.evaluateValue(null));
	}

	public void testAddSubExp4() throws Exception {
		Expression exp = new RuleParser("(1 + 2) * 3").parseExpression();
		assertEquals("1", new Integer(9), exp.evaluateValue(null));
	}

	public void testMultDivModExp() throws Exception {
		Expression exp = new RuleParser("2 * 3").parseExpression();
		assertEquals("1", new Integer(6), exp.evaluateValue(null));
	}

	public void testMultDivModExp2() throws Exception {
		Expression exp = new RuleParser("5 / 2").parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(null));
	}

	public void testMultDivModExp3() throws Exception {
		Expression exp = new RuleParser("5 % 2").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testMultDivModExp4() throws Exception {
		Expression exp = new RuleParser("12 / 3 * 4").parseExpression();
		assertEquals("1", new Integer(16), exp.evaluateValue(null));
	}

	public void testMultDivModExp5() throws Exception {
		Expression exp = new RuleParser("12 / (3 * 4)").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testMultDivModExp6() throws Exception {
		Expression exp = new RuleParser("1.2 * 1.2").parseExpression();
		assertEquals("1", new Double(1.44), exp.evaluateValue(null));
	}

	public void testIntegerExp() throws Exception {
		Expression exp = new RuleParser("1").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testLongExp() throws Exception {
		Expression exp = new RuleParser("10000000000").parseExpression();
		assertEquals("1", new Long(10000000000L), exp.evaluateValue(null));
	}

	public void testDoubleExp() throws Exception {
		Expression exp = new RuleParser("1.1").parseExpression();
		assertEquals("1", new Double("1.1"), exp.evaluateValue(null));
	}

	public void testStringExp() throws Exception {
		Expression exp = new RuleParser("'123'").parseExpression();
		assertEquals("1", "123", exp.evaluateValue(null));
	}

	public void testGetVariableExp() throws Exception {
		Flowlet rule = new Flowlet();
		InputType inputType = new InputType();
		ArgType argType = new ArgType("aaa", Integer.class);
		inputType.addArgType(argType);
		rule.setInputType(inputType);

		Expression exp = new RuleParser("aaa + 2").parseExpression();
		Map parameters = new EMap();
		parameters.put("aaa", new Integer(1));
		assertEquals(
			"1",
			new Integer(3),
			exp.evaluateValue(new RuleContextImpl(rule, parameters)));
	}

	public void testSetVariableExp() throws Exception {
		Flowlet rule = new Flowlet();
		LocalType localType = new LocalType();
		VarType varType = new VarType("aaa", Integer.class);
		localType.addVarType(varType);
		rule.setLocalType(localType);

		Expression exp = new RuleParser("aaa := 2").parseExpression();
		RuleContext ruleContext = new RuleContextImpl(rule, null);
		assertEquals("1", new Integer(2), exp.evaluateValue(ruleContext));
		assertEquals("2", new Integer(2), ruleContext.getValue("aaa"));
	}

	public void testSetVariableExp2() throws Exception {
		Flowlet rule = new Flowlet();
		InputType inputType = new InputType();
		ArgType argType = new ArgType("mybean", MyBean.class);
		inputType.addArgType(argType);
		rule.setInputType(inputType);

		Expression exp = new RuleParser("mybean.aaa := 2").parseExpression();
		Map parameters = new EMap();
		MyBean mybean = new MyBean();
		parameters.put("mybean", mybean);
		RuleContext ruleContext = new RuleContextImpl(rule, parameters);
		assertEquals("1", new Integer(2), exp.evaluateValue(ruleContext));
		assertEquals("2", 2, mybean.getAaa());
	}

	public void testToTImestampExp() throws Exception {
		Expression exp =
			new RuleParser("to_timestamp('20021119', 'yyyyMMdd')")
				.parseExpression();
		assertEquals(
			"1",
			Conversion.toTimestamp("20021119", "yyyyMMdd"),
			exp.evaluateValue(null));
	}

	public void testToStringExp() throws Exception {
		Expression exp =
			new RuleParser("to_string(1234, '#,###')").parseExpression();
		assertEquals("1", "1,234", exp.evaluateValue(null));
	}

	public void testToStringExp2() throws Exception {
		Flowlet rule = new Flowlet();
		InputType inputType = new InputType();
		ArgType argType = new ArgType("aaa", Timestamp.class);
		inputType.addArgType(argType);
		rule.setInputType(inputType);

		Expression exp =
			new RuleParser("to_string(aaa, 'yyyy-MM-dd')").parseExpression();
		Map parameters = new HashMap();
		parameters.put("aaa", Conversion.toTimestamp("20021119", "yyyyMMdd"));
		assertEquals(
			"1",
			"2002-11-19",
			exp.evaluateValue(new RuleContextImpl(rule, parameters)));
	}

	public void testToBigDecimalExp() throws Exception {
		Expression exp =
			new RuleParser("to_bigdecimal('1234')").parseExpression();
		assertEquals("1", new BigDecimal(1234), exp.evaluateValue(null));
	}

	public void testToDoubleExp() throws Exception {
		Expression exp = new RuleParser("to_double(1234)").parseExpression();
		assertEquals("1", new Double(1234), exp.evaluateValue(null));
	}

	public void testToLongExp() throws Exception {
		Expression exp = new RuleParser("to_long('1234')").parseExpression();
		assertEquals("1", new Long(1234), exp.evaluateValue(null));
	}

	public void testToIntegerExp() throws Exception {
		Expression exp = new RuleParser("to_integer('1234')").parseExpression();
		assertEquals("1", new Integer(1234), exp.evaluateValue(null));
	}

	public void testNowExp() throws Exception {
		Expression exp = new RuleParser("now()").parseExpression();
		assertEquals(
			"1",
			true,
			exp.evaluateValue(null) instanceof java.sql.Timestamp);
	}

	public void testNullExp() throws Exception {
		Expression exp = new RuleParser("null").parseExpression();
		assertEquals("1", null, exp.evaluateValue(null));
	}

	public void testCaseExp() throws Exception {
		Expression exp =
			new RuleParser("case when false then 'aaa' when true then 'bbb' end")
				.parseExpression();
		assertEquals("1", "bbb", exp.evaluateValue(null));
	}

	public void testCaseExp2() throws Exception {
		Expression exp =
			new RuleParser("case when false then 'aaa' when false then 'bbb' else 'ccc' end")
				.parseExpression();
		assertEquals("1", "ccc", exp.evaluateValue(null));
	}

	public void testCaseExp3() throws Exception {
		Flowlet rule = new Flowlet();
		InputType inputType = new InputType();
		ArgType argType = new ArgType("aaa", String.class);
		inputType.addArgType(argType);
		rule.setInputType(inputType);

		Expression exp =
			new RuleParser("case aaa when '1' then 'aaa' when '2' then 'bbb' else 'ccc' end")
				.parseExpression();
		Map parameters = new HashMap();
		parameters.put("aaa", "2");
		assertEquals(
			"1",
			"bbb",
			exp.evaluateValue(new RuleContextImpl(rule, parameters)));
	}

	public void testCoalesceExp() throws Exception {
		Expression exp =
			new RuleParser("coalesce(null, '111')").parseExpression();
		assertEquals("1", "111", exp.evaluateValue(null));
	}

	public void testConcatenateExp() throws Exception {
		Expression exp = new RuleParser("'aaa' || '111'").parseExpression();
		assertEquals("1", "aaa111", exp.evaluateValue(null));
	}

	public void testNvlExp() throws Exception {
		Expression exp = new RuleParser("nvl(null, '111')").parseExpression();
		assertEquals("1", "111", exp.evaluateValue(null));
	}

	public void testNvlExp2() throws Exception {
		Expression exp = new RuleParser("nvl('222', '111')").parseExpression();
		assertEquals("1", "222", exp.evaluateValue(null));
	}

	public void testNullifExp() throws Exception {
		Expression exp =
			new RuleParser("nullif('aaa', 'aaa')").parseExpression();
		assertEquals("1", null, exp.evaluateValue(null));
	}

	public void testNullifExp2() throws Exception {
		Expression exp =
			new RuleParser("nullif('aaa', null)").parseExpression();
		assertEquals("1", "aaa", exp.evaluateValue(null));
	}

	public void testNullifExp3() throws Exception {
		Expression exp =
			new RuleParser("nullif(null, 'aaa')").parseExpression();
		assertEquals("1", null, exp.evaluateValue(null));
	}

	public void testAbs() throws Exception {
		Expression exp =
			new RuleParser("java.lang.Math.abs(-1)").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testAbs2() throws Exception {
		Expression exp = new RuleParser("abs(-1) * 2").parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(null));
	}

	public void testAscii() throws Exception {
		Expression exp = new RuleParser("ascii('x')").parseExpression();
		assertEquals("1", new Integer(120), exp.evaluateValue(null));
	}

	public void testBtrim() throws Exception {
		Expression exp = new RuleParser("btrim(' x ')").parseExpression();
		assertEquals("1", "x", exp.evaluateValue(null));
	}

	public void testBtrim2() throws Exception {
		Expression exp = new RuleParser("btrim('aba', 'a')").parseExpression();
		assertEquals("1", "b", exp.evaluateValue(null));
	}

	public void testBtrim3() throws Exception {
		Expression exp =
			new RuleParser("btrim('xyxtrimyyx', 'xy')").parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testBtrim4() throws Exception {
		Expression exp = new RuleParser("btrim(' abc ')").parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testTrim() throws Exception {
		Expression exp =
			new RuleParser("trim(both from ' abc ')").parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testTrim2() throws Exception {
		Expression exp =
			new RuleParser("trim(leading from ' abc ')").parseExpression();
		assertEquals("1", "abc ", exp.evaluateValue(null));
	}

	public void testTrim3() throws Exception {
		Expression exp =
			new RuleParser("trim(trailing from ' abc ')").parseExpression();
		assertEquals("1", " abc", exp.evaluateValue(null));
	}

	public void testTrim4() throws Exception {
		Expression exp =
			new RuleParser("trim(both 'xy' from 'xyxtrimyyx')")
				.parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testTrim5() throws Exception {
		Expression exp =
			new RuleParser("trim('xy' from 'xyxtrimyyx')").parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testTrim6() throws Exception {
		Expression exp =
			new RuleParser("trim(from '   trim   ')").parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testLtrim() throws Exception {
		Expression exp =
			new RuleParser("ltrim('zzzytrim', 'xyz')").parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testLtrim2() throws Exception {
		Expression exp = new RuleParser("ltrim(' abc')").parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testRtrim() throws Exception {
		Expression exp =
			new RuleParser("rtrim('trimxxxx', 'x')").parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testRtrim2() throws Exception {
		Expression exp = new RuleParser("rtrim('abc ')").parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testChr() throws Exception {
		Expression exp = new RuleParser("chr(65)").parseExpression();
		assertEquals("1", "A", exp.evaluateValue(null));
	}

	public void testLength() throws Exception {
		Expression exp = new RuleParser("length('111')").parseExpression();
		assertEquals("1", new Integer(3), exp.evaluateValue(null));
	}

	public void testLower() throws Exception {
		Expression exp = new RuleParser("lower('ABC')").parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testLpad() throws Exception {
		Expression exp = new RuleParser("lpad('123', 5)").parseExpression();
		assertEquals("1", "  123", exp.evaluateValue(null));
	}

	public void testLpad2() throws Exception {
		Expression exp =
			new RuleParser("lpad('123', 6, 'xy')").parseExpression();
		assertEquals("1", "xyx123", exp.evaluateValue(null));
	}

	public void testRepeat() throws Exception {
		Expression exp = new RuleParser("repeat('Pg', 4)").parseExpression();
		assertEquals("1", "PgPgPgPg", exp.evaluateValue(null));
	}

	public void testReplace() throws Exception {
		Expression exp =
			new RuleParser("replace('123452', '2', 'abc')").parseExpression();
		assertEquals("1", "1abc345abc", exp.evaluateValue(null));
	}

	public void testRpad() throws Exception {
		Expression exp = new RuleParser("rpad('123', 5)").parseExpression();
		assertEquals("1", "123  ", exp.evaluateValue(null));
	}

	public void testRpad2() throws Exception {
		Expression exp =
			new RuleParser("rpad('123', 6, 'xy')").parseExpression();
		assertEquals("1", "123xyx", exp.evaluateValue(null));
	}

	public void testStrpos() throws Exception {
		Expression exp =
			new RuleParser("strpos('abcde', 'bc')").parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(null));
	}

	public void testPosition() throws Exception {
		Expression exp =
			new RuleParser("position('bc' in 'abcde')").parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(null));
	}

	public void testSubstr() throws Exception {
		Expression exp =
			new RuleParser("substr('abcde', 2, 3)").parseExpression();
		assertEquals("1", "bcd", exp.evaluateValue(null));
	}

	public void testSubstr2() throws Exception {
		Expression exp = new RuleParser("substr('abcde', 2)").parseExpression();
		assertEquals("1", "bcde", exp.evaluateValue(null));
	}

	public void testSubstring() throws Exception {
		Expression exp =
			new RuleParser("substring('abcde' from 2 for 3)").parseExpression();
		assertEquals("1", "bcd", exp.evaluateValue(null));
	}

	public void testSubstring2() throws Exception {
		Expression exp =
			new RuleParser("substring('abcde' from 2)").parseExpression();
		assertEquals("1", "bcde", exp.evaluateValue(null));
	}

	public void testUpper() throws Exception {
		Expression exp = new RuleParser("upper('abc')").parseExpression();
		assertEquals("1", "ABC", exp.evaluateValue(null));
	}

	public void testTranslate() throws Exception {
		Expression exp =
			new RuleParser("translate('12345', '14', 'ad')").parseExpression();
		assertEquals("1", "a23d5", exp.evaluateValue(null));
	}

	public void testCeil() throws Exception {
		Expression exp = new RuleParser("ceil(12.6)").parseExpression();
		assertEquals("1", new Double("13"), exp.evaluateValue(null));
	}

	public void testCeil2() throws Exception {
		Expression exp = new RuleParser("ceil(-12.6)").parseExpression();
		assertEquals("1", new Double("-12"), exp.evaluateValue(null));
	}

	public void testDegrees() throws Exception {
		Expression exp = new RuleParser("degrees(0.5)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("28.64788975654116"), d);
	}

	public void testExp() throws Exception {
		Expression exp = new RuleParser("exp(1.0)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("2.7182818284590455"), d);
	}

	public void testFloor() throws Exception {
		Expression exp = new RuleParser("floor(-42.8)").parseExpression();
		assertEquals("1", new Double("-43"), exp.evaluateValue(null));
	}

	public void testFloor2() throws Exception {
		Expression exp = new RuleParser("floor(42.8)").parseExpression();
		assertEquals("1", new Double("42"), exp.evaluateValue(null));
	}

	public void testLn() throws Exception {
		Expression exp = new RuleParser("ln(2.0)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("0.6931471805599453"), d);
	}

	public void testPow() throws Exception {
		Expression exp = new RuleParser("pow(9.0, 3.0)").parseExpression();
		assertEquals("1", new Double("729"), exp.evaluateValue(null));
	}

	public void testRadians() throws Exception {
		Expression exp = new RuleParser("radians(45.0)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("0.7853981633974483"), d);
	}

	public void testRandom() throws Exception {
		Expression exp = new RuleParser("random()").parseExpression();
		Object ret = exp.evaluateValue(null);
		System.out.println(ret);
		assertNotNull("1", ret);
	}

	public void testRound() throws Exception {
		Expression exp = new RuleParser("round(42.4)").parseExpression();
		assertEquals("1", new Long("42"), exp.evaluateValue(null));
	}

	public void testNew() throws Exception {
		Expression exp =
			new RuleParser("new java.math.BigDecimal('1')").parseExpression();
		assertEquals("1", new BigDecimal("1"), exp.evaluateValue(null));
	}

	public void testOut() throws Exception {
		Expression exp =
			new RuleParser("out.println('Hello')").parseExpression();
		assertEquals(
			"1",
			null,
			exp.evaluateValue(new RuleContextImpl(new Flowlet(), null)));
	}

	public void testNewArray() throws Exception {
		Expression exp =
			new RuleParser("new java.lang.String[2]").parseExpression();
		String[] ret = (String[]) exp.evaluateValue(null);
		assertEquals("1", 2, ret.length);
	}

	public void testNewArray2() throws Exception {
		Expression exp =
			new RuleParser("new java.lang.String[]{'aaa', 'bbb'}")
				.parseExpression();
		String[] ret = (String[]) exp.evaluateValue(null);
		assertEquals("1", 2, ret.length);
		assertEquals("2", "aaa", ret[0]);
		assertEquals("3", "bbb", ret[1]);
	}

	public void testGetArrayVariableExp() throws Exception {
		Flowlet rule = new Flowlet();
		InputType inputType = new InputType();
		ArgType argType = new ArgType("aaa", new String[0].getClass());
		inputType.addArgType(argType);
		rule.setInputType(inputType);

		Expression exp = new RuleParser("aaa[1]").parseExpression();
		Map parameters = new HashMap();
		parameters.put("aaa", new String[] { "111", "222", "333" });
		assertEquals(
			"1",
			"222",
			exp.evaluateValue(new RuleContextImpl(rule, parameters)));
	}

	public void testSetArrayVariableExp() throws Exception {
		Flowlet rule = new Flowlet();
		InputType inputType = new InputType();
		ArgType argType = new ArgType("aaa", new String[0].getClass());
		inputType.addArgType(argType);
		rule.setInputType(inputType);

		Expression exp = new RuleParser("aaa[0] := '111'").parseExpression();
		Map parameters = new HashMap();
		String[] array = new String[1];
		parameters.put("aaa", array);
		assertEquals(
			"1",
			"111",
			exp.evaluateValue(new RuleContextImpl(rule, parameters)));
		assertEquals("2", "111", array[0]);
	}

	public void testExecuteExp() throws Exception {
		String s =
			"execute('/test/org/seasar/nazuna/Hello2.xml', name := 'Nazuna')";
		Expression exp = new RuleParser(s).parseExpression();
		assertEquals("1", null, exp.evaluateValue(null));
	}

	public void testExecuteRuletExp() throws Exception {
		String s = "executeRulet('test.org.seasar.nazuna.AddRulet', 1, 2)";
		Expression exp = new RuleParser(s).parseExpression();
		assertEquals("1", new Integer(3), exp.evaluateValue(null));
	}

	public void testSign() throws Exception {
		Expression exp = new RuleParser("sign(2)").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));

		Expression exp2 = new RuleParser("sign(-2)").parseExpression();
		assertEquals("2", new Integer(-1), exp2.evaluateValue(null));

		Expression exp3 = new RuleParser("sign(0)").parseExpression();
		assertEquals("3", new Integer(0), exp3.evaluateValue(null));
	}

	public void testSign2() throws Exception {
		Expression exp = new RuleParser("sign(2.0)").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));

		Expression exp2 = new RuleParser("sign(-2.0)").parseExpression();
		assertEquals("2", new Integer(-1), exp2.evaluateValue(null));

		Expression exp3 = new RuleParser("sign(0.0)").parseExpression();
		assertEquals("3", new Integer(0), exp3.evaluateValue(null));
	}

	public void testSqrt() throws Exception {
		Expression exp = new RuleParser("sqrt(2.0)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("1.4142135623730951"), d);
	}

	public void testTrunc() throws Exception {
		Expression exp = new RuleParser("trunc(42.8)").parseExpression();
		assertEquals("1", new Double("42"), exp.evaluateValue(null));
	}

	public void testTrunc2() throws Exception {
		Expression exp = new RuleParser("trunc(-42.8)").parseExpression();
		assertEquals("1", new Double("-42"), exp.evaluateValue(null));
	}

	public void testAcos() throws Exception {
		Expression exp = new RuleParser("acos(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testAsin() throws Exception {
		Expression exp = new RuleParser("asin(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testAtan() throws Exception {
		Expression exp = new RuleParser("atan(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testAtan2() throws Exception {
		Expression exp = new RuleParser("atan2(1.0, 1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testCos() throws Exception {
		Expression exp = new RuleParser("cos(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testSin() throws Exception {
		Expression exp = new RuleParser("sin(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testYen() throws Exception {
		Expression exp = new RuleParser("'a\b'").parseExpression();
		assertEquals("1", "a\b", exp.evaluateValue(null));
	}

	public void testToBooleanExp() throws Exception {
		Expression exp = new RuleParser("to_boolean(true)").parseExpression();
		assertEquals("1", Boolean.TRUE, exp.evaluateValue(null));
	}

	public void testIsTrueExp() throws Exception {
		BooleanExpression exp =
			new RuleParser("true is true").parseBooleanExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	public void testIsTrueExp2() throws Exception {
		BooleanExpression exp =
			new RuleParser("to_boolean(true) is true").parseBooleanExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	public void testIsFalseExp() throws Exception {
		BooleanExpression exp =
			new RuleParser("false is false").parseBooleanExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	public void testIsFalseExp2() throws Exception {
		BooleanExpression exp =
			new RuleParser("to_boolean(false) is false")
				.parseBooleanExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	protected void setUp() throws Exception {
	}

	protected void tearDown() throws Exception {
	}

	public static Test suite() {
		return new TestSuite(RuleParserTest.class);
	}

	public static void main(String[] args) {
		junit.textui.TestRunner.main(
			new String[] { RuleParserTest.class.getName()});
	}
}