package test.org.seasar.naming;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.LinkRef;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NameClassPair;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.Reference;
import javax.naming.Referenceable;

import junit.framework.Test;
import junit.framework.TestSuite;

import org.seasar.naming.NamingContext;
import org.seasar.util.Serializer;

public class NamingContextTest extends NamingTestCase {

    private Context _ctx;

    public NamingContextTest(String name) {
        super(name);
    }

    public void testBind() throws Exception {
        String s = "hello";
        _ctx.bind("hello", s);
        assertEquals("1", s, _ctx.lookup("hello"));

        _ctx.bind("/hello2", s);
        assertEquals("2", s, _ctx.lookup("hello2"));

        _ctx.bind("test/hello", s);
        assertEquals("3", s, _ctx.lookup("test/hello"));
        assertEquals("4", NamingContext.class, _ctx.lookup("test").getClass());
        try {
            _ctx.bind("test/hello", s);
            fail("5");
        } catch (NameAlreadyBoundException ex) {
            System.out.println(ex);
        }
    }

    public void testRebind() throws Exception {
        String s = "hello";
        String s2 = "hello2";
        _ctx.rebind("hello", s);
        assertEquals("1", s, _ctx.lookup("hello"));

        _ctx.rebind("/hello2", s);
        assertEquals("2", s, _ctx.lookup("hello2"));

        _ctx.rebind("test/hello", s);
        assertEquals("3", s, _ctx.lookup("test/hello"));
        assertEquals("4", NamingContext.class, _ctx.lookup("test").getClass());
        _ctx.rebind("test/hello", s2);
        assertEquals("5", s2, _ctx.lookup("test/hello"));
    }

    public void testLookup() throws Exception {
        Context test = _ctx.createSubcontext("/test");
        assertEquals("1", test, test.lookup(""));
        assertEquals("2", test, test.lookup("/test"));
        assertEquals("3", test, test.lookup("/test/"));

        String s = "hello";
        test.bind("hello", s);
        assertEquals("4", s, _ctx.lookup("/test/hello"));

        test.bind("/hello", new LinkRef("/test/hello"));
        assertEquals("5", s, _ctx.lookup("/hello"));

        Reference ref = new Reference("java.lang.String", TestObjectFactory.class.getName(), null);
        test.bind("/ref", ref);
        assertEquals("6", TestObjectFactory.HELLO, _ctx.lookup("/ref"));
        Referenceable referenceable = new TestReferenceable();
        test.bind("/referenceable", referenceable);
        assertEquals("7", TestObjectFactory.HELLO, _ctx.lookup("/referenceable"));

        Reference ref2 = new Reference("java.util.Date");
        test.bind("/ref2", ref2);
        assertNotNull("8", _ctx.lookup("/ref2"));
        _ctx.bind("java:comp/s", s);
        assertEquals("9", s, _ctx.lookup("java:comp/s"));
        assertEquals("10", s, _ctx.lookup("java:comp/s"));
    }

    public void testUnbind() throws Exception {
        String s = "hello";
        _ctx.bind("/test/hello", s);
        _ctx.unbind("test/hello");
        try {
            _ctx.lookup("test/hello");
            fail("1");
        } catch (NameNotFoundException ex) {
            System.out.println(ex);
        }
        try {
            _ctx.unbind("test");
            fail("2");
        } catch (NamingException ex) {
            System.out.println(ex);
        }
    }

    public void testRename() throws Exception {
        String s = "hello";
        _ctx.bind("hello", s);
        _ctx.rename("hello", "hello2");
        try {
            _ctx.lookup("hello");
            fail("1");
        } catch (NameNotFoundException ex) {
            System.out.println(ex);
        }
        assertEquals("2", s, _ctx.lookup("hello2"));
    }

    public void testCreateSubcontext() throws Exception {
        _ctx.createSubcontext("test");
        assertEquals("1", NamingContext.class, _ctx.lookup("test").getClass());

        _ctx.createSubcontext("test/test2");
        assertEquals("2", NamingContext.class, _ctx.lookup("test/test2").getClass());

        _ctx.createSubcontext("test3/test4");
        assertEquals("3", NamingContext.class, _ctx.lookup("test3/test4").getClass());
    }

    public void testDestroySubcontext() throws Exception {
        _ctx.createSubcontext("test");
        _ctx.destroySubcontext("test");
        try {
            _ctx.lookup("test");
            fail("1");
        } catch (NameNotFoundException ex) {
            System.out.println(ex);
        }

        _ctx.createSubcontext("test/test2");
        _ctx.bind("test/hello", "hello");
        _ctx.destroySubcontext("test");
        try {
            _ctx.lookup("test");
            fail("2");
        } catch (NameNotFoundException ex) {
            System.out.println(ex);
        }
    }

    public void testSerialize() throws Exception {
        Serializer.serialize(_ctx.lookup(""));
    }

    public void testList() throws Exception {
        _ctx.createSubcontext("test/test2");
        _ctx.bind("test/hello", "hello");
        _ctx.bind("test/ref", new Reference("java.lang.String", TestObjectFactory.class.getName(), null));
        _ctx.bind("test/referenceable", new TestReferenceable());
        NamingEnumeration e = _ctx.list("test");

        NameClassPair ncp = (NameClassPair) e.next();
        assertEquals("1", "test2", ncp.getName());
        assertEquals("2", NamingContext.class.getName(), ncp.getClassName());

        NameClassPair ncp2 = (NameClassPair) e.next();
        assertEquals("3", "hello", ncp2.getName());
        assertEquals("4", String.class.getName(), ncp2.getClassName());

        NameClassPair ncp3 = (NameClassPair) e.next();
        assertEquals("5", "ref", ncp3.getName());
        assertEquals("6", String.class.getName(), ncp3.getClassName());

        NameClassPair ncp4 = (NameClassPair) e.next();
        assertEquals("7", "referenceable", ncp4.getName());
        assertEquals("8", String.class.getName(), ncp4.getClassName());
    }

    public void testListBindings() throws Exception {
        _ctx.createSubcontext("test/test2");
        _ctx.bind("test/hello", "hello");
        _ctx.bind("test/ref", new Reference("java.lang.String", TestObjectFactory.class.getName(), null));
        _ctx.bind("test/referenceable", new TestReferenceable());
        NamingEnumeration e = _ctx.listBindings("test");

        Binding b = (Binding) e.next();
        assertEquals("1", "test2", b.getName());
        assertEquals("2", NamingContext.class.getName(), b.getClassName());
        assertEquals("3", NamingContext.class, b.getObject().getClass());

        Binding b2 = (Binding) e.next();
        assertEquals("4", "hello", b2.getName());
        assertEquals("5", String.class.getName(), b2.getClassName());
        assertEquals("6", "hello", b2.getObject());

        Binding b3 = (Binding) e.next();
        assertEquals("7", "ref", b3.getName());
        assertEquals("8", String.class.getName(), b3.getClassName());
        assertEquals("9", "hello", b3.getObject());

        Binding b4 = (Binding) e.next();
        assertEquals("10", "referenceable", b4.getName());
        assertEquals("11", String.class.getName(), b4.getClassName());
        assertEquals("12", "hello", b4.getObject());
    }

    public void testComposeName() throws Exception {
        assertEquals("1", "test/hello",
                ((Context) _ctx.lookup("")).composeName("hello", "test").toString());
    }

    public void testGetEnvironment() throws Exception {
        Context test = _ctx.createSubcontext("test");
        assertEquals("1", "org.seasar.naming.NamingContextFactory",
                test.getEnvironment().get(Context.INITIAL_CONTEXT_FACTORY));
    }

    public void testAddToEnvironment() throws Exception {
        Context test = _ctx.createSubcontext("test");
        test.addToEnvironment("key", "value");
        assertEquals("1", "value", test.getEnvironment().get("key"));
    }

    public void testRemoveFromEnvironment() throws Exception {
        Context test = _ctx.createSubcontext("test");
        assertEquals("1", "org.seasar.naming.NamingContextFactory",
                test.removeFromEnvironment(Context.INITIAL_CONTEXT_FACTORY));
    }

    public void testGetNameInNamespace() throws Exception {
        Context test = _ctx.createSubcontext("test");
        assertEquals("1", "test", test.getNameInNamespace());
    }

    protected void setUp() throws Exception {
        super.setUp();
        _env.put(Context.PROVIDER_URL, "localhost:0");
        _ctx = new InitialContext(_env);
    }

    protected void tearDown() throws Exception {
        super.tearDown();
        _ctx = null;
    }

    public static Test suite() {
        return new TestSuite(NamingContextTest.class);
    }

    public static void main(String[] args) {
        junit.textui.TestRunner.main(new String[]{NamingContextTest.class.getName()});
    }
}
