package org.seasar.system;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.rmi.RemoteException;
import java.util.Map;

import org.seasar.util.Reflector;
import org.seasar.util.SMap;
import org.seasar.util.SeasarContext;
import org.seasar.util.SeasarException;

public final class MBeanProxy implements InvocationHandler, Externalizable {

    final static long serialVersionUID = 4230846384158691664L;
    private static transient Map _sigsMap = new SMap();
    private transient String _name;
    private transient String _hostName;
    private transient int _port;

	public static Object create(final Class mbeanInterface, final String name) throws SeasarException {
        return Proxy.newProxyInstance(mbeanInterface.getClassLoader(),
                new Class[]{mbeanInterface}, new MBeanProxy(name));
    }

	public static Object create(final Class mbeanInterface, final String name,
            final SeasarContext seasarContext) throws SeasarException {

        return create(mbeanInterface, name, seasarContext.getHostName(), seasarContext.getPort());
    }
    
    public static Object create(final Class mbeanInterface, final String name,
            final String url) throws SeasarException {

		if (url != null && url.length() > 0) {
            String host = null;
            int port = 1108;
            int colonPos = url.lastIndexOf(':');
	        int slashPos = url.lastIndexOf('/');
	        if (slashPos >= 0) {
	        	if (colonPos > slashPos) {	
	        		host = url.substring(slashPos + 1, colonPos);
	            	port = Integer.parseInt(url.substring(colonPos + 1));
	        	} else {
	        		host = url.substring(slashPos + 1);
	        	}
	        } else if (colonPos > 0) {
	        	host = url.substring(0, colonPos);
	            port = Integer.parseInt(url.substring(colonPos + 1));
	        } else {
	        	host = url;
	        }
            return create(mbeanInterface, name, host, port);
        } else {
        	return create(mbeanInterface, name);
        }
    }
        
    public static Object create(final Class mbeanInterface, final String name,
            final String host, final int port) {

        return Proxy.newProxyInstance(mbeanInterface.getClassLoader(),
                new Class[]{mbeanInterface}, new MBeanProxy(name, host, port));
    }

    public MBeanProxy() { }


	private MBeanProxy(final String name) {
        _name = name;
        _hostName = RMIAdaptorService.getInstance().getHostName();
        _port = RMIAdaptorService.getInstance().getPort();
    }
    
    private MBeanProxy(final String name,
    		final String hostName, final int port) {
    			
        _name = name;
        _hostName = hostName;
        _port = port;
    }

    public Object invoke(final Object proxy, final Method method, final Object[] args)
             throws Throwable {

		if (method.getDeclaringClass().equals(Object.class)) {
			return Reflector.invoke(method, this, args);
		} else {
			String[] sigs = getSigs(method);
			RMIAdaptor rmiAdaptor = RMIConnector.getRMIAdaptor(_hostName, _port);
			try {
				return rmiAdaptor.invoke(
	                _name, method.getName(), args, sigs);
			} catch (RemoteException ex) {
				rmiAdaptor = RMIConnector.regetRMIAdaptor(_hostName, _port);
				return rmiAdaptor.invoke(
	                _name, method.getName(), args, sigs);
			}
		}
    }

    public void writeExternal(final ObjectOutput out) throws IOException {
        out.writeObject(_name);
        out.writeObject(_hostName);
        out.writeInt(_port);
    }

    public void readExternal(final ObjectInput in)
             throws IOException, ClassNotFoundException {

        _name = (String) in.readObject();
        _hostName = (String) in.readObject();
        _port = in.readInt();
    }
    
    public boolean equals(Object o) {
    	if (o == null) {
    		return false;
    	}
    	if (Proxy.isProxyClass(o.getClass())) {
    		o = Proxy.getInvocationHandler(o);
    	}
    	return this == o;
    }

    private static String[] getSigs(final Method method) {
        String[] sigs = (String[]) _sigsMap.get(method);
        if (sigs != null) {
            return sigs;
        }
        synchronized (MBeanProxy.class) {
            sigs = (String[]) _sigsMap.get(method);
            if (sigs == null) {
                Class[] types = method.getParameterTypes();
                sigs = new String[types.length];
                for (int i = 0; i < types.length; i++) {
                    sigs[i] = types[i].getName();
                }
                _sigsMap.put(method, sigs);
            }
        }
        return sigs;
    }
}