package org.seasar.nazuna;

import java.util.ArrayList;
import java.util.List;

import org.seasar.util.EArrayList;
import org.seasar.util.EMap;
import org.seasar.util.Reflector;
import org.seasar.util.SeasarException;

public final class RuleParser {

	private transient RuleTokenizer _tokenizer;
	private int _token;

	public RuleParser(String s) throws SeasarException {
		_tokenizer = new RuleTokenizer(s);
		nextToken();
	}

	public synchronized BooleanExpression parseBooleanExpression()
		throws SeasarException {
		BooleanExpression boolExp = getOrExp();
		expect(RuleTokenizer.EOF);
		return boolExp;
	}

	public synchronized Expression parseExpression() throws SeasarException {
		Expression exp = getAddSubExp();
		expect(RuleTokenizer.EOF);
		return exp;
	}

	private BooleanExpression getOrExp() throws SeasarException {
		BooleanExpression left = getAndExp();
		while (_token == RuleTokenizer.OR) {
			nextToken();
			left = new OrExp(left, getAndExp());
		}
		return left;
	}

	private BooleanExpression getAndExp() throws SeasarException {
		BooleanExpression left = getNotExp();
		while (_token == RuleTokenizer.AND) {
			nextToken();
			left = new AndExp(left, getNotExp());
		}
		return left;
	}

	private BooleanExpression getNotExp() throws SeasarException {
		switch (_token) {
			case RuleTokenizer.NOT :
				nextToken();
				BooleanExpression e = getSimpleBoolExp();
				return new NotExp(e);
			default :
				return getSimpleBoolExp();
		}
	}

	private BooleanExpression getSimpleBoolExp() throws SeasarException {
		switch (_token) {
			case RuleTokenizer.LEFT_PAREN :
				nextToken();
				BooleanExpression e = getOrExp();
				expect(RuleTokenizer.RIGHT_PAREN);
				return e;
			case RuleTokenizer.TRUE :
				nextToken();
				if (_token == RuleTokenizer.IS) {
					nextToken();
					switch (_token) {
						case RuleTokenizer.TRUE :
							nextToken();
							return BooleanExp.TRUE;
						case RuleTokenizer.FALSE :
							nextToken();
							return BooleanExp.FALSE;
					}
				} else {
					return BooleanExp.TRUE;
				}
			case RuleTokenizer.FALSE :
				nextToken();
				if (_token == RuleTokenizer.IS) {
					nextToken();
					switch (_token) {
						case RuleTokenizer.TRUE :
							nextToken();
							return BooleanExp.FALSE;
						case RuleTokenizer.FALSE :
							nextToken();
							return BooleanExp.TRUE;
					}
				} else {
					return BooleanExp.FALSE;
				}
			default :
				Expression targetExp = getAddSubExp();
				switch (_token) {
					case RuleTokenizer.EQUAL :
						nextToken();
						return new EqualExp(targetExp, getAddSubExp());
					case RuleTokenizer.NOT_EQUAL :
						nextToken();
						return new NotEqualExp(targetExp, getAddSubExp());
					case RuleTokenizer.GREATER_EQUAL :
						nextToken();
						return new GreaterEqualExp(targetExp, getAddSubExp());
					case RuleTokenizer.GREATER_THAN :
						nextToken();
						return new GreaterThanExp(targetExp, getAddSubExp());
					case RuleTokenizer.LESS_EQUAL :
						nextToken();
						return new LessEqualExp(targetExp, getAddSubExp());
					case RuleTokenizer.LESS_THAN :
						nextToken();
						return new LessThanExp(targetExp, getAddSubExp());
					case RuleTokenizer.IN :
						return getInExp(targetExp);
					case RuleTokenizer.BETWEEN :
						return getBetweenExp(targetExp);
					case RuleTokenizer.IS :
						nextToken();
						switch (_token) {
							case RuleTokenizer.NULL :
								nextToken();
								return new IsNullExp(targetExp);
							case RuleTokenizer.NOT :
								nextToken();
								expect(RuleTokenizer.NULL);
								return new IsNotNullExp(targetExp);
							case RuleTokenizer.TRUE :
								nextToken();
								return new IsTrueExp(targetExp);
							case RuleTokenizer.FALSE :
								nextToken();
								return new IsFalseExp(targetExp);
						}
					case RuleTokenizer.NOT :
						nextToken();
						switch (_token) {
							case RuleTokenizer.LIKE :
								nextToken();
								return new NotExp(
									new LikeExp(targetExp, getAddSubExp()));
							case RuleTokenizer.BETWEEN :
								return getNotBetweenExp(targetExp);
							case RuleTokenizer.IN :
								return new NotExp(getInExp(targetExp));
						}
					case RuleTokenizer.LIKE :
						nextToken();
						return new LikeExp(targetExp, getAddSubExp());
					case RuleTokenizer.MATCH :
						nextToken();
						return new MatchExp(targetExp, getAddSubExp());
					case RuleTokenizer.MATCH_IGNORE_CASE :
						nextToken();
						return new MatchExp(targetExp, getAddSubExp(), true);
					case RuleTokenizer.NOT_MATCH :
						nextToken();
						return new NotExp(
							new MatchExp(targetExp, getAddSubExp()));
					case RuleTokenizer.NOT_MATCH_IGNORE_CASE :
						nextToken();
						return new NotExp(
							new MatchExp(targetExp, getAddSubExp(), true));
				}
		}
		throw new SeasarException(
			"ESSR0032",
			new Object[] { RuleTokenizer.getTokenName(_token)});
	}

	private BooleanExpression getInExp(Expression targetExp)
		throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		List inExpList = new ArrayList();
		inExpList.add(getAddSubExp());
		while (_token == RuleTokenizer.COMMA) {
			nextToken();
			inExpList.add(getAddSubExp());
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new InExp(targetExp, NazunaUtil.toExpressionArray(inExpList));
	}

	private BooleanExpression getBetweenExp(Expression targetExp)
		throws SeasarException {
		nextToken();
		Expression fromExp = getAddSubExp();
		expect(RuleTokenizer.AND);
		Expression toExp = getAddSubExp();
		return new BetweenExp(targetExp, fromExp, toExp);
	}

	private BooleanExpression getNotBetweenExp(Expression targetExp)
		throws SeasarException {
		nextToken();
		Expression fromExp = getAddSubExp();
		expect(RuleTokenizer.AND);
		Expression toExp = getAddSubExp();
		return new NotBetweenExp(targetExp, fromExp, toExp);
	}

	private Expression getAddSubExp() throws SeasarException {
		Expression arg1Exp = getMultDivModExp();
		while (true) {
			switch (_token) {
				case RuleTokenizer.ADD :
					nextToken();
					arg1Exp = new AddExp(arg1Exp, getMultDivModExp());
					break;
				case RuleTokenizer.SUBTRACT :
					nextToken();
					arg1Exp = new SubtractExp(arg1Exp, getMultDivModExp());
					break;
				default :
					return arg1Exp;
			}
		}
	}

	private Expression getMultDivModExp() throws SeasarException {
		Expression arg1Exp = getConcatenateExp();
		while (true) {
			switch (_token) {
				case RuleTokenizer.MULTIPLY :
					nextToken();
					arg1Exp = new MultiplyExp(arg1Exp, getConcatenateExp());
					break;
				case RuleTokenizer.DIVIDE :
					nextToken();
					arg1Exp = new DivideExp(arg1Exp, getConcatenateExp());
					break;
				case RuleTokenizer.MOD :
					nextToken();
					arg1Exp = new ModExp(arg1Exp, getConcatenateExp());
					break;
				default :
					return arg1Exp;
			}
		}
	}

	private Expression getConcatenateExp() throws SeasarException {
		Expression exp = getSimpleExp();
		if (_token != RuleTokenizer.CONCATENATE) {
			return exp;
		}
		List exps = new ArrayList();
		exps.add(exp);
		while (_token == RuleTokenizer.CONCATENATE) {
			nextToken();
			exps.add(getSimpleExp());
		}
		return new ConcatenateExp(NazunaUtil.toExpressionArray(exps));
	}

	private Expression getSimpleExp() throws SeasarException {
		switch (_token) {
			case RuleTokenizer.LEFT_PAREN :
				nextToken();
				Expression e = getAddSubExp();
				expect(RuleTokenizer.RIGHT_PAREN);
				return e;
			case RuleTokenizer.DOUBLE :
				return getDoubleExp();
			case RuleTokenizer.LONG :
				return getLongExp();
			case RuleTokenizer.INTEGER :
				return getIntegerExp();
			case RuleTokenizer.QUOTED_STRING :
				return getStringExp();
			case RuleTokenizer.WORD :
				return getWordExp();
			case RuleTokenizer.TO_TIMESTAMP :
				return getToTimestampExp();
			case RuleTokenizer.TO_STRING :
				return getToStringExp();
			case RuleTokenizer.TO_BIGDECIMAL :
				return getToBigDecimalExp();
			case RuleTokenizer.TO_DOUBLE :
				return getToDoubleExp();
			case RuleTokenizer.TO_LONG :
				return getToLongExp();
			case RuleTokenizer.TO_INTEGER :
				return getToIntegerExp();
			case RuleTokenizer.TO_BOOLEAN :
				return getToBooleanExp();
			case RuleTokenizer.NOW :
				return getNowExp();
			case RuleTokenizer.NULL :
				return getNullExp();
			case RuleTokenizer.SUBSTRING :
				return getSubstringExp();
			case RuleTokenizer.POSITION :
				return getPositionExp();
			case RuleTokenizer.CASE :
				return getCaseExp();
			case RuleTokenizer.COALESCE :
				return getCoalesceExp();
			case RuleTokenizer.NVL :
				return getNvlExp();
			case RuleTokenizer.NULLIF :
				return getNullifExp();
			case RuleTokenizer.TRIM :
				return getTrimExp();
			case RuleTokenizer.NEW :
				return getNewExp();
			case RuleTokenizer.TRUE :
				nextToken();
				return new BooleanExp(true);
			case RuleTokenizer.FALSE :
				nextToken();
				return new BooleanExp(false);
			case RuleTokenizer.EXECUTE :
			case RuleTokenizer.EXECUTE_QUERY :
			case RuleTokenizer.EXECUTE_RS_QUERY :
			case RuleTokenizer.EXECUTE_SINGLE_QUERY :
			case RuleTokenizer.EXECUTE_UPDATE :
				return getExecuteExp(_token);
			case RuleTokenizer.EXECUTE_RULET :
				return getExecuteRuletExp();
			default :
				throw new SeasarException(
					"ESSR0032",
					new Object[] { RuleTokenizer.getTokenName(_token)});
		}
	}

	private Expression getDoubleExp() throws SeasarException {
		Expression e = new DoubleExp(_tokenizer.getDouble());
		nextToken();
		return e;
	}

	private Expression getLongExp() throws SeasarException {
		Expression e = new LongExp(_tokenizer.getLong());
		nextToken();
		return e;
	}

	private Expression getIntegerExp() throws SeasarException {
		Expression e = new IntegerExp(_tokenizer.getInteger());
		nextToken();
		return e;
	}

	private Expression getStringExp() throws SeasarException {
		String s = _tokenizer.getString();
		nextToken();
		if (_token == RuleTokenizer.QUOTED_STRING) {
			s = s + "'" + _tokenizer.getString();
			nextToken();
		}
		return new StringExp(s);
	}

	private Expression getToTimestampExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression p1 = getSimpleExp();
		Expression p2 = null;
		if (_token == RuleTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new ToTimestampExp(p1, pattern);
	}

	private Expression getToStringExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == RuleTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new ToStringExp(p1, pattern);
	}

	private Expression getToBigDecimalExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == RuleTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new ToBigDecimalExp(p1, pattern);
	}

	private Expression getToDoubleExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == RuleTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new ToDoubleExp(p1, pattern);
	}

	private Expression getToLongExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == RuleTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new ToLongExp(p1, pattern);
	}

	private Expression getToIntegerExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == RuleTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new ToIntegerExp(p1, pattern);
	}

	private Expression getToBooleanExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		BooleanExpression argExp = getOrExp();
		expect(RuleTokenizer.RIGHT_PAREN);
		return new ToBooleanExp(argExp);
	}

	private Expression getNowExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		expect(RuleTokenizer.RIGHT_PAREN);
		return new NowExp();
	}

	private Expression getNullExp() throws SeasarException {
		nextToken();
		return new NullExp();
	}

	private Expression getSubstringExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		List argExps = new ArrayList();
		Expression arg1Exp = getAddSubExp();
		argExps.add(arg1Exp);
		expect(RuleTokenizer.FROM);
		Expression fromExp = getAddSubExp();
		argExps.add(fromExp);
		Expression forExp = null;
		if (_token == RuleTokenizer.FOR) {
			nextToken();
			forExp = getAddSubExp();
			argExps.add(forExp);
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new MethodExp(
			"org.seasar.util.StringUtil",
			"substr",
			NazunaUtil.toExpressionArray(argExps));
	}

	private Expression getPositionExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression argExp = getAddSubExp();
		expect(RuleTokenizer.IN);
		Expression strExp = getAddSubExp();
		expect(RuleTokenizer.RIGHT_PAREN);
		List argExps = new ArrayList();
		argExps.add(strExp);
		argExps.add(argExp);
		return new MethodExp(
			"org.seasar.util.StringUtil",
			"strpos",
			NazunaUtil.toExpressionArray(argExps));
	}

	private Expression getCaseExp() throws SeasarException {
		nextToken();
		List boolExpList = new ArrayList();
		List expList = new ArrayList();
		if (_token == RuleTokenizer.WHEN) {
			while (_token == RuleTokenizer.WHEN) {
				nextToken();
				boolExpList.add(getOrExp());
				expect(RuleTokenizer.THEN);
				expList.add(getAddSubExp());
			}
		} else {
			Expression targetExp = getAddSubExp();
			while (_token == RuleTokenizer.WHEN) {
				nextToken();
				boolExpList.add(new EqualExp(targetExp, getAddSubExp()));
				expect(RuleTokenizer.THEN);
				expList.add(getAddSubExp());
			}
		}
		if (_token == RuleTokenizer.ELSE) {
			nextToken();
			boolExpList.add(new BooleanExp(true));
			expList.add(getAddSubExp());
		}
		expect(RuleTokenizer.END);
		BooleanExpression[] boolExps =
			NazunaUtil.toBooleanExpressionArray(boolExpList);
		Expression[] exps = NazunaUtil.toExpressionArray(expList);
		return new CaseExp(boolExps, exps);
	}

	private Expression getCoalesceExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		List expList = new ArrayList();
		expList.add(getAddSubExp());
		while (_token == RuleTokenizer.COMMA) {
			nextToken();
			expList.add(getAddSubExp());
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new CoalesceExp(NazunaUtil.toExpressionArray(expList));
	}

	private Expression getNvlExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression arg1Exp = getAddSubExp();
		expect(RuleTokenizer.COMMA);
		Expression arg2Exp = getAddSubExp();
		expect(RuleTokenizer.RIGHT_PAREN);
		return new NvlExp(arg1Exp, arg2Exp);
	}

	private Expression getNullifExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression arg1Exp = getAddSubExp();
		expect(RuleTokenizer.COMMA);
		Expression arg2Exp = getAddSubExp();
		expect(RuleTokenizer.RIGHT_PAREN);
		return new NullifExp(arg1Exp, arg2Exp);
	}

	private Expression getTrimExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		String methodName = "btrim";
		switch (_token) {
			case RuleTokenizer.LEADING :
				methodName = "ltrim";
				nextToken();
				break;
			case RuleTokenizer.TRAILING :
				methodName = "rtrim";
				nextToken();
				break;
			case RuleTokenizer.BOTH :
				nextToken();
				break;
		}
		Expression trimStrExp = new StringExp(" ");
		if (_token != RuleTokenizer.FROM) {
			trimStrExp = getAddSubExp();
		}
		expect(RuleTokenizer.FROM);
		Expression strExp = getAddSubExp();
		expect(RuleTokenizer.RIGHT_PAREN);
		List argExps = new ArrayList();
		argExps.add(strExp);
		argExps.add(trimStrExp);
		return new MethodExp(
			"org.seasar.util.StringUtil",
			methodName,
			NazunaUtil.toExpressionArray(argExps));
	}

	private Expression getWordExp() throws SeasarException {
		String s = _tokenizer.getString();
		nextToken();
		if (_token != RuleTokenizer.LEFT_PAREN) {
			return getVariableExp(s);
		} else {
			return getMethodExp(s);
		}
	}

	private Expression getVariableExp(String name) throws SeasarException {
		if (_token == RuleTokenizer.LEFT_BRACKET) {
			nextToken();
			Expression indexExp = getAddSubExp();
			expect(RuleTokenizer.RIGHT_BRACKET);
			if (_token == RuleTokenizer.ASSIGN) {
				nextToken();
				return new SetArrayVariableExp(name, indexExp, getAddSubExp());
			} else {
				return new GetArrayVariableExp(name, indexExp);
			}
		} else {
			if (_token == RuleTokenizer.ASSIGN) {
				nextToken();
				return new SetVariableExp(name, getAddSubExp());
			} else {
				return new GetVariableExp(name);
			}
		}
	}

	private Expression getMethodExp(String s) throws SeasarException {
		nextToken();
		String targetName = null;
		String methodName = null;
		AliasConfig aliasConfig = Nazuna.getNazunaConfig().resolveAlias(s);
		if (aliasConfig != null) {
			targetName = aliasConfig.getClassName();
			methodName = aliasConfig.getMethodName();
		} else {
			int pos = s.lastIndexOf('.');
			targetName = s.substring(0, pos);
			methodName = s.substring(pos + 1);
		}
		List argExps = new ArrayList();
		if (_token != RuleTokenizer.RIGHT_PAREN) {
			argExps.add(getAddSubExp());
			while (_token == RuleTokenizer.COMMA) {
				nextToken();
				argExps.add(getAddSubExp());
			}
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		return new MethodExp(
			targetName,
			methodName,
			NazunaUtil.toExpressionArray(argExps));
	}

	private Expression getNewExp() throws SeasarException {
		nextToken();
		String className = _tokenizer.getString();
		nextToken();
		if (_token == RuleTokenizer.LEFT_PAREN) {
			nextToken();
			List argExpList = new ArrayList();
			if (_token != RuleTokenizer.RIGHT_PAREN) {
				argExpList.add(getAddSubExp());
				while (_token == RuleTokenizer.COMMA) {
					nextToken();
					argExpList.add(getAddSubExp());
				}
			}
			expect(RuleTokenizer.RIGHT_PAREN);
			return new NewExp(
				Reflector.getClass(className),
				NazunaUtil.toExpressionArray(argExpList));
		} else {
			expect(RuleTokenizer.LEFT_BRACKET);
			int arraySize = 0;
			if (_token == RuleTokenizer.INTEGER) {
				arraySize = _tokenizer.getInteger().intValue();
				nextToken();
			}
			expect(RuleTokenizer.RIGHT_BRACKET);
			List argExpList = new ArrayList();
			if (_token == RuleTokenizer.LEFT_BRACE) {
				nextToken();
				argExpList.add(getAddSubExp());
				while (_token == RuleTokenizer.COMMA) {
					nextToken();
					argExpList.add(getAddSubExp());
				}
				expect(RuleTokenizer.RIGHT_BRACE);
				if (arraySize < argExpList.size()) {
					arraySize = argExpList.size();
				}
			}
			return new NewArrayExp(
				Reflector.getClass(className),
				arraySize,
				NazunaUtil.toExpressionArray(argExpList));
		}
	}

	private Expression getExecuteExp(int methodType) throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression pathExp = getAddSubExp();
		EMap parameterExps = new EMap();
		while (_token == RuleTokenizer.COMMA) {
			nextToken();
			String name = _tokenizer.getString();
			nextToken();
			expect(RuleTokenizer.ASSIGN);
			parameterExps.put(name, getAddSubExp());
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		switch (methodType) {
			case RuleTokenizer.EXECUTE :
				return new ExecuteExp(pathExp, parameterExps);
			case RuleTokenizer.EXECUTE_QUERY :
				return new ExecuteQueryExp(pathExp, parameterExps);
			case RuleTokenizer.EXECUTE_RS_QUERY :
				return new ExecuteRSQueryExp(pathExp, parameterExps);
			case RuleTokenizer.EXECUTE_SINGLE_QUERY :
				return new ExecuteSingleQueryExp(pathExp, parameterExps);
			case RuleTokenizer.EXECUTE_UPDATE :
				return new ExecuteUpdateExp(pathExp, parameterExps);
			default :
				throw new SeasarException(
					"ESSR0032",
					new Object[] { RuleTokenizer.getTokenName(_token)});
		}

	}

	private Expression getExecuteRuletExp() throws SeasarException {
		nextToken();
		expect(RuleTokenizer.LEFT_PAREN);
		Expression nameExp = getAddSubExp();
		List argExpList = new EArrayList();
		while (_token == RuleTokenizer.COMMA) {
			nextToken();
			argExpList.add(getAddSubExp());
		}
		expect(RuleTokenizer.RIGHT_PAREN);
		Expression[] argExps =
			(Expression[]) argExpList.toArray(
				new Expression[argExpList.size()]);
		return new ExecuteRuletExp(nameExp, argExps);
	}

	private void nextToken() throws SeasarException {
		_token = _tokenizer.nextToken();
	}

	public void expect(int t) throws SeasarException {
		RuleTokenizer.assertToken(t, _token);
		nextToken();
	}
}
