package org.seasar.nazuna;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.seasar.log.Logger;
import org.seasar.util.Assertion;
import org.seasar.util.LineUtil;
import org.seasar.util.SeasarException;

public final class RuleContextImpl implements RuleContext {

	private static Logger _logger = Logger.getLogger(RuleContextImpl.class);
	private RuleBase _owner;
	private Map _args;
	private Map _vars;
	private Object _returnValue;
	private boolean _returned = false;
	private Throwable _throwable;
	private boolean _throwed = false;
	private String _throwedLocation;
	private String _breakedLabel;
	private String _continuedLabel;
	private boolean _whenBreaked = false;
	private StringBuffer _textBuf = new StringBuffer(255);
	private List _bindVariables = new ArrayList();
	private boolean _whereEnabled = false;

	public RuleContextImpl(RuleBase owner, Map parameters)
		throws SeasarException {
		Assertion.assertNotNull("owner", owner);

		_owner = owner;
		_args = owner.createArgs(parameters);
		_vars = owner.createVars();
	}

	public RuleBase getOwner() {
		return _owner;
	}

	public final Object getValue(String name) throws SeasarException {
		Var var = getVar(name);
		if (var != null) {
			return var.getValue();
		}
		Arg arg = getArg(name);
		if (arg != null) {
			return arg.getValue();
		}
		if (_owner.containsConst(name)) {
			return _owner.getConst(name);
		}
		throw new SeasarException("ESSR0001", new Object[] { name });
	}

	public final Arg getArg(String name) {
		return (Arg) _args.get(name);
	}

	public final Var getVar(String name) {
		return (Var) _vars.get(name);
	}

	public final void setValue(String name, Object value)
		throws SeasarException {

		Var var = getVar(name);
		if (var != null) {
			var.setValue(value);
		} else {
			Arg arg = getArg(name);
			if (arg != null) {
				arg.setValue(value);
			} else if (_owner.containsConst(name)) {
				throw new SeasarException("ESSR0035");
			} else {
				throw new SeasarException("ESSR0001", new Object[] { name });
			}
		}
	}

	public final Object getReturnValue() {
		return _returnValue;
	}

	public final void setReturnValue(Object returnValue)
		throws SeasarException {
		_owner.getOutputType().validate(returnValue);
		_returnValue = returnValue;
		_returned = true;
	}

	public final boolean isReturned() {
		return _returned;
	}

	public final void clearReturned() {
		_returned = false;
	}

	public final void throwSeasarExcepton() throws SeasarException, RuntimeException {
		SeasarException ex = new SeasarException(
			"ESSR0347",
			new Object[] { _owner.getName() + ":" + _throwedLocation, _throwable },
			_throwable);
		_logger.log(ex);
		if (_throwable instanceof RuntimeException) {
			throw (RuntimeException) _throwable;
		}
		if (_throwable instanceof Error) {
			throw (Error) _throwable;
		}
		throw SeasarException.convertSeasarException(_throwable);
	}

	public final void setThrowable(Throwable throwable, String location) {
		_throwable = throwable;
		_throwedLocation = location;
		_throwed = true;
	}

	public final boolean isThrowed() {
		return _throwed;
	}

	public final void clearThrowed() {
		_throwed = false;
	}

	public final boolean isBreaked() {
		return _breakedLabel != null;
	}

	public final boolean isBreaked(String label) {
		return label.equals(_breakedLabel);
	}

	public final void setBreak(String forLabel) throws SeasarException {
		_breakedLabel = forLabel;
	}

	public final boolean isContinued() {
		return _continuedLabel != null;
	}

	public final boolean isContinued(String label) {
		return label.equals(_continuedLabel);
	}

	public final void clearForState() {
		_breakedLabel = null;
		_continuedLabel = null;
	}

	public final void setContinue(String forLabel) throws SeasarException {
		_continuedLabel = forLabel;
	}

	public final boolean isWhenBreaked() {
		return _whenBreaked;
	}

	public final void clearWhenBreaked() {
		_whenBreaked = false;
	}

	public final void breakWhen() {
		_whenBreaked = true;
	}

	public String getText() {
		return _textBuf.toString();
	}

	public void addText(String text) {
		_textBuf.append(text);
	}

	public void addLineSeparaterText() {
		_textBuf.append(LineUtil.LINE_SP);
	}

	public boolean isEmptyText() {
		return _textBuf.length() == 0;
	}

	public void addBindVariable(Object bindVariable) {
		_bindVariables.add(bindVariable);
	}

	public void addBindVariables(List bindVariables) {
		_bindVariables.addAll(bindVariables);
	}

	public List getBindVariables() {
		return _bindVariables;
	}

	public boolean isWhereEnabled() {
		return _whereEnabled;
	}

	public void enableWhere() {
		_whereEnabled = true;
	}
}
