package org.seasar.nazuna;

import java.io.File;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.List;

import org.seasar.util.ArrayUtil;
import org.seasar.util.EArrayList;
import org.seasar.util.JarZipFilter;
import org.seasar.util.Reflector;
import org.seasar.util.ResourceUtil;
import org.seasar.util.SeasarException;
import org.seasar.util.SeasarRuntimeException;

public final class RuletCompiler {

	private static File _baseDir;
	private static File _classesDir;
	private static File _srcDir;
	private static File _libDir;
	private static String _classpath;
	private static String _encoding = "MS932";
	private static Object _compiler;
	private static Method _compileMethod;
	private static String[] _compileArgs;

	static {
		setup();
	}

	private RuletCompiler() {
	}

	public static final File getBaseDir() {
		return _baseDir;
	}

	public static final File getClassesDir() {
		return _classesDir;
	}

	public static final File getSrcDir() {
		return _srcDir;
	}

	public static final void setSrcDir(File srcDir) {
		_srcDir = srcDir;
	}

	public static final File getLibDir() {
		return _libDir;
	}

	public static final void setLibDir(File libDir) {
		_libDir = libDir;
	}

	public static final File getSrcFile(String className) {
		String path = className.replace('.', File.separatorChar);
		return new File(_srcDir, path + ".java");
	}

	public static final File getClassFile(String className) {
		String path = className.replace('.', File.separatorChar);
		return new File(_classesDir, path + ".class");
	}

	public static final String getEncoding() {
		return _encoding;
	}

	public static final void setEncoding(String encoding) {
		_encoding = encoding;
	}

	public static final void setupCompileArgs() {
		List argList = new EArrayList();
		argList.add("-nowarn");
		argList.add("-d");
		argList.add(_classesDir.getAbsolutePath());
		argList.add("-classpath");
		argList.add(_classpath);
		argList.add("-sourcepath");
		argList.add(_srcDir.getAbsolutePath());
		argList.add("-encoding");
		argList.add(_encoding);
		argList.add("-O");
		_compileArgs =
			(String[]) argList.toArray(new String[argList.size() + 1]);
	}

	public static void compile(String className) throws SeasarException {
		compile(getSrcFile(className), getClassFile(className));
	}

	public static void compile(File srcFile, File classFile)
		throws SeasarException {

		_compileArgs[_compileArgs.length - 1] = srcFile.getAbsolutePath();
		Integer ret =
			(Integer) Reflector.invoke(
				_compileMethod,
				_compiler,
				new Object[] { _compileArgs });
		if (ret.intValue() != 0) {
			throw new SeasarException(
				"ESSR0095",
				new Object[] { srcFile.getAbsolutePath()});
		}
		classFile.setLastModified(srcFile.lastModified());
	}

	private static final void setup() {
		setupConfig();
		setupCompiler();
		setupCompileArgs();
	}

	private static void setupBaseDir() {
		URL url = ResourceUtil.getResource("org/seasar/nazuna/Nazuna.class");
		String s = ResourceUtil.toExternalForm(url);
		int pos = s.lastIndexOf('!');
		if (pos > 0) {
			String s2 = s.substring(9, pos);
			File f = new File(s2);
			_baseDir = f.getParentFile().getParentFile();
			_classesDir = new File(_baseDir, "classes");
		} else {
			_classesDir =
				new File(ResourceUtil.getFileName(url))
					.getParentFile()
					.getParentFile()
					.getParentFile()
					.getParentFile();
			_baseDir = _classesDir.getParentFile();
		}
	}

	private static final void setupConfig() {
		setupBaseDir();
		_srcDir = new File(_baseDir, "src");
		_libDir = new File(_baseDir, "lib");
		StringBuffer buf = new StringBuffer();
		buf.append(_classesDir.getAbsolutePath());
		buf.append(File.pathSeparatorChar);
		File[] jars = JarZipFilter.listFiles(_libDir);
		for (int i = 0; i < jars.length; ++i) {
			buf.append(jars[i].getAbsolutePath());
			buf.append(File.pathSeparatorChar);
		}
		_classpath = buf.toString();
	}

	private static final void setupCompiler() {
		try {
			ClassLoader cl =
				new URLClassLoader(
					new URL[] { NazunaUtil.getToolsJarURL()},
					null);
			Class c = cl.loadClass("com.sun.tools.javac.Main");
			_compiler = Reflector.newInstance(c);
			_compileMethod =
				Reflector.getMethod(
					c,
					"compile",
					new Class[] { ArrayUtil.EMPTY_STRINGS.getClass()});
		} catch (ClassNotFoundException ex) {
			throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
		}
	}
}
