package org.seasar.naming;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Iterator;
import java.util.Map;

import javax.naming.InvalidNameException;
import javax.naming.Name;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;
import javax.naming.NotContextException;

import org.seasar.message.MessageFormatter;
import org.seasar.util.EMap;

public final class NamingServer implements NamingServerMBean, Externalizable {

    final static long serialVersionUID = -489094459554022028L;
    private transient Map _nameMap = new EMap();
    private transient Name _prefix;
    private transient NamingContext _namingContext;

    public NamingServer() throws NamingException {
        this(null);
    }

    public NamingServer(final Name prefix) throws NamingException {
        if (prefix == null) {
            _prefix = NamingParser.getInstance().parse("");
        } else {
            _prefix = prefix;
        }
        _namingContext = new NamingContext(null, _prefix, this);
    }
    
    public NamingContext getNamingContext() {
    	return _namingContext;
    }

    public void bind(final Name name, final Object obj) throws NamingException {
        checkName(name);
        if (name.size() > 1) {
            bindByChild(name, obj);
        } else {
            bindBySelf(name, obj);
        }
    }

    public void rebind(final Name name, final Object obj) throws NamingException {
        checkName(name);
        if (name.size() > 1) {
            rebindByChild(name, obj);
        } else {
            add(name, obj);
        }
    }

    public void unbind(final Name name) throws NamingException {
        checkName(name);
        if (name.size() > 1) {
            unbindByChild(name);
        } else {
            unbindBySelf(name);
        }
    }

    public Object lookup(final Name name) throws NamingException {
        if (name.isEmpty() || name.get(0).equals("")) {
            return _namingContext;
        } else if (name.size() > 1) {
            return lookupByChild(name);
        } else {
            return resolveIfChild(get(name));
        }
    }

    public Map getNameMap(final Name name) throws NamingException {
        if (name.isEmpty() || name.get(0).equals("")) {
            return getNameMapBySelf(name);
        } else {
            return getNameMapByChild(name);
        }
    }

    public NamingContext createSubcontext(final Name name) throws NamingException {
        checkName(name);
        if (name.size() > 1) {
            return createSubcontextByChild(name);
        } else {
            return createSubcontextBySelf(name);
        }
    }

    public void destroySubcontext(final Name name) throws NamingException {
        if (name.isEmpty() || name.get(0).equals("")) {
            destroy();
        } else if (name.size() > 1) {
            destroySubcontextByChild(name);
        } else {
            destroySubcontextBySelf(name);
        }
    }

    public void writeExternal(ObjectOutput s) throws IOException {
        s.writeObject(_nameMap);
        s.writeObject(_prefix);
        s.writeObject(_namingContext);
    }

    public void readExternal(ObjectInput s)
             throws IOException, ClassNotFoundException {

        _nameMap = (Map) s.readObject();
        _prefix = (Name) s.readObject();
        _namingContext = (NamingContext) s.readObject();
    }

    void destroy() {
        for (Iterator i = _nameMap.values().iterator(); i.hasNext(); ) {
            Object child = i.next();
            if (child instanceof NamingServer) {
                ((NamingServer) child).destroy();
            }
        }
        _nameMap.clear();
    }

    private synchronized void add(final Name name, final Object obj)
             throws NamingException {

        _nameMap.put(name.get(0), obj);
    }

    private Object get(final Name name) throws NamingException {
        Object obj = _nameMap.get(name.get(0));
        if (obj == null) {
            throw new NameNotFoundException(_namingContext.getAbsoluteName(name).toString());
        }
        return obj;
    }

    private boolean contains(final Name name) {
        return _nameMap.containsKey(name.get(0));
    }

    private synchronized void remove(final Name name) throws NamingException {
        if (_nameMap.remove(name.get(0)) == null) {
            throw new NameNotFoundException(
                    MessageFormatter.getMessage("ESSR0001", new Object[]{
                    _namingContext.getAbsoluteName(name).toString()}));
        }
    }

    private Object resolveIfChild(final Object obj) {
        if (obj instanceof NamingServer) {
            return ((NamingServer) obj).getNamingContext();
        } else {
            return obj;
        }
    }

    private void bindByChild(final Name name, final Object obj) throws NamingException {
        if (!contains(name)) {
            createSubcontextBySelf(name.getPrefix(1));
        }
        Object child = get(name);
        if (child instanceof NamingServer) {
            ((NamingServer) child).bind(name.getSuffix(1), obj);
        } else {
            throw new NotContextException();
        }
    }

    private void bindBySelf(final Name name, final Object obj) throws NamingException {
        if (contains(name)) {
            throw new NameAlreadyBoundException(_namingContext.getAbsoluteName(name).toString());
        } else {
            add(name, obj);
        }
    }

    private void rebindByChild(final Name name, final Object obj) throws NamingException {
        if (!contains(name)) {
            createSubcontextBySelf(name.getPrefix(1));
        }
        Object child = get(name);
        if (child instanceof NamingServer) {
            ((NamingServer) child).rebind(name.getSuffix(1), obj);
        } else {
            throw new NotContextException();
        }
    }

    private void unbindByChild(final Name name) throws NamingException {
        Object child = get(name);
        if (child instanceof NamingServer) {
            ((NamingServer) child).unbind(name.getSuffix(1));
        } else {
            throw new NotContextException();
        }
    }

    private void unbindBySelf(final Name name) throws NamingException {
        Object child = get(name);
        if (child instanceof NamingServer) {
            throw new NamingException(MessageFormatter.getMessage("ESSR0334", new Object[]
                    {_namingContext.getAbsoluteName(name).toString()}));
        } else {
            remove(name);
        }
    }

    private Object lookupByChild(final Name name) throws NamingException {
        Object child = get(name);
        if (child instanceof NamingServer) {
            return ((NamingServer) child).lookup(name.getSuffix(1));
        } else {
            throw new NotContextException();
        }
    }

    private Map getNameMapByChild(final Name name) throws NamingException {
        Object child = get(name);
        if (child instanceof NamingServer) {
            return ((NamingServer) child).getNameMap(name.getSuffix(1));
        } else {
            throw new NotContextException();
        }
    }

    private Map getNameMapBySelf(final Name name) throws NamingException {
        Map nm = new EMap();
        for (Iterator i = _nameMap.entrySet().iterator(); i.hasNext(); ) {
            Map.Entry e = (Map.Entry) i.next();
            Object obj = resolveIfChild(e.getValue());
            nm.put(e.getKey(), obj);
        }
        return nm;
    }

    private NamingContext createSubcontextByChild(final Name name) throws NamingException {
        if (!contains(name)) {
            createSubcontextBySelf(name.getPrefix(1));
        }
        Object child = get(name);
        if (child instanceof NamingServer) {
            return ((NamingServer) child).createSubcontext(name.getSuffix(1));
        } else {
            throw new NotContextException();
        }
    }

    private NamingContext createSubcontextBySelf(final Name name) throws NamingException {
        Name childName = (Name) (_prefix.clone());
        childName.addAll(name);
        NamingServer child = new NamingServer(childName);
        add(name, child);
        return child.getNamingContext();
    }

    private void destroySubcontextByChild(final Name name) throws NamingException {
        Object child = get(name);
        if (child instanceof NamingServer) {
            ((NamingServer) child).destroySubcontext(name.getSuffix(1));
        } else {
            throw new NotContextException();
        }
    }

    private void destroySubcontextBySelf(final Name name) throws NamingException {
        Object child = get(name);
        if (child instanceof NamingServer) {
            ((NamingServer) child).destroy();
            remove(name);
        } else {
            throw new NotContextException();
        }
    }

    private void checkName(final Name name) throws NamingException {
        if (name.isEmpty() || name.get(0).equals("")) {
            throw new InvalidNameException(_namingContext.getAbsoluteName(name).toString());
        }
    }
}
