package org.seasar.eclipse;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;


public final class FileUtil {

	private static final int BUF_SIZE = 8192;
	
	private FileUtil() {
	}
	
	public static void copyFile(File fromFile, File toFile) throws IOException {
		BufferedInputStream in = new BufferedInputStream(new FileInputStream(fromFile));
		BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(toFile));
		try {
			byte[] buf = new byte[BUF_SIZE];
			int n;
			while ((n = in.read(buf)) >= 0) {
				out.write(buf,0,n);
			}
		} finally {
			in.close();
			out.close();
		}
	}
	
	public static String readText(File file) throws IOException {
		BufferedReader in = new BufferedReader(new FileReader(file));
		StringBuffer buf = new StringBuffer(BUF_SIZE);
		try {
			String line = null;
			while ((line = in.readLine()) != null) {
				buf.append(line);
				buf.append('\n');
			}
		} finally {
			in.close();
		}
		return buf.toString();
	}
    
    public static String readText(Reader reader) throws IOException {
        BufferedReader in = new BufferedReader(reader);
        StringBuffer buf = new StringBuffer(BUF_SIZE);
        try {
            String line = null;
            while ((line = in.readLine()) != null) {
                buf.append(line);
                buf.append('\n');
            }
        } finally {
            in.close();
        }
        int lastPos = buf.length() - 1;
        if (lastPos >= 0 && buf.charAt(lastPos) == '\n') {
            buf.setLength(lastPos);
        }
        return buf.toString();
    }

	public static void writeText(File file, String text) throws IOException {
		BufferedWriter out = new BufferedWriter(new FileWriter(file));
		out.write(text);
		out.close();
	}

    public static String convertAsciiToNative(InputStream is) throws IOException {
        String text = readText(new InputStreamReader(is, "8859_1"));
        StringBuffer buf = new StringBuffer(BUF_SIZE);
        convertAsciiToNative(text, buf);
        return buf.toString();
    }
    
    public static void convertAsciiToNative(String asciiStr, StringBuffer buf) {
        for (int i = 0; i < asciiStr.length(); ++i){
            if (asciiStr.charAt(i) != '\\'){
                buf.append(asciiStr.charAt(i));
            } else {
                ++i;
                if (asciiStr.charAt(i) != 'u') {
                    buf.append('\\').append(asciiStr.charAt(i));
                } else {
                    ++i;
                    String escStr = asciiStr.substring(i, i + 4);
                    buf.append((char) Integer.parseInt(escStr, 16));
                    i += 3;
                }
            }
        }
    }
    
    public static String convertNativeToAscii(String nativeStr) throws IOException {
        StringBuffer buf = new StringBuffer(nativeStr.length());
        for (int i = 0; i < nativeStr.length(); i++) {
            char ch = nativeStr.charAt(i);
            if (ch <= 0x7F) {
                buf.append(ch);
            } else {
                String escStr = "0000" + Integer.toHexString(ch & 0xFFFF);
                buf.append("\\u").append(escStr.substring(escStr.length() - 4));
            }
        }
        return buf.toString();
    }
}
