package test.org.seasar.nazuna;

import java.util.HashMap;
import java.util.Map;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.seasar.nazuna.ArgType;
import org.seasar.nazuna.Flowlet;
import org.seasar.nazuna.FlowletBuilder;
import org.seasar.nazuna.OutputType;
import org.seasar.nazuna.VarType;
import org.seasar.util.SMap;
import org.seasar.util.SeasarException;

public class NazunaXMLHandlerRuleTest extends TestCase {

    public NazunaXMLHandlerRuleTest(String name) {
        super(name);
    }
    
    public void testParse2() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Hello2.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	Map parameters = new HashMap();
    	parameters.put("name", "Seasar");
    	rule.execute(parameters);
    }
    
    public void testParse3() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Hello3.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", "Hello", rule.execute());
    }
    
    public void testReturn() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Return.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", "Hello", rule.execute());
    }
    
    public void testReturn2() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Return2.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", null, rule.execute());
    }
    
    public void testVariable() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Variable.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", "test", rule.execute());
    }
    
    public void testFor() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/For.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", new Integer(15) , rule.execute());
    }
    
    public void testFor2() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/For2.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", new Integer(30) , rule.execute());
    }
    
    public void testForBreak() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/ForBreak.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", new Integer(6) , rule.execute());
    }
    
    public void testNestedForBreak() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/NestedForBreak.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", new Integer(4) , rule.execute());
    }
    
    public void testForContinue() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/ForContinue.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", new Integer(9) , rule.execute());
    }
    
    public void testNestedForContinue() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/NestedForContinue.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", new Integer(5) , rule.execute());
    }
    
    public void testCase() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Case.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	Map parameters = new HashMap();
    	parameters.put("aaa", new Integer(1));
    	assertEquals("1", new Integer(1) , rule.execute(parameters));
    	parameters.put("aaa", new Integer(2));
    	assertEquals("2", new Integer(2) , rule.execute(parameters));
    	parameters.put("aaa", new Integer(0));
    	assertEquals("3", new Integer(3) , rule.execute(parameters));
    }
    
    public void testThrow() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Throw.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	try {
    		rule.execute();
    		fail("1");
    	} catch (SeasarException ex) {
    		System.out.println(ex);
    	}
    }
    
    public void testThrow2() throws Exception {
    	Flowlet rule = FlowletBuilder.parse("/test/org/seasar/nazuna/Throw2.xml");
    	try {
    		rule.execute();
    		fail("1");
    	} catch (SeasarException ex) {
    		System.out.println(ex);
    		//SeasarException cause = (SeasarException) ex.getCause();
    		assertEquals("2", "ESSR0028", ex.getMessageCode());
    	}
    }
    
    public void testConst() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Const.xml";
    	Flowlet flowlet = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", "Hello", flowlet.execute());
    }

    
    public void testInclude2() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Include2.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	assertEquals("1", new Integer(1), rule.execute());
    }
    
    public void testAssert() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Assert.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	Map parameters = new SMap();
    	parameters.put("aaa", "111");
    	assertEquals("1", "a", rule.execute(parameters));
    	try {
    		rule.execute();
    		fail("2");
    	} catch (SeasarException ex) {
    		System.out.println(ex);
    	}
    }
    
    public void testAssert2() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Assert2.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	try {
    		rule.execute();
    		fail("1");
    	} catch (SeasarException ex) {
    		System.out.println(ex);
    	}
    }
    
    public void testInputArg() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Input.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	ArgType argType = rule.getInputType().getArgType("a");
    	assertNotNull("1", argType);
    	assertEquals("2", String.class, argType.getArgClass());
    	ArgType argType2 = rule.getInputType().getArgType("b");
    	assertNotNull("3", argType2);
    	assertEquals("4", Integer.class, argType2.getArgClass());
    }
    
    public void testLocalVar() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Local.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	VarType varType = rule.getLocalType().getVarType("a");
    	assertNotNull("1", varType);
    	assertEquals("2", String.class, varType.getVarClass());
    	assertEquals("3", "aaa", varType.getDefaultValue());
    	VarType varType2 = rule.getLocalType().getVarType("b");
    	assertNotNull("4", varType2);
    	assertEquals("5", Integer.class, varType2.getVarClass());
    }
    
    public void testOutput() throws Exception {
    	String xmlFileName = "/test/org/seasar/nazuna/Return.xml";
    	Flowlet rule = FlowletBuilder.parse(xmlFileName);
    	OutputType outputType = rule.getOutputType();
    	assertNotNull("1", outputType);
    	assertEquals("2", String.class, outputType.getOutputClass());
    }
    
    protected void setUp() throws Exception {
    }

    protected void tearDown() throws Exception {
    }

    public static Test suite ( ) {
        return new TestSuite(NazunaXMLHandlerRuleTest.class);
    }

    public static void main (String[] args) {
        junit.textui.TestRunner.main(new String[]{NazunaXMLHandlerRuleTest.class.getName()});
    }
}