package org.seasar.eclipse.wizards;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.jdt.internal.ui.util.ExceptionHandler;
import org.eclipse.jdt.internal.ui.wizards.NewElementWizard;
import org.eclipse.jdt.internal.ui.wizards.NewWizardMessages;
import org.eclipse.jdt.ui.wizards.NewJavaProjectWizardPage;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.ui.actions.WorkspaceModifyDelegatingOperation;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;
import org.seasar.eclipse.SeasarPlugin;
import org.seasar.eclipse.SeasarProject;

public final class SeasarProjectCreationWizard extends NewElementWizard implements IExecutableExtension {

	private static final String TITLE = SeasarPlugin.getResourceString("create.title");
	private static final String DESCRIPTION = SeasarPlugin.getResourceString("create.description");
    private static final String FAIL_MESSAGE = SeasarPlugin.getResourceString("create.fail.message");
	private NewSeasarProjectWizardPage _seasarPage;
	private NewJavaProjectWizardPage _javaPage;
	private WizardNewProjectCreationPage _firstPage;
	private IConfigurationElement _configuratonElement;

	public SeasarProjectCreationWizard() {
		super();
		ImageDescriptor logo = getLogo();
		if (logo != null) {
			setDefaultPageImageDescriptor(logo); 		
		}
		setDialogSettings(JavaPlugin.getDefault().getDialogSettings());
		setWindowTitle(TITLE);
	}

	public void addPages() {
		super.addPages(); 
		addFirstPage();
		addSeasarPage();
		createJavaPage();
	}		
	
	public boolean performFinish() {
		try {
			IRunnableWithProgress op = new WorkspaceModifyDelegatingOperation(
				_javaPage.getRunnable());
			getContainer().run(false, true, op);
			if (!_seasarPage.isDisplayedOnce()) {
				_seasarPage.setContextPath("/" + _firstPage.getProjectName());	
			}
			IJavaProject javaProject = _javaPage.getNewJavaProject();
			SeasarProject.addSeasarNature(javaProject);
			SeasarProject seasarProject = SeasarProject.getSeasarProject(javaProject);
			seasarProject.setContextPath(_seasarPage.getContextPath());
			seasarProject.storeProperties();
			seasarProject.configureAll();
			BasicNewProjectResourceWizard.updatePerspective(_configuratonElement);
			selectAndReveal(seasarProject.getProject());
			return true;
		} catch (InvocationTargetException e) {
			String title= NewWizardMessages.getString("NewProjectCreationWizard.op_error.title");
			String message= NewWizardMessages.getString("NewProjectCreationWizard.op_error.message");
			ExceptionHandler.handle(e, getShell(), title, message);
			return false;
		} catch  (InterruptedException e) {
			return false;
		} catch (CoreException e) {
            SeasarPlugin.handleException(e, TITLE, FAIL_MESSAGE);
			return false;	
		} catch (IOException e) {
            SeasarPlugin.handleException(e, TITLE, FAIL_MESSAGE);
			return false;	
		}
	}
	
	public void setInitializationData(IConfigurationElement configurationElement,
			String propertyName, Object data) {
				
		_configuratonElement = configurationElement;
	}
	
	public IWizardPage getNextPage(IWizardPage page) {
		if (page instanceof WizardNewProjectCreationPage) {
			if (!_seasarPage.isDisplayedOnce()) {
				_seasarPage.setContextPath("/" + _firstPage.getProjectName());	
			}			
		}
		return super.getNextPage(page);
	}
	
	private ImageDescriptor getLogo() {
		try {
			URL prefix = new URL(SeasarPlugin.getDefault().getDescriptor().getInstallURL(), "icons/");
			return ImageDescriptor.createFromURL(new URL(prefix, "newproject_wiz.gif"));
		} catch (MalformedURLException e) {
			return null;
		}
	}
	
	private void addFirstPage() {
		_firstPage = new WizardNewProjectCreationPage("Page 1"); 
		_firstPage.setTitle(TITLE); 
		_firstPage.setDescription(DESCRIPTION);
		_firstPage.setMessage("hoge");
		addPage(_firstPage);
	}
		
	private void addSeasarPage() {
		_seasarPage = new NewSeasarProjectWizardPage("Page 2");
		_seasarPage.setTitle(TITLE); 
		_seasarPage.setDescription(DESCRIPTION);
		addPage(_seasarPage);
	}
	
	private void createJavaPage() {
		IWorkspaceRoot root= JavaPlugin.getWorkspace().getRoot();
		_javaPage= new NewJavaProjectWizardPage(root, _firstPage);
	}	

}