package test.org.seasar.util;

import junit.framework.*;
import java.util.*;
import org.seasar.util.*;

public class SMapTest extends TestCase {

    private SMap _map;

    public SMapTest(String name) {
        super(name);
    }

    public void testSize() throws Exception {
        assertEquals("1", 3, _map.size());
        _map.put("3", "test3");
        assertEquals("2", 4, _map.size());
    }

    public void testIsEmpty() throws Exception {
        assertTrue("1", !_map.isEmpty());
        _map.clear();
        assertTrue("2", _map.isEmpty());
    }

    public void testContainsValue() throws Exception {
        assertTrue("1", _map.containsValue("test2"));
        assertTrue("2", !_map.containsValue("test3"));
    }

    public void testContainsKey() throws Exception {
        assertTrue("1", _map.containsKey("2"));
        assertTrue("2", !_map.containsKey("3"));
        _map.put("3", null);
        assertTrue("3", _map.containsKey("3"));
    }

    public void testGet() throws Exception {
        assertEquals("1", "test", _map.get("1"));
        assertEquals("2", null, _map.get(null));
        assertEquals("3", null, _map.get("test3"));
    }

    public void testPut() throws Exception {
        assertEquals("1", "test", _map.put("1", "test3"));
        assertEquals("2", "test3", _map.get("1"));
        _map.put(null, "test4");
        _map.put(null, "test5");
    }

    public void testRemove() throws Exception {
        assertEquals("1", "test", _map.remove("1"));
        assertEquals("2", 2, _map.size());
        assertEquals("3", null, _map.remove("dummy"));
    }

    public void testPutAll() throws Exception {
        Map m = new HashMap();
        m.put("3", "test3");
        m.put("4", "test4");
        _map.putAll(m);
        assertEquals("1", "test3", _map.get("3"));
        assertEquals("2", "test4", _map.get("4"));
        assertEquals("3", 5, _map.size());
    }

    public void testEqaulas() throws Exception {
        SMap copy = (SMap) _map.clone();
        assertTrue("1", _map.equals(copy));
        assertTrue("2", !_map.equals(null));
        _map.put("3", "test3");
        assertTrue("3", !_map.equals(copy));
    }

    public void testToString() throws Exception {
        assertNotNull("1", _map.toString());
    }

    public void testClear() throws Exception {
        _map.clear();
        assertEquals("1", 0, _map.size());
    }

    public void testEntrySet() throws Exception {
        Iterator i = _map.entrySet().iterator();
        assertEquals("1", "2", ((Map.Entry) i.next()).getKey());
        assertEquals("2", "1", ((Map.Entry) i.next()).getKey());
        assertEquals("3", null, ((Map.Entry) i.next()).getKey());
        assertEquals("4", false, i.hasNext());
    }

    public void testSerialize() throws Exception {
        SMap copy = (SMap) Serializer.serialize(_map);
        assertEquals("1", null, copy.get(null));
        assertEquals("2", "test", copy.get("1"));
        assertEquals("3", "test2", copy.get("2"));
        _map.equals(copy);
    }


    public void testPerformance() throws Exception {
        int num = 100000;
        Map hmap = new HashMap();
        Map emap = new EMap();
        Map smap = new SMap();

        long start = System.currentTimeMillis();
        for (int i = 0; i < num; i++) {
            hmap.put(String.valueOf(i), null);
        }
        System.out.println("HashMap.put:" + (System.currentTimeMillis() - start));

		start = System.currentTimeMillis();
        for (int i = 0; i < num; i++) {
            emap.put(String.valueOf(i), null);
        }
        System.out.println("EMap.put:" + (System.currentTimeMillis() - start));
        
        start = System.currentTimeMillis();
        for (int i = 0; i < num; i++) {
            smap.put(String.valueOf(i), null);
        }
        System.out.println("SMap.put:" + (System.currentTimeMillis() - start));

        start = System.currentTimeMillis();
        for (int i = 0; i < num; i++) {
            hmap.get(String.valueOf(i));
        }
        System.out.println("HashMap.get:" + (System.currentTimeMillis() - start));
        
        start = System.currentTimeMillis();
        for (int i = 0; i < num; i++) {
            emap.get(String.valueOf(i));
        }
        System.out.println("EMap.get:" + (System.currentTimeMillis() - start));

        start = System.currentTimeMillis();
        for (int i = 0; i < num; i++) {
            smap.get(String.valueOf(i));
        }
        System.out.println("SMap.get:" + (System.currentTimeMillis() - start));

        start = System.currentTimeMillis();
        for (Iterator i = hmap.entrySet().iterator(); i.hasNext(); ) {
            i.next();
        }
        System.out.println("HashMap iteration:" + (System.currentTimeMillis() - start));
        
        start = System.currentTimeMillis();
        for (Iterator i = emap.entrySet().iterator(); i.hasNext(); ) {
            i.next();
        }
        System.out.println("EMap iteration:" + (System.currentTimeMillis() - start));

        start = System.currentTimeMillis();
        for (Iterator i = smap.entrySet().iterator(); i.hasNext(); ) {
            i.next();
        }
        System.out.println("SMap iteration:" + (System.currentTimeMillis() - start));

        start = System.currentTimeMillis();
        Serializer.serialize(hmap);
        System.out.println("HashMap serialize:" + (System.currentTimeMillis() - start));
        
        start = System.currentTimeMillis();
        Serializer.serialize(emap);
        System.out.println("EMap serialize:" + (System.currentTimeMillis() - start));

        start = System.currentTimeMillis();
        Serializer.serialize(smap);
        System.out.println("SMap serialize:" + (System.currentTimeMillis() - start));
        
        start = System.currentTimeMillis();
        for (int i = 0; i < num; i++) {
            hmap.remove(String.valueOf(i));
        }
        System.out.println("HashMap.remove:" + (System.currentTimeMillis() - start));

        start = System.currentTimeMillis();
        for (int i = 0; i < num; i++) {
            smap.remove(String.valueOf(i));
        }
        System.out.println("SMap.remove:" + (System.currentTimeMillis() - start));

    }


    /**
     *  The JUnit setup method
     *
     *@exception  Exception  Description of the Exception
     */
    protected void setUp() throws Exception {
        _map = new SMap();
        _map.put(null, null);
        _map.put("1", "test");
        _map.put("2", "test2");
    }


    /**
     *  The teardown method for JUnit
     *
     *@exception  Exception  Description of the Exception
     */
    protected void tearDown() throws Exception {
        _map = null;
    }


    /**
     *  A unit test suite for JUnit
     *
     *@return    The test suite
     */
    public static Test suite() {
        return new TestSuite(SMapTest.class);
    }


    /**
     *  The main program for the SMapTest class
     *
     *@param  args  The command line arguments
     */
    public static void main(String[] args) {
        junit.textui.TestRunner.main(new String[]{"test.jp.co.isid.ark.bomf.util.SMapTest"});
    }
}
