package test.org.seasar.util;

import junit.framework.*;
import java.util.*;
import java.lang.reflect.*;
import org.seasar.util.*;

public class ReflectorTest extends TestCase {

    private final static String REFLECTOR_NAME = "org.seasar.util.Reflector";

    public ReflectorTest(String name) {
        super(name);
    }

    public void testGetClass() throws Exception {
        assertEquals("1", Reflector.class, Reflector.getClass(REFLECTOR_NAME));
    }

    public void testNewInstance() throws Exception {
        assertEquals("1", String.class, Reflector.newInstance("java.lang.String").getClass());
        assertEquals("2", String.class, Reflector.newInstance(String.class).getClass());
    }

    public void testGetMethod() throws Exception {
        Class clazz = Reflector.getClass(REFLECTOR_NAME);
        Class[] parameterTypes = new Class[]{Class.class, String.class, Class[].class};
        assertNotNull("1", Reflector.getMethod(clazz, "getMethod", parameterTypes));
        assertNotNull("2", Reflector.getMethod(REFLECTOR_NAME, "getMethod", parameterTypes));
    }

    public void testCreate() throws Exception {
        Reflector reflector = Reflector.create("java.util.HashMap");
        assertEquals("1", HashMap.class, reflector.getTargetClass());
        assertEquals("2", HashMap.class, reflector.getTarget().getClass());

        Integer i = new Integer(10);
        Reflector reflector2 = Reflector.create("java.lang.Integer", new Object[]{i});
        assertEquals("3", i, reflector2.getTarget());

        Reflector reflector3 = Reflector.create("java.lang.Runtime", "getRuntime");
        assertEquals("4", Runtime.class, reflector3.getTarget().getClass());
        assertNotNull("5", reflector3.invoke("freeMemory", null));
    }

    public void testGetConstructor() throws Exception {
        Integer i = new Integer(10);
        Object[] args = new Object[]{i};
        Constructor constructor = Reflector.getConstructor("java.lang.Integer", args);
        assertEquals("1", i, constructor.newInstance(args));

        HashMap map = new HashMap();
        map.put("1", "2");
        Object[] args2 = new Object[]{map};
        Constructor constructor2 = Reflector.getConstructor("java.util.HashMap", args2);
        Map m = (Map)constructor2.newInstance(args2);
        assertEquals("2", 1, m.size());
    }


    public void testInvoke() throws Exception {
        Reflector reflector = Reflector.create("java.util.HashMap");
        assertEquals("1", new Integer(0), reflector.invoke("size"));
        reflector.invoke("putAll", new Object[]{new HashMap()});
    }
    
    public void testInvoke2() throws Exception {
        MyClass2 me = new MyClass2();
        Method m = Reflector.getMethod(MyClass2.class, "throwError", null);
        Method m2 = Reflector.getMethod(MyClass2.class, "throwException", null);
        try {
        	Reflector.invoke(m, me, null);
        	fail("1");
        } catch (Error error) {
        	System.out.println(error);
        }
        try {
        	Reflector.invoke(m2, me, null);
        	fail("2");
        } catch (SeasarException ex) {
        	System.out.println(ex);
        }
    }

    public void testSetProperties() throws Exception {
        MyClass myClass = new MyClass();
        Properties props = new Properties();
        props.setProperty("myChar", String.valueOf('a'));
        props.setProperty("myShort", "98");
        props.setProperty("myInt", "99");
        props.setProperty("myLong", "100");
        props.setProperty("myFloat", "101");
        props.setProperty("myDouble", "102");
        props.setProperty("myBoolean", "true");
        props.setProperty("myString", "hello");
        Reflector.setProperties(myClass, props);
        assertEquals("1", 'a', myClass._myChar);
        assertEquals("2", 98, myClass._myShort);
        assertEquals("3", 99, myClass._myInt);
        assertEquals("4", 100, myClass._myLong);
        assertEquals("5", Float.toString(101f), Float.toString(myClass._myFloat));
        assertEquals("6", Double.toString(102d), Double.toString(myClass._myDouble));
        assertEquals("7", true, myClass._myBoolean);
        assertEquals("8", "hello", myClass._myString);
    }
    
    public void testSetProperties2() throws Exception {
        MyClass myClass = new MyClass();
        Map props = new HashMap();
        props.put("myChar", String.valueOf('a'));
        props.put("myShort", "98");
        props.put("myInt", "99");
        props.put("myLong", "100");
        props.put("myFloat", "101");
        props.put("myDouble", "102");
        props.put("myBoolean", "true");
        props.put("myString", "hello");
        Reflector.setProperties(myClass, props);
        assertEquals("1", 'a', myClass._myChar);
        assertEquals("2", 98, myClass._myShort);
        assertEquals("3", 99, myClass._myInt);
        assertEquals("4", 100, myClass._myLong);
        assertEquals("5", Float.toString(101f), Float.toString(myClass._myFloat));
        assertEquals("6", Double.toString(102d), Double.toString(myClass._myDouble));
        assertEquals("7", true, myClass._myBoolean);
        assertEquals("8", "hello", myClass._myString);
    }

    public void testClone() throws Exception {
        String[] a = new String[]{"1", "2"};
        String[] b = (String[]) Reflector.clone(a);
        assertEquals("1", a.length, b.length);
        assertEquals("2", "1", b[0]);
        assertEquals("3", "2", b[1]);
    }

    public void testGetField() throws Exception {
        assertNotNull("1", Reflector.getField(MyClass.class, "_myString"));
        assertNotNull("2", Reflector.getField(MyClass2.class, "_myString"));
    }

    public void testSetFieldValue() throws Exception {
        MyClass myClass = new MyClass();
        Reflector.setFieldValue(myClass, "_myString", "test");
        assertEquals("1", "test", myClass._myString);
    }
    
    public void testGetProperty() throws Exception {
        MyClass me = new MyClass();
        assertEquals("1", new Integer(-1), Reflector.getProperty(me, "myInt"));
        assertEquals("2", new Integer(-1), Reflector.getProperty(me, "myInteger"));
    }
    
    public void testSetProperty2() throws Exception {
        MyClass me = new MyClass();
        Reflector.setProperty2(me, "myInt", new Integer(3));
        assertEquals("1", 3, me.getMyInt());
    }

	public void testGetReadMethod() throws Exception {
        Method m = Reflector.getReadMethod(MyClass.class, "myInt");
        assertEquals("1", "getMyInt", m.getName());
    }
    
    public void testGetWriteMethod() throws Exception {
        Method m = Reflector.getWriteMethod(MyClass.class, "myInt");
        assertEquals("1", "setMyInt", m.getName());
    }
    
    public void testToStringForBean() throws Exception {
    	MyClass bean = new MyClass();
        String s = Reflector.toStringForBean(bean);
        System.out.println(s);
        assertNotNull("1", s);
    }
    
    public void testGetConstants() throws Exception {
    	Map consts = Reflector.getConstants(Constants.class);
    	assertEquals("1", new Integer(1), consts.get("ONE"));
    	assertEquals("2", "a", consts.get("A"));
    }
    
    public void testGetMethods() throws Exception {
    	List methods = Reflector.getMethods(String.class, "getBytes", 1);
    	assertEquals("1", 1, methods.size());
    }
    
    public void testDecapitalizeProperyName() throws Exception {
    	assertEquals("1", "x", Reflector.decapitalizePropertyName("X"));
    	assertEquals("2", "SQL", Reflector.decapitalizePropertyName("SQL"));
    }
    
    public void testNewPropertyDescMap() throws Exception {
    	EMap map = Reflector.newPropertyDescMap(MyBean.class);
    	System.out.println(map);
    	assertEquals("1", 3, map.size());
    	PropertyDesc desc = (PropertyDesc) map.get("aaa");
    	assertEquals("2", "aaa", desc.getPropertyName());
    	assertEquals("3", String.class, desc.getPropertyType());
    	assertNotNull("4", desc.getReadMethod());
    	assertNull("5", desc.getWriteMethod());
    	
    	desc = (PropertyDesc) map.get("CCC");
    	assertEquals("6", "CCC", desc.getPropertyName());
    	assertEquals("7", boolean.class, desc.getPropertyType());
    	assertNotNull("8", desc.getReadMethod());
    	assertNull("9", desc.getWriteMethod());
    	
    	desc = (PropertyDesc) map.get("eee");
    	assertEquals("10", "eee", desc.getPropertyName());
    	assertEquals("11", String.class, desc.getPropertyType());
    	assertNotNull("12", desc.getReadMethod());
    	assertNotNull("13", desc.getWriteMethod());	
    }
    
    public void testGetPropertyDescMap() throws Exception {
    	EMap map = Reflector.getPropertyDescMap(MyBean.class);
    	EMap map2 = Reflector.getPropertyDescMap(MyBean.class);
    	assertSame("1", map, map2);
    }

    protected void setUp() throws Exception { }


    protected void tearDown() throws Exception { }


    public static Test suite() {
        return new TestSuite(ReflectorTest.class);
    }


    public static void main(String[] args) {
        junit.textui.TestRunner.main(new String[]{ReflectorTest.class.getName()});
    }

    public static class MyClass {

        private char _myChar = '0';
        private short _myShort = -1;
        private int _myInt = -1;
        private long _myLong = -1;
        private float _myFloat = -1;
        private double _myDouble = -1;
        private boolean _myBoolean = false;
        private String _myString;
        private Integer _myInteger = new Integer(-1);

        public void setMyChar(char myChar) {
            _myChar = myChar;
        }

        public void setMyShort(short myShort) {
            _myShort = myShort;
        }

		public int getMyInt() {
            return _myInt;
        }
        
        public void setMyInt(int myInt) {
            _myInt = myInt;
        }

        public void setMyLong(long myLong) {
            _myLong = myLong;
        }

        public void setMyFloat(float myFloat) {
            _myFloat = myFloat;
        }

        public void setMyDouble(double myDouble) {
            _myDouble = myDouble;
        }

        public void setMyBoolean(boolean myBoolean) {
            _myBoolean = myBoolean;
        }

        public void setMyString(String myString) {
            _myString = myString;
        }
        
        public Integer getMyInteger() {
            return _myInteger;
        }

        private void greeting() {
            System.out.println("hello");
        }
    }

    public static class MyClass2 extends MyClass {
    	
		public void throwError() {
			throw new Error("test");
		}
		
		public void throwException() throws SeasarException {
			throw new SeasarException("ESSR0001", new Object[]{"aaa"});
		}
    }
    
    public static class MyBean {
    	
		public String getAaa() {
			return null;
		}
		
		public String getBbb(Object a) {
			return null;
		}
		
		public boolean isCCC() {
			return true;
		}
		
		public Object isDdd() {
			return null;
		}
		
		public String getEee() {
			return null;
		}
		
		public void setEee(String eee) {
		}
    }
}
