package test.org.seasar.util;

import java.io.File;
import java.net.URL;
import java.util.Properties;

import junit.framework.TestCase;

import org.seasar.util.ResourceUtil;

public class ResourceUtilTest extends TestCase {

    public ResourceUtilTest(String name) {
        super(name);
    }

    public void testResolveName() throws Exception {
        String resolvedName = "test/org/seasar/util/ResourceUtilTest.class";
        assertEquals("1", resolvedName, ResourceUtil.resolveName("test.org.seasar.util.ResourceUtilTest.class", "class"));
        assertEquals("2", resolvedName, ResourceUtil.resolveName("/test/org/seasar/util/ResourceUtilTest.class", "class"));
        assertEquals("3", resolvedName, ResourceUtil.resolveName("/test/org/seasar/util/ResourceUtilTest", "class"));
		assertEquals("4", "test/HelloFlowlet.xml", ResourceUtil.resolveName("test.HelloFlowlet", "xml"));
    }

    public void testTrimExtension() throws Exception {
        String trimedName = "ResourceUtilTest";
        assertEquals("1", trimedName, ResourceUtil.trimExtension("ResourceUtilTest.class", "class"));
        assertEquals("2", trimedName, ResourceUtil.trimExtension("ResourceUtilTest", "class"));
        assertEquals("3", trimedName, ResourceUtil.trimExtension("ResourceUtilTest.class", null));
    }

    public void testUnresolveName() throws Exception {
        String unresolvedName = "test.org.seasar.util.ResourceUtilTest";
        assertEquals("1", unresolvedName, ResourceUtil.unresolveName("test.org.seasar.util.ResourceUtilTest", "class"));
        assertEquals("2", unresolvedName, ResourceUtil.unresolveName("/test/org/seasar/util/ResourceUtilTest.class", "class"));
        assertEquals("3", unresolvedName, ResourceUtil.unresolveName("test/org/seasar/util/ResourceUtilTest.class", "class"));
        assertEquals("4", unresolvedName, ResourceUtil.unresolveName("test/org/seasar/util/ResourceUtilTest.class", null));
    }


    public void testGetResourceAsStream() throws Exception {
        assertNotNull("1", ResourceUtil.getResourceAsStream("test.org.seasar.util.ResourceUtilTest", "class"));
        assertNotNull("2", ResourceUtil.getResourceAsStream("/test/org/seasar/util/ResourceUtilTest.class", "class"));
        assertNotNull("3", ResourceUtil.getResourceAsStream("/test/org/seasar/util/ResourceUtilTest.class"));
        assertNotNull("4", ResourceUtil.getResourceAsStream("test/org/seasar/util/ResourceUtilTest.class"));
        assertNotNull("5", ResourceUtil.getResourceAsStream("/message-config.properties"));
        assertNotNull("6", ResourceUtil.getResourceAsStream("/message-config.properties", "properties"));
    }

    public void testGetResourceAsStream2() throws Exception {
        assertNotNull("1", Thread.currentThread().getContextClassLoader().getResourceAsStream("test/org/seasar/util/ResourceUtilTest.class"));
    }

    public void testGetFileFromClassPath() throws Exception {
        assertNotNull("1", ResourceUtil.getFileFromClassPath("log4j.jar"));
    }

    public void testIsJarFile() throws Exception {
        File file = ResourceUtil.getFileFromClassPath("log4j.jar");
        assertEquals("1", true, ResourceUtil.isJarFile(file));
        File file2 = new File(".");
        assertEquals("2", false, ResourceUtil.isJarFile(file2));
    }

    public void testIsAvailableDirectory() throws Exception {
        File file = ResourceUtil.getFileFromClassPath("log4j.jar");
        assertEquals("1", false, ResourceUtil.isAvailableDirectory(file));
        File file2 = new File(".");
        assertEquals("2", true, ResourceUtil.isAvailableDirectory(file2));
    }

    public void testGetProperties() throws Exception {
        Properties props = ResourceUtil.getProperties("message-config");
        Properties props2 = ResourceUtil.getProperties("message-config.properties");
        assertTrue("1", props.isEmpty() == false);
        assertTrue("2", props2.isEmpty() == false);
    }

    public void testIsExist() throws Exception {
        assertEquals("1", true, ResourceUtil.isExist("message-config.properties"));
        assertEquals("2", false, ResourceUtil.isExist("hoge.xml"));
        assertEquals("3", true, ResourceUtil.isExist("/java/lang/String.class"));
    }
    
    public void testGetResource() throws Exception {
    	URL url = ResourceUtil.getResource("/test/org/seasar/util/resource.data");
        assertNotNull("1", url);
    }
    
    public void testGetLastModified() throws Exception {
    	URL url = ResourceUtil.getResource("/test/org/seasar/util/resource.data");
    	long lm = ResourceUtil.getLastModified(url);
    	System.out.println(lm);
        assertEquals("1", true, lm > 0);
        
        URL url2 = ResourceUtil.getResource("/java/lang/String.class");
    	long lm2 = ResourceUtil.getLastModified(url2);
    	System.out.println(lm2);
        assertEquals("2", true, lm2 > 0);
    }
    
    public void testFile() throws Exception {
        File file = ResourceUtil.getFile("/seasar-context.xml");
        System.out.println(file);
        assertNotNull("1", file);
    }

	public void testGetExtension() throws Exception {
        assertEquals("1", "xml", ResourceUtil.getExtension("hoge.xml"));
        assertEquals("2", "", ResourceUtil.getExtension("hoge"));
    }

    protected void setUp() throws Exception {
    }


    protected void tearDown() throws Exception {
    }

    public static void main(String[] args) {
        junit.textui.TestRunner.main(new String[]{ResourceUtilTest.class.getName()});
    }
}
