package org.seasar.sql;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import javax.sql.XAConnection;
import javax.transaction.xa.XAResource;

import org.seasar.util.SeasarRuntimeException;

public final class XAConnectionImpl implements XAConnection {

    private XAConnection _originalXAConnection;
    private Connection _physicalConnection;
    private ConnectionImpl _logicalConnection;
    private XAResource _xaResource;
    private List _listeners = new ArrayList();

    public XAConnectionImpl(final XAConnection originalXAConnection) throws SQLException {
        if (originalXAConnection == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"originalXAConnection"});
        }
        _originalXAConnection = originalXAConnection;
        _physicalConnection = originalXAConnection.getConnection();
        _xaResource = originalXAConnection.getXAResource();
        _logicalConnection = new ConnectionImpl(this, _physicalConnection);
    }

    public XAConnectionImpl(final Connection physicalConnection) throws SQLException {
        if (physicalConnection == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"physicalConnection"});
        }
        _physicalConnection = physicalConnection;
        _xaResource = new SQLXAResource(this, physicalConnection);
        _logicalConnection = new ConnectionImpl(this, physicalConnection);
    }

    public XAResource getXAResource() {
        return _xaResource;
    }

    public Connection getConnection() throws SQLException {
        return _logicalConnection;
    }

    public ConnectionImpl getLogicalConnection() throws SQLException {
        return _logicalConnection;
    }
    
    public Connection getPhysicalConnection() throws SQLException {
        return _physicalConnection;
    }

    public void close() throws SQLException {
        if (isClosed()) {
            return;
        }
        notifyXAClose();
        closePhysicalConnection();
    }
    
    public void closePhysicalConnection() throws SQLException {
        if (isClosed()) {
            return;
        }
        _physicalConnection.close();
        _physicalConnection = null;
        _logicalConnection = null;
        _xaResource = null;
    }

    public boolean isClosed() {
        return _physicalConnection == null;
    }

    public synchronized void addConnectionEventListener(final ConnectionEventListener listener) {
        _listeners.add(listener);
    }

    public synchronized void removeConnectionEventListener(final ConnectionEventListener listener) {
        _listeners.remove(listener);
    }

    public synchronized void notifyClose() {
        for (int i = 0; i < _listeners.size(); i++) {
            ConnectionEventListener listener = (ConnectionEventListener) _listeners.get(i);
            listener.connectionClosed(new ConnectionEvent(this));
        }
    }

    public synchronized void notifyError(final SQLException ex) {
        for (int i = 0; i < _listeners.size(); i++) {
            ConnectionEventListener listener = (ConnectionEventListener) _listeners.get(i);
            listener.connectionErrorOccurred(new ConnectionEvent(this, ex));
        }
    }
    
    public synchronized void notifyXAClose() {
        for (int i = 0; i < _listeners.size(); i++) {
            ConnectionEventListener listener = (ConnectionEventListener) _listeners.get(i);
            if (listener instanceof XAConnectionEventListener) {
            	((XAConnectionEventListener) listener).xaConnectionClosed(new ConnectionEvent(this));
            }
        }
    }

    synchronized void init() throws SQLException {
        if (_originalXAConnection != null && _physicalConnection.isClosed()) {
        	Connection physicalConnection = _originalXAConnection.getConnection();
        	_xaResource = new SQLXAResource(this, physicalConnection);
            _logicalConnection = new ConnectionImpl(this, physicalConnection);
        }
    }
}
