/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.rtti.ognl;

import org.ognl.el.ExecutionEnvironment;
import org.ognl.el.Extensions;
import org.ognl.el.MethodFailedException;
import org.ognl.el.OgnlException;
import org.ognl.el.PropertyAccessor;
import org.ognl.util.ClassRegistry;
import org.seasar.kijimuna.core.Kijimuna;
import org.seasar.kijimuna.core.rtti.HasErrorRtti;
import org.seasar.kijimuna.core.rtti.IRtti;
import org.seasar.kijimuna.core.rtti.IRttiConstructorDesctiptor;
import org.seasar.kijimuna.core.rtti.IRttiFieldDescriptor;
import org.seasar.kijimuna.core.rtti.IRttiMethodDesctiptor;
import org.seasar.kijimuna.core.rtti.IRttiPropertyDescriptor;
import org.seasar.kijimuna.core.rtti.RttiLoader;

/**
 * @author Masataka Kurihara (Gluegent, Inc)
 */
public class OgnlExtensions extends Extensions {

	private RttiLoader loader;
	private ClassRegistry accessors = new ClassRegistry();
	
	public OgnlExtensions(RttiLoader loader) {
		this.loader = loader;
	}
	
	private IRtti convert(RttiLoader localLoader, Object obj)
			throws OgnlRttiUnprocessable {
	    if (obj instanceof IRtti) {
		    if(obj instanceof HasErrorRtti) {
		        String message = ((HasErrorRtti)obj).getErrorMessage();
				throw new OgnlRttiUnprocessable(message);
		    }
			return (IRtti)obj;
		} else {
			return localLoader.loadRtti(obj.getClass());
		}
	}
	
	private IRtti[] convertArgs(RttiLoader localLoader, Object[] args)
			throws OgnlRttiUnprocessable {
		if(args != null) {
			IRtti[] rttiArgs = new IRtti[args.length];
			for (int i = 0; i < args.length; i++) {
				if(args[i] != null) {
					rttiArgs[i] = convert(localLoader, args[i]);
				}
			}
			return rttiArgs;
		} else {
			return new IRtti[0];
		}
	}
	
	private IRtti invoke(IRtti rtti, String methodName, 
	        Object[] args, boolean staticAccess) throws OgnlRttiUnprocessable {
		RttiLoader localLoader = rtti.getRttiLoader();
		IRtti[] rttiArgs = convertArgs(localLoader, args);
		IRttiMethodDesctiptor descriptor = rtti.getMethod(
		        methodName, rttiArgs, staticAccess);
		if(descriptor != null) {
			return descriptor.getReturnType();
		} else {
		    return loader.loadHasErrorRtti(null, Kijimuna.getResourceString(
		            "rtti.ognl.OgnlExtensions.1",
		            new Object[] {  rtti.getQualifiedName(), methodName, rttiArgs }));
		}
	}
	
	private PropertyAccessor getPropertyAccessor(Class forClass) {
		return (PropertyAccessor)accessors.get(forClass);
	}

	public void setPropertyAccessor(Class forClass, PropertyAccessor accessor) {
		accessors.put(forClass, accessor);
	}
	
	public Object callArrayConstructor(ExecutionEnvironment environment,
			String componentClassName, Object[] args) throws OgnlException {
		return loader.loadRtti(componentClassName + "[]");
	}

	public Object callConstructor(ExecutionEnvironment environment,
			String targetClassName, Object[] args) throws OgnlException {
		IRtti rtti = loader.loadRtti(targetClassName);
		if(rtti instanceof HasErrorRtti) {
		    return rtti;
		}
		IRtti[] rttiArgs = convertArgs(loader, args);
		IRttiConstructorDesctiptor descriptor = rtti.getConstructor(rttiArgs);
		if(descriptor != null) {
			return rtti;
		} else {
		    return loader.loadHasErrorRtti(null,
		            Kijimuna.getResourceString("rtti.ognl.OgnlExtensions.2",
		            new Object[] {  targetClassName, rtti.getShortName(), rttiArgs }));
		}
	}
	
	public Object callMethod(ExecutionEnvironment environment, Object target,
			String methodName, Object[] args) throws MethodFailedException,
			OgnlException {
		IRtti rtti = convert(loader, target);
		if(rtti instanceof HasErrorRtti) {
		    return rtti;
		}
	    return invoke(rtti, methodName, args, false);
	}
	
	public Object callStaticMethod(ExecutionEnvironment environment,
			String targetClassName, String methodName, Object[] args)
			throws MethodFailedException, OgnlException {
		IRtti rtti = loader.loadRtti(targetClassName);
		if(rtti instanceof HasErrorRtti) {
		    return rtti;
		}
	    return invoke(rtti, methodName, args, true);
	}
	
	public Object getIndexedPropertyValue(ExecutionEnvironment environment,
			Object source, Object index) throws OgnlException {
		throw new OgnlRttiUnprocessable();
	}
	
	public Object getNamedIndexedPropertyValue(
			ExecutionEnvironment environment, Object source,
			String propertyName, Object index) throws OgnlException {
		throw new OgnlRttiUnprocessable();
	}
	
	public Object getPropertyValue(ExecutionEnvironment environment,
			Object source, Object property) throws OgnlException {
		PropertyAccessor accessor = getPropertyAccessor(source.getClass());
		if(accessor != null) {
			return accessor.getPropertyValue(environment, source, property);
		} else {
			IRtti rtti = convert(loader, source);
			if(rtti instanceof HasErrorRtti) {
			    return rtti;
			}
			String propertyName = property.toString();
		    IRttiPropertyDescriptor prop = rtti.getProperty(propertyName);
		    if((prop != null) && prop.isReadable()) {
		        return prop.getType();
		    } else {
		        IRttiFieldDescriptor field = rtti.getField(propertyName, false);
		        if(field != null) {
		            return field.getType();
		        } else {
				    return loader.loadHasErrorRtti(null,
				            Kijimuna.getResourceString("rtti.ognl.OgnlExtensions.3",
				            new Object[] {  rtti.getQualifiedName(), propertyName }));
		        }
			}
		}
	}

	public Object getStaticFieldValue(ExecutionEnvironment environment,
			String targetClassName, String fieldName) throws OgnlException {
		IRtti rtti = loader.loadRtti(targetClassName);
		if(rtti instanceof HasErrorRtti) {
		    return rtti;
		}
        IRttiFieldDescriptor field = rtti.getField(fieldName, true);
        if(field != null) {
            return field.getType();
        } else {
		    return loader.loadHasErrorRtti(null,
		            Kijimuna.getResourceString("rtti.ognl.OgnlExtensions.4",
		            new Object[] {  rtti.getQualifiedName(), fieldName }));
        }
	}

	public void setIndexedPropertyValue(ExecutionEnvironment environment,
			Object target, Object index, Object value) throws OgnlException {
		throw new OgnlRttiUnprocessable();
	}
	
	public void setNamedIndexedPropertyValue(ExecutionEnvironment environment,
			Object target, String propertyName, Object index, Object value)
			throws OgnlException {
		throw new OgnlRttiUnprocessable();
	}
	
	public void setPropertyValue(ExecutionEnvironment environment,
			Object target, Object property, Object value) throws OgnlException {
		PropertyAccessor accessor = getPropertyAccessor(target.getClass());
		if(accessor != null) {
			accessor.setPropertyValue(environment, target, property, value);
		} else {
			IRtti rtti = convert(loader, target);
			if(rtti instanceof HasErrorRtti) {
			    return;
			}
			String propertyName = property.toString();
			IRtti rttiValue = convert(loader, value);
		    IRttiPropertyDescriptor prop = rtti.getProperty(propertyName);
		    if((prop == null) || !prop.isWritable() || 
		            prop.getType().isAssignableFrom(rttiValue)) {
		        IRttiFieldDescriptor field = rtti.getField(propertyName, false);
		        if((field == null) || (field.isFinal()) ||
		               !field.getType().isAssignableFrom(rttiValue)) {
		    		throw new OgnlRttiUnprocessable(Kijimuna.getResourceString(
				            "rtti.ognl.OgnlExtensions.5",
				            new Object[] {  rtti.getQualifiedName(), propertyName }));
		        }
			}
		}
	}

}