/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.rtti;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IStorage;
import org.eclipse.core.resources.IResource;
import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IField;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.Signature;
import org.seasar.kijimuna.core.util.ProjectUtils;

/**
 * @author Masataka Kurihara (Gluegent,Inc.)
 */
class DefaultRtti implements IRtti {

	private static final Map WRAPPERS;
	static {
		WRAPPERS = new HashMap();
		WRAPPERS.put("java.lang.Boolean", "boolean");
		WRAPPERS.put("java.lang.Byte", "byte");
		WRAPPERS.put("java.lang.Character", "char");
		WRAPPERS.put("java.lang.Double", "double");
		WRAPPERS.put("java.lang.Float", "float");
		WRAPPERS.put("java.lang.Integer", "int");
		WRAPPERS.put("java.lang.Long", "long");
		WRAPPERS.put("java.lang.Short", "short");
		WRAPPERS.put("java.lang.Void", "void");
	}
	
	private class RttiInfo {
	    private DefaultRtti rtti;	    
	    RttiInfo(DefaultRtti rtti) {
	        this.rtti = rtti;
	    }
		boolean isArray() {
		    return rtti.isArray();
		}
		boolean isPrimitive() {
		    return rtti.isPrimitive();
		}
		String getWrapperName() {
		    return rtti.getWrapperName();
		}
		IRtti getArrayItemClass() {
		    return rtti.getArrayItemClass();
		}
	}
    
    public static boolean isMatchArgs(IRtti[] source, IRtti[] test) {
		if(test == null) {
		    test = new IRtti[0];
		}
		if (source.length == test.length) {
			boolean match = true;
			for (int k = 0; k < source.length; k++) {
				if (!source[k].isAssignableFrom(test[k])) {
					match = false;
					break;
				}
			}
			if (match) {
				return true;
			}
		}
		return false;
    }
	
	private transient IType thisType;
		
	private RttiLoader loader;
	private String projectName;
	private String qualifiedName;
	private boolean primitive;
	private int arrayDepth;
	private IRtti arrayItem;
	private boolean autoConvert;

	DefaultRtti(RttiLoader loader, IType thisType, String qualifiedName,
			boolean primitive, int arrayDepth, IRtti arrayItem, boolean autoConvert) {
		this.loader = loader;
		this.thisType = thisType;
		this.qualifiedName = qualifiedName;
		this.primitive = primitive;
		this.arrayDepth = arrayDepth;
		this.arrayItem = arrayItem;
		this.autoConvert = autoConvert;
		projectName = loader.getProject().getElementName();
	}

	private boolean isTypeAvailable() {
		return getType() != null;
	}

	private boolean isArray() {
		return arrayDepth > 0;
	}
	
	private boolean isPrimitive() {
	    return primitive;
	}

	private String getWrapperName() {
		return qualifiedName;
	}

	private IRtti getArrayItemClass() {
		if (isArray()) {
			return arrayItem;
		} else {
			return null;
		}
	}

	private boolean isWideningPrimitiveConversion(IRtti testRtti) {
	    String thisQName = getQualifiedName();
	    String testQName = testRtti.getQualifiedName();
	    if(autoConvert) {
		    if(WRAPPERS.containsKey(thisQName)) {	
		    	thisQName = WRAPPERS.get(thisQName).toString();
		    }
		    if(WRAPPERS.containsKey(testQName)) {	
		    	testQName = WRAPPERS.get(testQName).toString();
		    }
	    }
	    if(testQName.equals("byte")) {
	        return "short,int,long,float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("short")) {
	        return "int,long,float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("char")) {
	        return "int,long,float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("int")) {
	        return "long,float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("long")) {
	        return "float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("float")) {
	        return "double".indexOf(thisQName) != -1;
	    } else {
	        return false;
	    }
	}

	private boolean isWideningArrayConversion(IRtti testRtti) {
	    String thisQName = getQualifiedName();
	    if(thisQName.equals("java.lang.Object") ||
	            thisQName.equals("java.lang.Cloneable") ||
	            thisQName.equals("java.io.Serializable")) {
	        return true;
	    } else {
	        IRtti thisArray = getArrayItemClass();
			RttiInfo testInfo = (RttiInfo)testRtti.getAdapter(RttiInfo.class);
	        IRtti testArray = testInfo.getArrayItemClass();
	        return (thisArray != null) && (testArray != null) &&
	        	thisArray.isAssignableFrom(testArray);
	    }
	}

	private boolean isPublicMember(IMember member) {
		if(!isInterface()) {
		    try {
                int flags = member.getFlags();
                if(!Flags.isPublic(flags)) {
                    return false;
                }
            } catch (JavaModelException e) {
            }
		}
		return true;
	}
	
	private String createDescriptorKey(IRttiInvokableDesctiptor descriptor) {
	    StringBuffer buffer = new StringBuffer();
	    buffer.append(descriptor.getMethodName()).append("(");
	    IRtti[] args = descriptor.getArgs();
	    for(int i = 0; i < args.length; i++) {
	        if(i != 0) {
	            buffer.append(", ");
	        }
	        buffer.append(args[i].getQualifiedName());
	    }
	    return buffer.toString();
	}
	
    private Map getInvokableMap(Pattern pattern, boolean isConstructor) {
        Map descriptors = new HashMap();
		if(isTypeAvailable()) {
			try {
				IMethod[] methods = getType().getMethods();
				for(int i = 0; i < methods.length; i++) {
				    if(isPublicMember(methods[i])) {
				        String methodName = methods[i].getElementName();
			            if(pattern.matcher(methodName).matches()) {
				    		IRttiInvokableDesctiptor descriptor;
			                if(isConstructor) {
			                    descriptor = createConstructorDescriptor(methods[i]);
			                } else {
			                    descriptor = createMethodDescriptor(methods[i]);
			                }
			                descriptors.put(createDescriptorKey(descriptor), descriptor);
				        }
					}
				}
				if(!isConstructor) {
					DefaultRtti superClass = (DefaultRtti)getSuperClass();
					if (superClass != null) {
					    Map superMethods = superClass.getInvokableMap(pattern, isConstructor);
					    for(Iterator it = superMethods.keySet().iterator(); it.hasNext();) {
					        String key = (String)it.next();
					        if(!descriptors.containsKey(key)) {
					            descriptors.put(key, superMethods.get(key));
					        }
						}
					}
				}
			} catch (JavaModelException ignore) {
			}
		}
		return descriptors;	
    }

    private IRttiConstructorDesctiptor createConstructorDescriptor(
    		IMethod constructor) {
		return new DefaultRttiConstructorDescriptor(constructor, this, false);
    }
    
    private IRttiMethodDesctiptor createMethodDescriptor(IMethod method) {
		return new DefaultRttiMethodDescriptor(method, this);
    }
    
	private IRttiConstructorDesctiptor createDefaultConstructor() {
		try {
            IMethod[] methods = getType().getMethods();
            String shortName = getShortName();
            for(int i = 0; i < methods.length; i++) {
                if((methods[i].getElementName().equals(shortName) &&
                        methods[i].getParameterTypes().length == 0)) {
                    return null;
                }
            }
    	    return new DefaultRttiConstructorDescriptor(getType(), this, true);
        } catch (JavaModelException e) {
            return null;
        }
	}
	
	private String getPropertyType(
	        IMethod method, String returnType, int parameterNum) {
        try {
	        String[] args = method.getParameterTypes();
	        if(args.length == parameterNum) {
	            if(parameterNum == 0) {
	                String ret = Signature.toString(method.getReturnType());
	                if((returnType == null) || returnType.equals(ret)) {
	                    return ret;
	                }
	            } else {
	                return Signature.toString(args[0]);
	            }
	        }
        } catch (Exception e) {
        }
        return null;
	}
	
	private DefaultRttiPropertyDescriptor getPropertyDescriptor(IMethod method) {
	    String methodName = method.getElementName();
	    String propertyName;
	    boolean isReader;
	    String propertyType;
	    if(methodName.startsWith("set")) {
	        propertyName = methodName.substring(3);
	        isReader = false;
	        propertyType = getPropertyType(method, "void", 1);
	    } else if(methodName.startsWith("get")) {
	        propertyName = methodName.substring(3);
	        isReader = true;
	        propertyType = getPropertyType(method, null, 0);
	    } else if(methodName.startsWith("is")) {
	        propertyName = methodName.substring(2);
	        isReader = true;
	        propertyType = getPropertyType(method, "boolean", 0);
	    } else {
	        return null;
	    }
        if(propertyType == null) {
            return null;
        }
	    if((propertyName.length() == 0) || 
                Character.isLowerCase(propertyName.charAt(0))) {
            return null;
        }
        if((propertyName.length() == 1) || 
                (Character.isLowerCase(propertyName.charAt(1)))) {
            char[] chars = propertyName.toCharArray();
            chars[0] = Character.toLowerCase(chars[0]);
            propertyName = new String(chars);
        }
		IRtti propertyRtti = loader.loadRtti(propertyType);
	    return new DefaultRttiPropertyDescriptor(
	            this, propertyName, propertyRtti, isReader);
	}
	
	private void margePropertyDescriptor(
	        Map map, DefaultRttiPropertyDescriptor descriptor) {
        String propertyName = descriptor.getName();
	    DefaultRttiPropertyDescriptor old = 
	        (DefaultRttiPropertyDescriptor)map.get(propertyName);
	    if(old != null) {
		    IRtti propertyType = descriptor.getType();
		    boolean isReadable = descriptor.isReadable();
		    boolean isWritable = descriptor.isWritable();
		    if(propertyType.equals(old.getType())) {
		        if(isReadable) {
		            old.doReadable(propertyType);
	            }
		        if(isWritable) {
		            old.doWritable(propertyType);
		        }
		    }
	    } else {
	        map.put(propertyName, descriptor);
	    }
	}

	private Map getPropertyMap() {
	    if(isTypeAvailable()) {
			try {
				IMethod[] methods = getType().getMethods();
				Map map = new HashMap();
				for(int i = 0; i < methods.length; i++) {
				    if(isPublicMember(methods[i])) {
				        DefaultRttiPropertyDescriptor descriptor =
				            getPropertyDescriptor(methods[i]);
				        if(descriptor != null) {
				            margePropertyDescriptor(map, descriptor);
				        }
				    }
				}
				IRtti parent = getSuperClass();
				if(parent != null) {
				    IRttiPropertyDescriptor[] props = parent.getProperties();
				    if(props != null) {
					    for(int i = 0; i < props.length; i++) {
						    margePropertyDescriptor(map, 
						            (DefaultRttiPropertyDescriptor)props[i]);
					    }
				    }
				}
				return map;
			} catch (JavaModelException ignore) {
			}
		}
		return null;	
	}
	
	public boolean equals(Object test) {
		if((test != null) && (test instanceof IRtti)) {
			IRtti testRtti = (IRtti) test;
			RttiInfo testInfo = (RttiInfo)testRtti.getAdapter(RttiInfo.class);
			if (autoConvert && (isPrimitive() || testInfo.isPrimitive())) {
				if (getWrapperName().equals(testInfo.getWrapperName())) {
					return true;
				}
			}
			if (isArray()) {
				if (testInfo.isArray()) {
					IRtti thisArrayItem = getArrayItemClass();
					IRtti testArrayItem = testInfo.getArrayItemClass();
					return thisArrayItem.equals(testArrayItem);
				}
				return false;
			}
			String qname = testRtti.getQualifiedName();
			if (getQualifiedName().equals(qname)) {
				return true;
			}
		}
		return false;
	}
    
	public String toString() {
		return getQualifiedName();
	}

    public Object getAdapter(Class adapter) {
        if(RttiInfo.class.equals(adapter)) {
            return new RttiInfo(this);
        } else if(IStorage.class.equals(adapter)) {
            if(isTypeAvailable()) {
                IType type = getType();
                if(type != null) {
                    IResource resource = type.getResource();
                    if(resource != null) {
                        return resource;
                    }
                }
            }
        } else if(IRtti.class.equals(adapter)){
            return this;
        }
        return null;
    }
	
	public IType getType() {
	    if(thisType == null) {
	        IJavaProject project = ProjectUtils.getJavaProject(projectName);
	        try {
                thisType = project.findType(qualifiedName);
            } catch (JavaModelException e) {
            }
	    }
	    return thisType;
	}
	
	public RttiLoader getRttiLoader() {
	    return loader;
	}

	public boolean isInterface() {
		if (isTypeAvailable()) {
			try {
				return getType().isInterface();
			} catch (JavaModelException ignore) {
			}
		}
		return false;
	}

	public boolean isFinal() {
		if (isTypeAvailable()) {
			try {
				return Flags.isFinal(getType().getFlags());
			} catch (JavaModelException ignore) {
			}
		}
		return false;
	}

	public boolean isAssignableFrom(IRtti testRtti) {
		if(testRtti == null) {
		    return true;
		}
		if (equals(testRtti)) {
			return true;
		} else {
			RttiInfo testInfo = (RttiInfo)testRtti.getAdapter(RttiInfo.class);
			if (isPrimitive() || testInfo.isPrimitive()) {
			    // widening primitive conversion
			    return isWideningPrimitiveConversion(testRtti);
			} else if(testInfo.isArray()) {
			    return isWideningArrayConversion(testRtti);
			} else {
				if (isInterface()) {
					IRtti[] superInterfaces = testRtti.getInterfaces();
					for (int i = 0; i < superInterfaces.length; i++) {
						if (equals(superInterfaces[i])) {
							return true;
						}
					}
				}
				if (testRtti.isInterface()) {
				    return getQualifiedName().equals("java.lang.Object");
				} else {
				    IRtti superClass = testRtti.getSuperClass();
					if (superClass == null) {
						return false;
					} else {
						return isAssignableFrom(superClass);
					}
				}
			}
		}
	}
	
	public String getQualifiedName() {
		if (isArray()) {
			StringBuffer buffer = new StringBuffer(arrayItem.getQualifiedName());
			for (int i = 0; i < arrayDepth; i++) {
				buffer.append("[]");
			}
			return buffer.toString();
		}
		if (isPrimitive()) {
			return (String) WRAPPERS.get(qualifiedName);
		}
		return qualifiedName;
	}

	public String getShortName() {
		String qname = getQualifiedName();
		int pos = qname.lastIndexOf('.');
		if (pos != -1) {
			return qname.substring(pos + 1);
		} else {
			return qname;
		}
	}

	public IRtti[] getInterfaces() {
		if (isTypeAvailable()) {
			try {
				String[] interfaces = getType().getSuperInterfaceNames();
				IRtti[] ret = new IRtti[interfaces.length];
				for (int i = 0; i < interfaces.length; i++) {
					ret[i] = loader.loadRtti(interfaces[i]);
				}
				return ret;
			} catch (Exception ignore) {
			}
		}
		return new IRtti[0];
	}

	public IRtti getSuperClass() {
		if (isTypeAvailable() && !isInterface()
				&& !getQualifiedName().equals("java.lang.Object")) {
			try {
				String superClassName = getType().getSuperclassName();
				if (superClassName == null) {
					superClassName = "java.lang.Object";
				}
				return loader.loadRtti(superClassName);
			} catch (Exception ignore) {
			}
		}
		return null;
	}

	public IRttiFieldDescriptor getField(String name, boolean staticAccess) {
		try {
		    if(name.equals("class") && staticAccess) {
		        return new DefaultRttiFieldDescriptor(
		                this, name, loader.loadRtti("java.lang.Class"), true, true);
		    }
		    if (isArray() && "length".equals(name)) {
				return new DefaultRttiFieldDescriptor( 
				        this, name, loader.loadRtti("int"), true, false); 
			}
			if (isTypeAvailable()) {
				IField value = getType().getField(name);
				if (value.exists() && isPublicMember(value)) {
					int flags = value.getFlags();
					if (staticAccess) {
						if (!Flags.isStatic(flags)) {
							return null;
						}
					}
					String typeSignature = 
					    Signature.toString(value.getTypeSignature());
					return new DefaultRttiFieldDescriptor(
					        this, name, loader.loadRtti(typeSignature),
					        Flags.isFinal(flags), Flags.isStatic(flags));
				} else {
					IRtti[] interfaces = getInterfaces();
					for (int i = 0; i < interfaces.length; i++) {
					    IRttiFieldDescriptor ret = 
					        interfaces[i].getField(name, staticAccess);
					    if(ret != null) {
					        return ret;
					    }
					}
					IRtti superClass = getSuperClass();
					if (superClass != null) {
						return superClass.getField(name, staticAccess);
					}
				}
			}
		} catch (Exception ignore) {
		}
		return null;
	}
	
	public IRttiConstructorDesctiptor getConstructor(IRtti[] args) {
		IRttiConstructorDesctiptor[] constructors = getConstructors();
	    if(constructors.length > 0) {
	    	for(int i = 0; i < constructors.length; i++) {
	    		IRtti[] rttiArgs = constructors[i].getArgs();
	    		if(isMatchArgs(rttiArgs, args)) {
	    			return constructors[i];
	    		}
	    	}
	    } else if((args == null) || (args.length == 0)) {
	        return createDefaultConstructor();
	    }
	    return null;
	}
	
	public IRttiConstructorDesctiptor[] getConstructors() {
	    Pattern pattern = Pattern.compile(getShortName());
        Map descriptors = getInvokableMap(pattern, true);
        if(descriptors.size() == 0) {
	    	IRttiConstructorDesctiptor def = createDefaultConstructor();
	        if(def != null) {
	            return new IRttiConstructorDesctiptor[] { def };
	        } else {
	        	return new IRttiConstructorDesctiptor[0];
	        }
        } else {
	        IRttiConstructorDesctiptor[] ret = 
	            new IRttiConstructorDesctiptor[descriptors.size()];
	        int i = 0;
	        for(Iterator it = descriptors.entrySet().iterator(); it.hasNext(); i++) {
	            Map.Entry entry = (Map.Entry)it.next();
	            ret[i] = (IRttiConstructorDesctiptor)entry.getValue();
	        }
	        return ret;
        }
	}

	public IRttiMethodDesctiptor getMethod(
	        String methodName, IRtti[] args, boolean staticAccess) {
		Pattern pattern = Pattern.compile(methodName); 
		IRttiMethodDesctiptor[] methods = getMethods(pattern);
	    if(methods.length > 0) {
	    	for(int i = 0; i < methods.length; i++) {
	    		if(staticAccess && !methods[i].isStatic()) {
	    			continue;
	    		} else {
		    		IRtti[] rttiArgs = methods[i].getArgs();
		    		if(isMatchArgs(rttiArgs, args)) {
		    			return methods[i];
		    		}
	    		}
	    	}
	    }	    
		return null;
	}
	
	public IRttiMethodDesctiptor[] getMethods(Pattern pattern) {
        Map descriptors = getInvokableMap(pattern, false);
        IRttiMethodDesctiptor[] ret = 
            new IRttiMethodDesctiptor[descriptors.size()];
        int i = 0;
        for(Iterator it = descriptors.entrySet().iterator(); it.hasNext(); i++) {
            Map.Entry entry = (Map.Entry)it.next();
            ret[i] = (IRttiMethodDesctiptor)entry.getValue();
        }
        return ret;
	}

	public IRttiPropertyDescriptor getProperty(String name) {
	    if(isTypeAvailable()) {
		    Map map = getPropertyMap();
		    return (IRttiPropertyDescriptor)map.get(name);
	    } else {
	        return null;
	    }
	}
	
	public IRttiPropertyDescriptor[] getProperties() {
	    if(isTypeAvailable()) {
		    Map map = getPropertyMap();
		    IRttiPropertyDescriptor[] ret = new IRttiPropertyDescriptor[map.size()];
		    Iterator it = map.entrySet().iterator();
		    for(int i = 0; i < ret.length && it.hasNext(); i++) {
		        Map.Entry entry = (Map.Entry)it.next();
		        ret[i] = (IRttiPropertyDescriptor)entry.getValue();
		    }
			return ret;
	    } else {
	        return new IRttiPropertyDescriptor[0]; 	        
	    }
	}
	
}