/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.rtti;

import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.Signature;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
abstract class AbstractRttiInvokableDescriptor 
		implements IRttiInvokableDesctiptor {

	private transient IMember member;

	private boolean isIMethod;
	private IRtti parent;
	private String methodName;
	private IRtti returnType;
	private IRtti[] args;
	private boolean fFinal;
	private boolean fStatic;
	private IRtti[] values;
	
	AbstractRttiInvokableDescriptor(IMember member, IRtti parent) {
	    if(member instanceof IMethod) {
	        isIMethod = true;
	    }
	    this.member = member;
		this.parent = parent;
		methodName = member.getElementName();
		returnType = createReturnType(member, parent.getRttiLoader());
		args = createArgRttis(member, parent.getRttiLoader());
        fFinal = isFinal(member);
        fStatic = isStatic(member);
	}

	private IRtti createReturnType(IMember member, RttiLoader loader) {
		if(member instanceof IMethod) {
		    IMethod method = (IMethod)member;
		    try {
	            String retType = method.getReturnType();
	            String resolvedRet = Signature.toString(retType);
	            return loader.loadRtti(resolvedRet);
	        } catch (JavaModelException e) {
	            return null;
	        }
		} else {
		    return loader.loadRtti("void");
		}
	}
    
    private IRtti[] createArgRttis(IMember member, RttiLoader loader) {
		if(member instanceof IMethod) {
		    IMethod method = (IMethod)member;
			String[] argTypes = method.getParameterTypes();
			IRtti[] rttiArgs = new IRtti[argTypes.length];
			for (int k = 0; k < argTypes.length; k++) {
				String resolvedName = Signature.toString(argTypes[k]);
				rttiArgs[k] = loader.loadRtti(resolvedName);
			}
			return rttiArgs;
		} else {
		    return new IRtti[0];
		}
    }
	
	private boolean isFinal(IMember member) {
		if(member instanceof IMethod) {
	    	try {
				int flag = member.getFlags();
				return Flags.isFinal(flag);
			} catch (JavaModelException ignore) {
				return false;
			}
		} else {
		    return false;
		}
    }
    
    private boolean isStatic(IMember member) {
		if(member instanceof IMethod) {
	    	try {
				int flag = member.getFlags();
				return Flags.isStatic(flag);
			} catch (JavaModelException ignore) {
				return false;
			}
		} else {
		    return false;
		}
    }
	
    private String[] reverseArgRttis() {
        boolean binary = parent.getType().isBinary();
        String[] signature = new String[args.length];
        for(int i = 0; i < args.length; i++) {
            String qname = args[i].getQualifiedName();
            signature[i] = Signature.createTypeSignature(qname, binary);
        }
        return signature;
    }
    
    private IMember findIMethod() {
        IType type = parent.getType();
        IMethod method = type.getMethod(methodName, reverseArgRttis());
        if(!(methodName.equals(parent.getShortName()))) {
	        while(method == null) {
				try {
					String superClassName = type.getSuperclassName();
					if (superClassName == null) {
						superClassName = "java.lang.Object";
					}
					IType superType = type.getJavaProject().findType(superClassName);
					method = superType.getMethod(methodName, reverseArgRttis());  
				} catch (Exception ignore) {
				}
	        }
        }
        if(method != null) {
            return method;
        } else {
            return type;
        }
    }
    
	public boolean equals(Object test) {
        if(test instanceof IRttiInvokableDesctiptor) {
            IRttiInvokableDesctiptor desc = (IRttiInvokableDesctiptor)test;
            return parent.equals(desc.getParent()) &&
            	getMethodName().equals(desc.getMethodName()) &&
            	DefaultRtti.isMatchArgs(args, desc.getArgs());
        } else {
            return false;
        }
    }
	
	public IMember getMember() {
	    if(member != null) {
	        return member;
	    } else {
	        if(isIMethod) {
	            return findIMethod();
	        } else {
	            return parent.getType();
	        }
	    }
	}
    
	public IRtti getParent() {
		return parent;
	}

	public String getMethodName() {
	    return methodName;
	}
	
    public IRtti getReturnType() {
        return returnType;
    }

    public IRtti[] getArgs() {
		return args;
	}

	public boolean isFinal() {
	    return fFinal;
	}

	public boolean isStatic() {
        return fStatic;
    }
	
	public IRtti[] getValues() {
		return values;
	}

	public void setValues(IRtti[] values) {
		this.values = values;
	}
	
    public int compareTo(Object test) {
        if(test instanceof IRttiInvokableDesctiptor) {
            IRttiInvokableDesctiptor invokabler = (IRttiInvokableDesctiptor)test;
            return getArgs().length - invokabler.getArgs().length;
        } else {
            return 1;
        }
    }

}
