/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.parser;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.IAdaptable;
import org.seasar.kijimuna.core.util.ProjectUtils;
import org.seasar.kijimuna.core.util.StringUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class Element implements IAdaptable, Serializable {

	private Element root;

	private transient IProject project;
	private String projectName;
	private transient IStorage storage;
	private String fullPath;
	private String elementName;
	private int depth;
	private int startLine;
	private int startColumn;
	private Map attributes;

	private Element parent;
	private List children;
	
	private int endLine;
	private int endColumn;
	private String body;
	
	public Element(IProject project, IStorage storage, String elementName) {
	    this.elementName = elementName;
		this.project = project;
		this.storage = storage;
		projectName = project.getName();
		fullPath = ProjectUtils.getPathString(storage);
		children = new ArrayList();
	}
	
	public Element getRootElement() {
		return root;
	}
	
	public void setRootElement(Element root) {
		this.root = root;
	}

	public String getElementName() {
		return elementName;
	}
	
	public int getDepth() {
		return depth;
	}
	
	public int getStartLine() {
		return startLine;
	}
	
	public int getStartColumn() {
		return startColumn;
	}
	
	public void setStartLocation(int depth, int startLine, int startColumn) {
		this.startLine = startLine;
		this.startColumn = startColumn;
		this.depth = depth;
	}
	
	public String[] getAttributeNames() {
		return (String[]) attributes.keySet().toArray(new String[attributes.size()]);
	}

	public String getAttribute(String name) {
		return (String) attributes.get(name);
	}

	public void setAttributes(Map properties) {
		this.attributes = properties;
	}

	public Element getParent() {
		return parent;
	}

	public void setParent(Element parent) {
		this.parent = parent;
		parent.addChild(this);
	}
	
	public int getEndLine() {
		return endLine;
	}

	public int getEndColumn() {
		return endColumn;
	}

	public String getBody() {
		return body;
	}

	public void setBody(String body) {
		this.body = body;
	}
	
	public void setEndLocation(int endLine, int endColumn) {
		this.endLine = endLine;
		this.endColumn = endColumn;
	}

	public List getChildren() {
		return children;
	}

	public List getChildren(String elementName) {
		List list = new ArrayList();
		for(Iterator it = children.iterator(); it.hasNext();) {
			Element child = (Element)it.next();
			if(child.getElementName().equals(elementName)) {
				list.add(child);
			}
		}
		return list;
	}
	
	protected void addChild(Element child) {
		children.add(child);
	}

	public IProject getProject() {
    	if((project == null) && StringUtils.existValue(projectName)) {
    		project = ProjectUtils.getProject(projectName);
    	}
    	return project;
	}

	public IStorage getStorage() {
    	if((storage == null) && StringUtils.existValue(fullPath)) {
    		storage = ProjectUtils.getStorage(getProject(), fullPath);
    	}
        return storage;
	}
	
    public Object getAdapter(Class adapter) {
        if(IProject.class.equals(adapter)) {
            return getProject();
        } else if(IStorage.class.equals(adapter)) {
            return getStorage();
        } else if(Element.class.equals(adapter)) {
            return this;
        }
        return null;
    }
}