/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.dicon.validation;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IStorage;
import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.Kijimuna;
import org.seasar.kijimuna.core.dicon.DiconNature;
import org.seasar.kijimuna.core.parser.Element;
import org.seasar.kijimuna.core.preference.IPreferences;
import org.seasar.kijimuna.core.util.MarkerUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class MarkerSetting implements ConstCore {

    private static String convertKey(int category) {
       return MARKER_SEVERITY_ALL[category];
    }

	private static boolean hasSetting(String[] settings, String key) {
	    for(int i = 0; i < settings.length; i++) {
	        if(settings[i].equals(key)) {
	            return true;
	        }
	    }
	    return false;
	}
	
	private static int getCategoryID(String key) {
	    if(hasSetting(MARKER_SET_XML_ERROR, key)) {
	        return MARKER_CATEGORY_XML_ERROR;
	    } else if(hasSetting(MARKER_SET_XML_WARNING, key)) {
	        return MARKER_CATEGORY_XML_WARNING;
		} else if(hasSetting(MARKER_SET_NULL_INJECTION, key)) {
	        return MARKER_CATEGORY_NULL_INJECTION;
		} else if(hasSetting(MARKER_SET_AUTO_INJECTION, key)) {
	        return MARKER_CATEGORY_AUTO_INJECTION;
		} else if(hasSetting(MARKER_SET_JAVA_FETAL, key)) {
	        return MARKER_CATEGORY_JAVA_FETAL;
		} else if(hasSetting(MARKER_SET_DICON_FETAL, key)) {
	        return MARKER_CATEGORY_DICON_FETAL;
		} else if(hasSetting(MARKER_SET_DICON_PROBLEM, key)) {
	        return MARKER_CATEGORY_DICON_PROBLEM;
		} else {
			return MARKER_CATEGORY_UNKNOWN;
		}
	}
	
	private static String getCategory(String key) {
	    int category = getCategoryID(key); 
		if(category == MARKER_CATEGORY_UNKNOWN) {
			return "";
		} else {
			return convertKey(category);
		}
	}
	
	private static int getSeveritySetting(IProject project, String key) {
	    int category = getCategoryID(key); 
		if(category == MARKER_CATEGORY_UNKNOWN) {
		    Kijimuna.reportInfo("unknown marker id [" + key + "]");
		    return MARKER_SEVERITY_IGNORE;
		} else {
			return getDiconMarkerPreference(project, category, false);
		}
	}
    
	public static void setDiconMarkerPreference(
	        IProject project, int category, int severity) {
	    IPreferences pref;
	    if(project == null) {
	        pref = Kijimuna.getPreferences();
	    } else {
		    DiconNature nature = DiconNature.getInstance(project);
		    if(nature != null) {
		        pref = nature.getPreferences();
	    	} else {
	    	    return;
	    	}
	    }
        String categoryKey = convertKey(category);
        pref.putInt(categoryKey, severity);
	}
	
	public static int getDiconMarkerPreference(
			IProject project, int category, boolean isDefault) {
	    IPreferences pref;
	    if(project == null) {
	        pref = Kijimuna.getPreferences();
	    } else {
		    DiconNature nature = DiconNature.getInstance(project);
		    if(nature != null) {
		        pref = nature.getPreferences();
	    	} else {
		        pref = Kijimuna.getPreferences();
	    	}
	    }
        String categoryKey = convertKey(category);
        if(isDefault) {
        	return pref.getDefaultInt(categoryKey);
        } else {
            return pref.getInt(categoryKey);
        }
	}
	
    public static void createDiconMarker(
    		String id, Element element, String message) {
        IStorage storage = element.getStorage();
        if((storage != null) && (storage instanceof IFile)) {
            IFile file = (IFile)storage;
	    	MarkerUtils.createMarker(ID_MARKER_DICONVALIDAION,
	    			getCategory(id), getSeveritySetting(file.getProject(), id),
					file, element.getStartLine(), 
	    			"[" + element.getElementName() + "] " +  message);
        }
    }

    public static void createDiconMarker(
            String id, Element element, Object[] info) {
        createDiconMarker(id, element, Kijimuna.getResourceString(id, info));
    }

    public static void createDiconMarker(
            String id, Element element) {
        createDiconMarker(id, element, Kijimuna.getResourceString(id));
    }
	
    public static void createProjectMarker(
            String id, IProject project, String message) {
        MarkerUtils.createMarker(ID_MARKER_DICONVALIDAION, 
    			getCategory(id), getSeveritySetting(project, id),
                project, 0, "[project] " +  message);
    }
}
