/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.dicon.model;

import java.util.List;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IStorage;
import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.dicon.info.IComponentKey;
import org.seasar.kijimuna.core.dicon.info.IApplyMethodInfo;
import org.seasar.kijimuna.core.rtti.IRtti;
import org.seasar.kijimuna.core.rtti.IRttiMethodDesctiptor;
import org.seasar.kijimuna.core.util.StringUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class MethodElement extends DiconElement
		implements ConstCore {

	private IApplyMethodInfo info;
	
	public MethodElement(IProject project, IStorage storage, String elementName) {
		super(project, storage, elementName);
	}

	public String getMethodName() {
		return getAttribute(DICON_ATTR_NAME);
	}
	
	public List getArgList() {
	    return getChildren(DICON_TAG_ARG);
	}
	
	public boolean isOGNL() {
	    return StringUtils.noneValue(getMethodName());
	}
	
	public String getDisplayName() {
		StringBuffer buffer = new StringBuffer();
		String method = getMethodName();
		if(isOGNL()) {
			String expression = getExpression();
			if(StringUtils.existValue(expression)) {
				buffer.append("[").append(expression).append("]");
			} else {
				buffer.append("[...]");
			}
		} else {
			buffer.append(method);
		}
		return buffer.toString();
	}

    private IRttiMethodDesctiptor getSuitableMethod(IRtti component, String methodName) {
		if(component != null) {
			IRttiMethodDesctiptor[] methods = 
			    component.getMethods(Pattern.compile(methodName));
			IRttiMethodDesctiptor suitable = null;
			int size = -1;
			for(int i = 0; i < methods.length; i++) {
				IRtti[] rttiArgs = methods[i].getArgs();
				if(size < rttiArgs.length) {
					boolean flag = true;
					for(int k = 0; k < rttiArgs.length; k++) {
						if(!rttiArgs[k].isInterface()) {
							flag = false;
							break;
						}
					}
					if(flag) {
						size = rttiArgs.length;
						suitable = methods[i];
					}
				}
			}
			return suitable;
		}
		return null;
    }
	
	public IRttiMethodDesctiptor findAutoInjectedMethod() {
		String methodName = getMethodName();
		if(StringUtils.existValue(methodName)) {
			ComponentElement component = (ComponentElement)getParent();
			if(StringUtils.existValue(component.getComponentClassName())) {
				IRtti rtti = (IRtti)component.getAdapter(IRtti.class);
				if(rtti != null) {
					IRttiMethodDesctiptor suitable = getSuitableMethod(rtti, getMethodName());
					if(suitable != null) {
					    IRtti[] suitableArgs = suitable.getArgs();
					    IRtti[] injectedArgs = new IRtti[suitableArgs.length];
					    for(int i = 0; i < suitableArgs.length; i++) {
				            ContainerElement container = getContainerElement();
				            IComponentKey key = 
				            	container.createComponentKey(suitableArgs[i]);
				            IRtti inject = container.getComponent(key);
				            injectedArgs[i] = inject;
					    }
					    suitable.setValues(injectedArgs);
						return suitable;
					}
				}
			}
		}
		return null;
	}
	
	public Object getAdapter(Class adapter) {
		if(IApplyMethodInfo.class.equals(adapter)) {
			if(info == null) {
				info = new IApplyMethodInfo() {
					public IRttiMethodDesctiptor getAutoInjectedMethod() {
						return findAutoInjectedMethod();
					}
				};
			}
			return info;
		}
		return super.getAdapter(adapter);
	}
}
