/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.dicon.model;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.IAdaptable;
import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.Kijimuna;
import org.seasar.kijimuna.core.dicon.info.IComponentInfo;
import org.seasar.kijimuna.core.dicon.info.IComponentKey;
import org.seasar.kijimuna.core.rtti.IRtti;
import org.seasar.kijimuna.core.rtti.IRttiConstructorDesctiptor;
import org.seasar.kijimuna.core.rtti.IRttiPropertyDescriptor;
import org.seasar.kijimuna.core.rtti.RttiLoader;
import org.seasar.kijimuna.core.util.ModelUtils;
import org.seasar.kijimuna.core.util.StringUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class ComponentElement extends AbstractExpressionElement
		implements ConstCore {

    private Set componentKeySet;
    private IComponentInfo info;
    
	public ComponentElement(IProject project, IStorage storage) {
		super(project, storage, DICON_TAG_COMPONENT);
	}
	
	protected IRtti getNonExpressionValue() {
		String className = getComponentClassName();
		RttiLoader loader = getRttiLoader();
		if(StringUtils.existValue(className)) {		
			return loader.loadRtti(className);
		} else if(isOuterInjection()) {
			return null;
		} else {
		    return loader.loadHasErrorRtti(null,
		            Kijimuna.getResourceString("dicon.model.ComponentElement.1"));
		}
	}
	
	public String getComponentName() {
		return getAttribute(DICON_ATTR_NAME);
	}
	
	public String getComponentClassName() {
		return getAttribute(DICON_ATTR_CLASS);
	}
	
	public String getAutoBindingMode() {
		String autoBinding = getAttribute(DICON_ATTR_AUTOBINDING);
		if(StringUtils.noneValue(autoBinding)) {
			autoBinding = DICON_VAL_AUTO_BINDING_AUTO;
		}
		return autoBinding;
	}
	
	public String getInstanceMode() {
		String instance = getAttribute(DICON_ATTR_INSTANCE);
		if(instance == null) {
			instance = DICON_VAL_INSTANCE_SINGLETON;
		}
		return instance;
	}
	
	private Set createKeys() {
	    if((componentKeySet == null) || !isLocking()) {
	        ContainerElement container = getContainerElement();
		    componentKeySet = new TreeSet();
			String componentName = getComponentName();
			if(StringUtils.existValue(componentName)) {
			    IComponentKey nameKey = 
			        container.createComponentKey(componentName);
			    componentKeySet.add(nameKey);
			}
		    if(StringUtils.existValue(getComponentClassName())) {
				String instance = getInstanceMode();
				if (!DICON_VAL_INSTANCE_OUTER.equals(instance)) {
				    setLocking(true);
					IRtti rtti = (IRtti)getAdapter(IRtti.class);
					setLocking(false);
					if (rtti != null) {
						if(rtti.isInterface()) {
						    IComponentKey thisKey = 
						    	container.createComponentKey(rtti);
						    componentKeySet.add(thisKey);
						}
						do {
							IRtti interfaces[] = rtti.getInterfaces();
							for (int i = 0; i < interfaces.length; i++) {
							    IComponentKey interfaceKey = 
							        container.createComponentKey(interfaces[i]);
							    componentKeySet.add(interfaceKey);
							}
							rtti = rtti.getSuperClass();
						} while (rtti != null);
					}
				}
		    }
	    }
	    return componentKeySet;
	}
	
	public List getArgList() {
	    return getChildren(DICON_TAG_ARG);
	}
	
	public List getAspectList() {
	    return getChildren(DICON_TAG_ASPECT);
	}
	
	public List getPropertyList() {
	    return getChildren(DICON_TAG_PROPERTY);
	}
	
	public List getInitMethodList() {
	    return getChildren(DICON_TAG_INITMETHOD);
	}
	
	public List getDestroyMethodList() {
	    return getChildren(DICON_TAG_DESTROYMETHOD);
	}
	
	private boolean isOuterInjection() {
	    return DICON_VAL_INSTANCE_OUTER.equals(getInstanceMode());
	}

	private IRttiConstructorDesctiptor getSuitableConstructor(IRtti rtti) {
		int size = -1;
		IRttiConstructorDesctiptor suitable = null;
		IRttiConstructorDesctiptor[] constructors = rtti.getConstructors();
		for(int i = 0; i < constructors.length; i++) {
			IRtti[] rttiArgs = constructors[i].getArgs();
			if(rttiArgs.length == 0) {
			    return null;
			}
			if(size < rttiArgs.length) {
				boolean flag = true;
				for(int k = 0; k < rttiArgs.length; k++) {
					if(!rttiArgs[k].isInterface()) {
						flag = false;
						break;
					}
				}
				if(flag) {
					size = rttiArgs.length;
					suitable = constructors[i];
				}
			}
		}
		return suitable;
	}
	
	private IRttiConstructorDesctiptor findAutoInjectedConstructor() {
        String el = getExpression();
        if((getArgList().size() == 0) && StringUtils.noneValue(el)) {
            String autoBinding = getAutoBindingMode();
            if(autoBinding.equals(DICON_VAL_AUTO_BINDING_AUTO) || 
                    autoBinding.equals(DICON_VAL_AUTO_BINDING_CONSTRUCTOR)) {
        	    IRtti rtti = (IRtti)getAdapter(IRtti.class);
        		if(rtti != null) {
        			IRttiConstructorDesctiptor suitable = getSuitableConstructor(rtti);
        			if(suitable != null) {
        			    IRtti[] suitableArgs = suitable.getArgs();
        			    IRtti[] injectedArgs = new IRtti[suitableArgs.length];
        				for(int i = 0; i < suitableArgs.length; i++) {
    			            ContainerElement container = getContainerElement();
    			            IComponentKey key = 
    			            	container.createComponentKey(suitableArgs[i]);
    			            injectedArgs[i] = container.getComponent(key);
        			    }
        				suitable.setValues(injectedArgs);
        				return suitable;
        			}
        		}
            }
        }
    	return null;
	}
	
	private IRttiPropertyDescriptor[] autoInjectedProperties() {
	    String autoBinding = getAutoBindingMode();
        if (autoBinding.equals(DICON_VAL_AUTO_BINDING_AUTO)
                || autoBinding.equals(DICON_VAL_AUTO_BINDING_PROPERTY)) {
    	    IRtti rtti = (IRtti)getAdapter(IRtti.class);
    	    if(rtti != null) {
        	    List list = new ArrayList();
    	        IRttiPropertyDescriptor[] props = rtti.getProperties();
    	        for(int i = 0; i < props.length; i++) {
    	            String name = props[i].getName();
    	            if(props[i].getType().isInterface() && props[i].isWritable()) {
    		            if(!ModelUtils.hasPropertyElement(this, name)) {
				            ContainerElement container = getContainerElement();
				            IComponentKey key = 
				            	container.createComponentKey(props[i].getType());
				            IRtti inject = container.getComponent(key);
				    		props[i].setValue(inject);
				        	list.add(props[i]);
    		            }
    	            }
    	        }
    	        return (IRttiPropertyDescriptor[])list.toArray(
    	                new IRttiPropertyDescriptor[list.size()]);
    	    }
        }
    	return new IRttiPropertyDescriptor[0];
	}
	
    public Object getAdapter(Class adapter) {
        if(IComponentInfo.class.equals(adapter)) {
        	if(info == null) {
        		info = new IComponentInfo() {
	            	public IComponentKey[] getComponentKeys() {
	            		if(getDepth() == 2) {
		            		Set keys = createKeys();
		            	    return (IComponentKey[])keys.toArray(
		            	            new IComponentKey[keys.size()]);
	            		} else {
	            			return new IComponentKey[0];
	            		}	
	            	}
                    public IRttiConstructorDesctiptor getAutoInjectedConstructor() {
                    	return findAutoInjectedConstructor();
                    }
                    public IRttiPropertyDescriptor[] getAutoInjectedProperties() {
                    	return autoInjectedProperties();
                    }
	            };
        	}
        	return info;
        }
        return super.getAdapter(adapter);
    }
    
	public boolean isOGNL() {
	    if(isOuterInjection()) {
	        return false;
	    } else if(StringUtils.noneValue(getComponentClassName())) {
	        Object rtti = getAdapter(IRtti.class);
	        if(rtti != null) {
	            ComponentElement component = (ComponentElement)
	                ((IAdaptable)rtti).getAdapter(ComponentElement.class);
	            if(component != null) {
	                return component.isOGNL();
	            }
	        }
	        return true;
	    } else {
	        return super.isOGNL();
	    }
	}
	
	public String getDisplayName() {
	    StringBuffer buffer = new StringBuffer();
	    if(isOuterInjection() && StringUtils.noneValue(getComponentClassName())) {
	        buffer.append(
	                Kijimuna.getResourceString("dicon.model.ComponentElement.2"));
	    } else {
	        buffer.append(super.getDisplayName());
	    }
	    String name = getComponentName();
	    if(StringUtils.existValue(name)) {
	        buffer.append("<").append(name).append(">");
	    }
	    return buffer.toString();
	}
	
}
