/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.dicon;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.seasar.kijimuna.core.Kijimuna;
import org.seasar.kijimuna.core.dicon.model.ContainerElement;
import org.seasar.kijimuna.core.project.IFileProcessor;
import org.seasar.kijimuna.core.project.IProjectRecordChangeListener;
import org.seasar.kijimuna.core.util.ProjectUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class ModelManager implements Serializable {

	private transient IProject project;
	private String projectName;
	
	private Map containerMap = new HashMap();
    private Map relatedMap = new HashMap();
    private Set listeners = new HashSet();

	private transient IFileProcessor builder;
	private transient IFileProcessor validator;
    
    public void setProjectName(String projectName) {
    	this.projectName = projectName;
    	builder = new DiconBuilder();
    	validator = new DiconValidator();
    }
    
    public boolean isDirty() {
    	return relatedMap.size() != 0;
	}
    
    private IProject getProject() {
    	if(project == null) {
    		project = ProjectUtils.getProject(projectName);
    	}
    	return project;
    }
    
	private void addRelatedMapItem(IStorage key, IStorage entry) {
        Set item = null;
        String keyString = ProjectUtils.getPathString(key);
        item = (Set)relatedMap.get(keyString);
        if(item == null) {
            item = new HashSet(3);
        }
        if(entry != null) {
            item.add(ProjectUtils.getPathString(entry));
        }
        relatedMap.put(keyString, item);
	}

	public void validate(IProgressMonitor monitor) {
		validator.processProject(getProject(), monitor);
	}
	
    public void init(IProgressMonitor monitor) {
		builder.processProject(getProject(), monitor);
		validate(monitor);
    }
	
    public void addContainerAndRelatedFile(
            ContainerElement container, IStorage relatedFile) {
    	IStorage file = (IStorage)container.getAdapter(IStorage.class);
    	String keyString = ProjectUtils.getPathString(file);
    	synchronized(this) {
	        containerMap.put(keyString, container);
	        addRelatedMapItem(file, relatedFile);
    	}
    }
    
	public ContainerElement getContainer(IStorage storage, IProgressMonitor monitor) {
		String keyString = ProjectUtils.getPathString(storage);
	    ContainerElement ret = (ContainerElement)containerMap.get(keyString);
	    if(ret == null) {
	    	builder.process(getProject(), storage, monitor);
	        ret = (ContainerElement)containerMap.get(keyString);
	    }
	    return ret;
	}
    
    public ContainerElement[] getContainers(IProgressMonitor monitor) {
    	if(monitor != null) {
    		monitor.beginTask(Kijimuna.getResourceString("dicon.ModelManager.1"),
    				relatedMap.size());
    	}
        Set ret = new HashSet();
        for(Iterator it = relatedMap.keySet().iterator(); it.hasNext();) {
            String keyString = (String)it.next();
            ret.add(getContainer(
                    ProjectUtils.getStorage(getProject(), keyString), monitor));
            if(monitor != null) {
            	monitor.worked(1);
            }
        }
    	if(monitor != null) {
    		monitor.done();
    	}
        return (ContainerElement[])ret.toArray(new ContainerElement[ret.size()]);
    }
    
    public IFile[] getRelatedFiles(IStorage depend, boolean infiniti) {
        Set ret = new HashSet();
        for(Iterator it = relatedMap.entrySet().iterator(); it.hasNext();) {
            Map.Entry entry = (Map.Entry)it.next();
            Set item = (Set)entry.getValue();
            if(item.contains(ProjectUtils.getPathString(depend))) {
                String keyString = (String)entry.getKey();
                IStorage storage = ProjectUtils.getStorage(getProject(), keyString);
                if(storage instanceof IFile) {
                    ret.add(storage);
                }
                if(infiniti) {
	                IFile[] parents = getRelatedFiles(storage, true);
	                for(int i = 0; i < parents.length; i++) {
	                    ret.add(parents[i]);
	                }
                }
            }
        }
        return (IFile[])ret.toArray(new IFile[ret.size()]);
    }

    public List getRootContainers(IProgressMonitor monitor) {
    	if(monitor != null) {
    		monitor.beginTask(Kijimuna.getResourceString("dicon.ModelManager.2"),
    				relatedMap.size());
    	}
    	List ret = new ArrayList();
    	for(Iterator it = relatedMap.keySet().iterator(); it.hasNext();) {
    		String keyString = (String)it.next();
    		boolean isRoot = true;
    		for(Iterator kt = relatedMap.entrySet().iterator(); kt.hasNext();) {
    			Set set = (Set)((Map.Entry)kt.next()).getValue();
	    		if(set.contains(keyString)){
	    			isRoot = false;
	    			break;
	    		}
    		}
    		if(isRoot) {
    			ret.add(getContainer(
    			        ProjectUtils.getStorage(getProject(), keyString), monitor));
    		}
    		if(monitor != null) {
            	monitor.worked(1);
            }
    	}
    	if(monitor != null) {
    		monitor.done();
    	}
        return ret;
    }
    
    public void removeContainer(IStorage build) {
    	String keyString = ProjectUtils.getPathString(build);
        containerMap.remove(keyString);
        relatedMap.remove(keyString);
    }

    public void clearContainer() {
        containerMap.clear();
        relatedMap.clear();
    }
    
    public void fireRecordChanged() {
    	for(Iterator it = listeners.iterator(); it.hasNext();) {
    	    final IProjectRecordChangeListener listener =
    	        (IProjectRecordChangeListener)it.next();
    	    IWorkspaceRunnable runnable = new IWorkspaceRunnable() {
    	    	public void run(IProgressMonitor monitor) throws CoreException {
    	    	    listener.finishChanged();
    	    	}
    	    };
    	    try {
                getProject().getWorkspace().run(runnable, null);
            } catch (CoreException e) {
                Kijimuna.reportException(e);
            }
    	}
    }
    
    public void addRecordChangeListener(IProjectRecordChangeListener listener) {
    	listeners.add(listener);
	}
	
	public void removeRecordChangeListener(IProjectRecordChangeListener listener) {
		listeners.remove(listener);
	}
	
	public void prepareStoraging() {
		listeners.clear();
	}

	public void afterRestoring() {
    	builder = new DiconBuilder();
    	validator = new DiconValidator();
	}
}
