/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.dicon;

import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.Kijimuna;
import org.seasar.kijimuna.core.dicon.model.ContainerElement;
import org.seasar.kijimuna.core.dicon.model.DiconElement;
import org.seasar.kijimuna.core.dicon.validation.DiconValidationFactory;
import org.seasar.kijimuna.core.parser.Element;
import org.seasar.kijimuna.core.project.AbstractProcessor;
import org.seasar.kijimuna.core.project.IFileProcessor;
import org.seasar.kijimuna.core.util.FileUtils;
import org.seasar.kijimuna.core.util.MarkerUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class DiconValidator extends AbstractProcessor
	implements ConstCore {
    
    private static IValidation[] validations;
    private static IProjectValidation[] projectValidations;
    
    static {
        validations = DiconValidationFactory.createValidation();
        projectValidations = DiconValidationFactory.createProjectValidation();
    }

    //-------------------------------------------------
    
    public String getNatureID() {
        return ID_NATURE_DICON;
    }

    public void handleFileAdded(IFile addedFile, boolean fullBuild, IProgressMonitor monitor) {
        if(!fullBuild && FileUtils.isJavaFile(addedFile)) {
            fullValidate(addedFile.getProject(), monitor);
        } else {
            process(addedFile.getProject(), addedFile, monitor);
        }
    }
    
    public void handleFileRemoved(IFile removedFile, IProgressMonitor monitor) {
        relatedValidate(removedFile, monitor);
    }
    
    public void handleFileChanged(IFile changedFile, IProgressMonitor monitor) {
        process(changedFile.getProject(), changedFile, monitor);
        relatedValidate(changedFile, monitor);
    }
    
    public void handlePrepareFullProcess(IProject project, IProgressMonitor monitor) {
    }
	
    public void handleClassPassChanged(IProject project, IProgressMonitor monitor) {
        fullValidate(project, monitor);
    }

    public IFileProcessor getFileBuilder() {
        return this;
    }

    private void fullValidate(IProject project, IProgressMonitor monitor) {
        DiconNature nature = DiconNature.getInstance(project);
        if(nature != null) {
	        ContainerElement[] containers = nature.getModel().getContainers(monitor);
	        for(int i = 0; i < containers.length; i++) {
                IFile file = (IFile)containers[i].getAdapter(IFile.class);
                MarkerUtils.deleteMarker(file, ID_MARKER_DICONVALIDAION);
                validElement(containers[i], monitor);
	        }
        }
    }
    
    public void process(IProject project, IStorage storage, IProgressMonitor monitor) {
		if (FileUtils.isDiconFile(storage)) {
		    if(storage instanceof IFile) {
		        MarkerUtils.deleteMarker((IFile)storage, ID_MARKER_DICONVALIDAION);
		    }
	        DiconNature nature = DiconNature.getInstance(project);
            if(nature != null) {
                ContainerElement container = 
                    nature.getModel().getContainer(storage, monitor);
	            if(container != null) {
	                validElement(container, monitor);
	            }
            }
		}
    }

	public void handleFinish(IProject project, IProgressMonitor monitor) {
        try {
            project.deleteMarkers(ID_MARKER_DICONVALIDAION,
                    true, IResource.DEPTH_ZERO);
        } catch (CoreException e) {
        }
        for(int i = 0; i < projectValidations.length; i++) {
            try {
                projectValidations[i].validation(project);
            } catch(Exception e) {
                Kijimuna.reportException(e);
            }
        }
	}

    private void validElement(Element element, IProgressMonitor monitor) {
        if(element instanceof DiconElement) {
            ((DiconElement)element).setLocking(true);
        }
        for(int i = 0; i < validations.length; i++) {
            try {
                validations[i].validation(element);
            } catch(Exception e) {
                Kijimuna.reportException(e);
            }
        }
        List children = element.getChildren();
        for(Iterator it = children.iterator(); it.hasNext();) {
            Element child = (Element)it.next();
            validElement(child, monitor);
        }
        if(element instanceof DiconElement) {
            ((DiconElement)element).setLocking(false);
        }
    }
    
    private void relatedValidate(IFile file, IProgressMonitor monitor) {
		if (FileUtils.isDiconFile(file) || FileUtils.isJavaFile(file)) {
		    DiconNature nature = DiconNature.getInstance(file.getProject());
		    if(nature != null) {
		        IFile[] relateds = nature.getModel().getRelatedFiles(file, true);
		        for(int i = 0; i < relateds.length; i++) {
		            process(file.getProject(), relateds[i], monitor);
		        }
		    }
		}
    }
}
