/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.ui.dicon.provider.walker;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.resources.IResource;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.views.properties.IPropertySource;
import org.seasar.kijimuna.core.dicon.info.IAspectInfo;
import org.seasar.kijimuna.core.dicon.info.IComponentInfo;
import org.seasar.kijimuna.core.dicon.info.IComponentKey;
import org.seasar.kijimuna.core.dicon.info.IApplyMethodInfo;
import org.seasar.kijimuna.core.dicon.model.ArgElement;
import org.seasar.kijimuna.core.dicon.model.AspectElement;
import org.seasar.kijimuna.core.dicon.model.ComponentElement;
import org.seasar.kijimuna.core.dicon.model.ContainerElement;
import org.seasar.kijimuna.core.dicon.model.DestroyMethodElement;
import org.seasar.kijimuna.core.dicon.model.DiconElement;
import org.seasar.kijimuna.core.dicon.model.IncludeElement;
import org.seasar.kijimuna.core.dicon.model.InitMethodElement;
import org.seasar.kijimuna.core.dicon.model.PropertyElement;
import org.seasar.kijimuna.core.parser.Element;
import org.seasar.kijimuna.core.rtti.HasErrorRtti;
import org.seasar.kijimuna.core.rtti.IRtti;
import org.seasar.kijimuna.core.rtti.IRttiConstructorDesctiptor;
import org.seasar.kijimuna.core.rtti.IRttiMethodDesctiptor;
import org.seasar.kijimuna.core.rtti.IRttiPropertyDescriptor;
import org.seasar.kijimuna.core.util.ModelUtils;
import org.seasar.kijimuna.ui.ConstUI;
import org.seasar.kijimuna.ui.KijimunaUI;
import org.seasar.kijimuna.ui.dicon.provider.IContentWalker;
import org.seasar.kijimuna.ui.dicon.provider.IInternalContainer;
import org.seasar.kijimuna.ui.dicon.provider.property.ContentProperty;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class ContentItem 
		implements IInternalContainer, ConstUI {

    private IPropertySource source;
    private IContentWalker parent;
    private DiconElement element;
    private boolean isOutline;
    
    public ContentItem(
    		DiconElement element, IContentWalker parent, boolean isOutline) {
        this.element = element;
        this.parent = parent;
        this.isOutline = isOutline;
    }
    
    public Element getElement() {
        return element;
    }
    
    public ContainerElement getContainer() {
    	IProject project = (IProject)element.getAdapter(IProject.class);
    	IStorage storage = (IStorage)element.getAdapter(IStorage.class);
    	if((project != null) && (storage != null)) {
    		return ModelUtils.getContainer(project, storage);
    	} 
    	return null;
    }
    
    public Object getParent() {
        return parent;
    }
    
    public Object[] getChildren() {
        List children = element.getChildren();
        List ret = new ArrayList();
        IComponentInfo info = (IComponentInfo)
        	element.getAdapter(IComponentInfo.class);
        if(info != null) {
            IComponentKey[] keys = info.getComponentKeys();
            if(keys.length > 0) {
                ret.add(new ComponentKeyRoot(this, keys));
            }
            IRttiConstructorDesctiptor constructor = info.getAutoInjectedConstructor();
            if(constructor != null) {
                IRtti values[] = constructor.getValues();
                for(int i = 0; i < values.length; i++) {
                	ret.add(new AutoInjectedArgItem(this, constructor, i));
                }
            }
            IRttiPropertyDescriptor[] properties = info.getAutoInjectedProperties();
            for(int i = 0; i < properties.length; i++) {
                ret.add(new AutoInjectedPropertyItem(this, properties[i]));
            }
        } else {
	        IApplyMethodInfo methodInfo = (IApplyMethodInfo)element.getAdapter(IApplyMethodInfo.class);
	        if(methodInfo != null) {
	            IRttiMethodDesctiptor method = methodInfo.getAutoInjectedMethod();
	            if(method != null) {
	                IRtti values[] = method.getValues();
	                for(int i = 0; i < values.length; i++) {
	                    ret.add(new AutoInjectedArgItem(this, method, i));
	                }
	            }
	        } else {
	            IAspectInfo aspectInfo = (IAspectInfo)element.getAdapter(IAspectInfo.class);
		        if(aspectInfo != null) {
		            ret.add(new PointcutRoot(this, aspectInfo));
		        }
	        }
        }
        for(Iterator it = children.iterator(); it.hasNext();) {
            Object obj = it.next();
        	if(isOutline) {
        		if(obj instanceof DiconElement) {
            		ret.add(new ContentItem((DiconElement)obj, this, true));
        		} else if(obj instanceof IncludeElement) {
                	ret.add(new IncludeItem((IncludeElement)obj, this));
                }
            } else {
        		if(obj instanceof ComponentElement) {
        			ret.add(new ContentItem((ComponentElement)obj, this, false));
        		} else if(obj instanceof IncludeElement) {
        			ContainerElement container = ((IncludeElement)obj).getChildContainer();
        			ret.add(new ContentItem(container, this, false));
        		}
            }
        }
        return ret.toArray();
    }
    
    public String getDisplayName() {
   		return element.getDisplayName();
    }
    
    public Image getImage() {
        IRtti rtti = (IRtti)element.getAdapter(IRtti.class);
        if(element instanceof ArgElement) {
            if(rtti instanceof HasErrorRtti) {
                return KijimunaUI.getImage(IMAGE_ICON_ARG_NG);
            } else {
	            if(element.isOGNL()) {
	                return KijimunaUI.getImage(IMAGE_ICON_ARG_OGNL);
	            } else {
	                return KijimunaUI.getImage(IMAGE_ICON_ARG);
	            }
            }
        } else if(element instanceof AspectElement) {
            if(rtti instanceof HasErrorRtti) {
                return KijimunaUI.getImage(IMAGE_ICON_ASPECT_NG);
            } else {
	            if(element.isOGNL()) {
	                return KijimunaUI.getImage(IMAGE_ICON_ASPECT_OGNL);
	            } else {
	                return KijimunaUI.getImage(IMAGE_ICON_ASPECT);
	            }
            }
        } else if(element instanceof ComponentElement) {
            if(rtti instanceof HasErrorRtti) {
                return KijimunaUI.getImage(IMAGE_ICON_COMPONENT_NG);
            } else {
	            if(element.isOGNL()) {
	                return KijimunaUI.getImage(IMAGE_ICON_COMPONENT_OGNL);
	            } else {
	                return KijimunaUI.getImage(IMAGE_ICON_COMPONENT);
	            }
            }
        } else if(element instanceof ContainerElement) {
            return KijimunaUI.getImage(IMAGE_ICON_CONTAINER);
        } else if(element instanceof DestroyMethodElement) {
            if(rtti instanceof HasErrorRtti) {
                return KijimunaUI.getImage(IMAGE_ICON_DESTROYMETHOD_NG);
            } else {
	            if(element.isOGNL()) {
	                return KijimunaUI.getImage(IMAGE_ICON_DESTROYMETHOD_OGNL);
	            } else {
	                return KijimunaUI.getImage(IMAGE_ICON_DESTROYMETHOD);
	            }
            }
        } else if(element instanceof InitMethodElement) {
            if(rtti instanceof HasErrorRtti) {
                return KijimunaUI.getImage(IMAGE_ICON_INITMETHOD_NG);
            } else {
	            if(element.isOGNL()) {
	                	return KijimunaUI.getImage(IMAGE_ICON_INITMETHOD_OGNL);
	            } else {
	                	return KijimunaUI.getImage(IMAGE_ICON_INITMETHOD);
	            }
            }
        } else if(element instanceof PropertyElement) {
            if(rtti instanceof HasErrorRtti) {
                return KijimunaUI.getImage(IMAGE_ICON_PROPERTY_NG);
            } else {
	            if(element.isOGNL()) {
	                return KijimunaUI.getImage(IMAGE_ICON_PROPERTY_OGNL);
	            } else {
	                return KijimunaUI.getImage(IMAGE_ICON_PROPERTY);
	            }
            }
        }
        return null;
    }

    public Object getAdapter(Class adapter) {
        if(adapter.equals(IPropertySource.class)) {
            if(source == null) {
                source = new ContentProperty(element);
            }
            return source;
        } else if(adapter.equals(IResource.class)) {
        	return (IResource)element.getAdapter(IResource.class);
        } else if(Element.class.equals(adapter)) {
            return element;
        }
        return element.getAdapter(adapter);
    }
}
