/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.ui.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jdt.internal.ui.javaeditor.EditorUtility;
import org.eclipse.jdt.internal.ui.javaeditor.JarEntryEditorInput;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;
import org.seasar.kijimuna.core.dicon.model.ComponentElement;
import org.seasar.kijimuna.core.dicon.model.ContainerElement;
import org.seasar.kijimuna.core.parser.Element;
import org.seasar.kijimuna.core.rtti.IRtti;
import org.seasar.kijimuna.core.util.StringUtils;
import org.seasar.kijimuna.ui.ConstUI;
import org.seasar.kijimuna.ui.KijimunaUI;
import org.seasar.kijimuna.ui.dicon.provider.IExternalContainer;
import org.seasar.kijimuna.ui.dicon.provider.IInternalContainer;
import org.seasar.kijimuna.ui.dicon.provider.walker.ComponentKeyItem;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class WorkbenchUtils implements ConstUI {
    
    public static IWorkbenchPage getWorkbenchPage() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		if(workbench != null) {
			IWorkbenchWindow window =workbench.getActiveWorkbenchWindow();
			if(window != null) {
				return window.getActivePage();
			}
		}
		return null;
    }

    public static IStorage getInputResource(IEditorPart part) {
        IEditorInput input = part.getEditorInput();
        if(input instanceof IFileEditorInput) {
            return ((IFileEditorInput)input).getFile();
        } else if(input instanceof IStorageEditorInput) {
            try {
                return ((IStorageEditorInput)input).getStorage();
            } catch (CoreException e) {
	            KijimunaUI.reportException(e);
            }
        }
        return null;
    }

    private static IEditorPart openEditor(IStorage storage, String editorID) {
    	if(storage != null) {
	        IWorkbenchPage page = getWorkbenchPage();
	        if(page != null) {
		        try {
		            if(storage instanceof IFile) {
			            if(editorID != null) {
			            	IEditorInput input = new FileEditorInput((IFile)storage);
			                return page.openEditor(input, editorID);
			            } else {
			                return IDE.openEditor(page, (IFile)storage);
			            }
		            } else {
		            	IEditorInput input = new JarEntryEditorInput(storage); 
			            if(editorID != null) {
			                return page.openEditor(input, editorID);
			            } else {
			                return IDE.openEditor(page, input, null);
			            }
		            }
		        } catch (CoreException e) {
		            KijimunaUI.reportException(e);
		        }
	        }
    	}
        return null;
    }
    
    public static IEditorPart openDefaultEditor(IStorage storage) {
       return openEditor(storage, null);
    }
    
    public static IEditorPart openDiconEditor(IStorage storage) {
        return openEditor(storage, ID_EDITOR_DICON);
    }
    
    public static IEditorPart openJavaEditor(IRtti rtti) {
    	try {
			return EditorUtility.openInEditor(rtti.getType());
		} catch (Exception e) {
			KijimunaUI.reportException(e);
			return null;
		}
    }
    
    public static IEditorPart showSource(IEditorPart part, Object obj) {
        if(obj instanceof ComponentKeyItem) {
            IRtti java = (IRtti)((ComponentKeyItem)obj).getAdapter(IRtti.class);
            if(java != null) {
                return openJavaEditor(java);
            }
        } else if(obj instanceof IExternalContainer) {
			ContainerElement container = 
				((IExternalContainer)obj).getExternalContainer();
			if(container != null) {
				return openDiconEditor((IStorage)container.getAdapter(IStorage.class));
			}
		} else if(obj instanceof IInternalContainer) {
		    Element element = ((IInternalContainer)obj).getElement();
		    IStorage file = (IStorage)element.getAdapter(IStorage.class);
			if(file != null) {
			    IStorage source = null;
	            if(part != null) {
	                source = WorkbenchUtils.getInputResource(part);
	            }
	            if(!file.equals(source)) {
	        		int lineNumber = element.getStartLine();
	        		IEditorPart editor = WorkbenchUtils.openDiconEditor(file);
	        		WorkbenchUtils.moveLine(editor, lineNumber);
	                return editor;
	            } else {		    
			        if(element.getElementName().equals(DICON_TAG_COMPONENT) &&
			                StringUtils.existValue(((ComponentElement)
			                        element).getComponentClassName())) {
			            IRtti java = (IRtti)element.getAdapter(IRtti.class);
			            if(java != null) {
		                    return openJavaEditor(java);
			            }
			        } else {
			            IRtti rtti = (IRtti)element.getAdapter(IRtti.class);
			            if(rtti != null) {
				            ComponentElement injected = 
				                (ComponentElement)rtti.getAdapter(ComponentElement.class);
					        if(injected != null) {
					            IStorage injectedFile = (IStorage)injected.getAdapter(IStorage.class);
					            IEditorPart editor;
					            if(injectedFile != null) {
					                if(!injectedFile.equals(source)) {
						                editor = WorkbenchUtils.openDiconEditor(injectedFile);
						            } else {
						                editor = part;
						            }
					                WorkbenchUtils.moveLine(editor, injected.getStartLine());
					                return editor;
					            }
					        }
			            }
			        }
	            }
			}
		}
        return null;
    }
    
    public static void moveLine(IEditorPart part, int lineNumber) {
        if((part != null) && (lineNumber > 0)) {
            try {
                IStorage file = getInputResource(part); 
	            if((file != null) && (file instanceof IFile)) {
					IMarker marker = ((IFile)file).createMarker(IMarker.TEXT);
					marker.setAttribute(IMarker.LINE_NUMBER, lineNumber);
					IDE.gotoMarker(part, marker);
					marker.delete();
	            }
	        } catch (CoreException e) {
	            KijimunaUI.reportException(e);
	        }
        }
    }
    
    public static IWorkbenchPart getWorkbenchPart() {
    	IWorkbenchPage page = getWorkbenchPage();
    	if(page != null) {
    		return page.getActivePart();
    	} else {
    		return null;
    	}
    }
    
    public static IEditorPart getActiveEditor() {
		IWorkbenchPage page = WorkbenchUtils.getWorkbenchPage();
		if(page != null) {
			return  page.getActiveEditor();
		}
		return null;
    }
    
    public static IProject getCurrentProject(ISelection selection) {
    	if (selection instanceof IStructuredSelection) {
    		for(Iterator it = ((IStructuredSelection)selection).iterator(); it.hasNext();) {
    			Object obj = it.next();
    			if (obj instanceof IAdaptable) {
    				IAdaptable adaptable = (IAdaptable)obj;
    				IResource resource = (IResource)adaptable.getAdapter(IResource.class);
    				if(resource != null) {
    					return resource.getProject();
    				}
    			}
    		}
    	}
    	return null;
    }
    
    public static Object getFirstSelectedElement(ISelection selection) {
	    if(selection instanceof IStructuredSelection) {
	        return ((IStructuredSelection)selection).getFirstElement();
	    }
	    return null;
    }
    
    public static String[] getAllWorkbenchEncodings() {
        List list = new ArrayList();
        for(int i = 0; i < ECLIPSE_ENCODINGS.length; i++) {
            list.add(ECLIPSE_ENCODINGS[i]);
        }
		String defaultEnc = System.getProperty("file.encoding", "UTF-8");
        if(!list.contains(defaultEnc)) {
            list.add(defaultEnc);
        }
        String eclipseEnc = getWorkbenchEncoding();
        if(StringUtils.existValue(eclipseEnc) && !list.contains(eclipseEnc)) {
            list.add(eclipseEnc);
        }
        Collections.sort(list);
        return (String[])list.toArray(new String[list.size()]);
    }
    
    public static String getWorkbenchEncoding() {
        String enc = ResourcesPlugin.getPlugin().getPluginPreferences()
			.getString(ResourcesPlugin.PREF_ENCODING);
        if(StringUtils.noneValue(enc)) {
            enc = System.getProperty("file.encoding", "UTF-8");
        }
        return enc;
    }
}
