/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.ui.preference;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.PropertyPage;
import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.Kijimuna;
import org.seasar.kijimuna.core.dicon.validation.MarkerSetting;
import org.seasar.kijimuna.core.util.ProjectUtils;
import org.seasar.kijimuna.ui.KijimunaUI;
import org.seasar.kijimuna.ui.preference.design.ProjectPropertyGUI;
import org.seasar.kijimuna.ui.util.WidgetUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class ProjectPreferencePage extends PropertyPage implements ConstCore {

	private Button natureCheck;
	private TabFolder tabFolder;
	private Combo xmlErrorCombo;
	private Combo xmlWarningCombo;
	private Combo nullInjectionCombo;
	private Combo autoInjectionCombo;
	private Combo javaFetalCombo;
	private Combo diconFetalCombo;
	private Combo diconProblemCombo;

	private IProject getProject() {
		return (IProject)getElement();
	}

	protected Control createContents(Composite parent) {
		ProjectPropertyGUI view = new ProjectPropertyGUI(parent, SWT.NULL);
		natureCheck = view.getNatureCheck();
		tabFolder = view.getTabFolder();
		xmlErrorCombo = view.getXmlErrorCombo();
		xmlWarningCombo = view.getXmlWarningCombo();
		nullInjectionCombo = view.getNullInjectionCombo();
		autoInjectionCombo = view.getAutoInjectionCombo();
		javaFetalCombo = view.getJavaFetalCombo();
		diconFetalCombo = view.getDiconFetalCombo();
		diconProblemCombo = view.getDiconProblemCombo();
		IProject project = getProject();
		try {
			boolean flag = project.hasNature(ID_NATURE_DICON);
			natureCheck.setSelection(flag);
			tabFolder.setEnabled(flag);
			settingCombos(false);
		} catch(CoreException e) {
			KijimunaUI.reportException(e);
		}
		return view;
	}

	private void settingCombos(boolean isDefault) {
	    IProject project = getProject();
		WidgetUtils.setDiconMarkerSettingCombo(
		        project, xmlErrorCombo, MARKER_CATEGORY_XML_ERROR, isDefault);
		WidgetUtils.setDiconMarkerSettingCombo(
		        project, xmlWarningCombo, MARKER_CATEGORY_XML_WARNING, isDefault);
		WidgetUtils.setDiconMarkerSettingCombo(
		        project, nullInjectionCombo, MARKER_CATEGORY_NULL_INJECTION, isDefault);
		WidgetUtils.setDiconMarkerSettingCombo(
		        project, autoInjectionCombo, MARKER_CATEGORY_AUTO_INJECTION, isDefault);
		WidgetUtils.setDiconMarkerSettingCombo(
		        project, javaFetalCombo, MARKER_CATEGORY_JAVA_FETAL, isDefault);
		WidgetUtils.setDiconMarkerSettingCombo(
		        project, diconFetalCombo, MARKER_CATEGORY_DICON_FETAL, isDefault);
		WidgetUtils.setDiconMarkerSettingCombo(
		        project, diconProblemCombo, MARKER_CATEGORY_DICON_PROBLEM, isDefault);
	}
	
	public boolean performOk() {
		try {
			if(natureCheck.getSelection()) {
			    final IProject project = getProject();
				boolean alreadyHasNature = ProjectUtils.hasNature(
				        project, ID_NATURE_DICON); 
				if(!alreadyHasNature) {
				    ProjectUtils.addNature(project, ID_NATURE_DICON);
				}
				MarkerSetting.setDiconMarkerPreference(project, 
				        MARKER_CATEGORY_XML_ERROR, 
				        xmlErrorCombo.getSelectionIndex());
				MarkerSetting.setDiconMarkerPreference(project, 
				        MARKER_CATEGORY_XML_WARNING, 
				        xmlWarningCombo.getSelectionIndex());
				MarkerSetting.setDiconMarkerPreference(project, 
				        MARKER_CATEGORY_NULL_INJECTION,
				        nullInjectionCombo.getSelectionIndex());
				MarkerSetting.setDiconMarkerPreference(project,
				        MARKER_CATEGORY_AUTO_INJECTION,
				        autoInjectionCombo.getSelectionIndex());
				MarkerSetting.setDiconMarkerPreference(project,
				        MARKER_CATEGORY_JAVA_FETAL,
				        javaFetalCombo.getSelectionIndex());
				MarkerSetting.setDiconMarkerPreference(project,
				        MARKER_CATEGORY_DICON_FETAL, 
				        diconFetalCombo.getSelectionIndex());
				MarkerSetting.setDiconMarkerPreference(project,
				        MARKER_CATEGORY_DICON_PROBLEM, 
				        diconProblemCombo.getSelectionIndex());
				if(alreadyHasNature) {
					IWorkbenchWindow window = 
						PlatformUI.getWorkbench().getActiveWorkbenchWindow();
					try {
						window.run(true, true, new IRunnableWithProgress() {
							public void run(IProgressMonitor monitor)
									throws InvocationTargetException, InterruptedException {
							    Kijimuna.getProjectRecorder().cleanup(project, monitor);
							}
						});
					} catch (InvocationTargetException e) {
					} catch (InterruptedException e) {
					}
				}
			} else {
			    ProjectUtils.removeNature(getProject(), ID_NATURE_DICON);
			}
		} catch (CoreException e) {
			KijimunaUI.reportException(e);
			return false;
		}
		return true;
	}

    protected void performDefaults() {
		if(natureCheck.getSelection()) {
			settingCombos(true);
		}
    }
}