/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.rtti;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IStorage;
import org.eclipse.core.resources.IResource;
import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IField;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.Signature;
import org.seasar.kijimuna.core.util.ProjectUtils;

/**
 * @author Masataka Kurihara (Gluegent,Inc.)
 */
class DefaultRtti implements IRtti {

	private static final Map WRAPPERS;
	static {
		WRAPPERS = new HashMap();
		WRAPPERS.put("java.lang.Boolean", "boolean");
		WRAPPERS.put("java.lang.Byte", "byte");
		WRAPPERS.put("java.lang.Character", "char");
		WRAPPERS.put("java.lang.Double", "double");
		WRAPPERS.put("java.lang.Float", "float");
		WRAPPERS.put("java.lang.Integer", "int");
		WRAPPERS.put("java.lang.Long", "long");
		WRAPPERS.put("java.lang.Short", "short");
		WRAPPERS.put("java.lang.Void", "void");
	}
	
	private transient IType thisType;
	
	private RttiLoader loader;
	private String projectName;
	private String qualifiedName;
	private boolean primitive;
	private int arrayDepth;
	private IRtti arrayItem;
	private boolean autoConvert;

	DefaultRtti(RttiLoader loader, IType thisType, String qualifiedName,
			boolean primitive, int arrayDepth, IRtti arrayItem, boolean autoConvert)
			throws RttiClassNotFoundException {
		if((loader == null) || (thisType == null)) {
			throw new RttiClassNotFoundException(qualifiedName);
		}
		this.loader = loader;
		this.thisType = thisType;
		this.qualifiedName = qualifiedName;
		this.primitive = primitive;
		this.arrayDepth = arrayDepth;
		this.arrayItem = arrayItem;
		this.autoConvert = autoConvert;
		projectName = loader.getProject().getElementName();
	}
	
	public IType getType() {
	    if(thisType == null) {
	        IJavaProject project = ProjectUtils.getJavaProject(projectName);
	        try {
                thisType = project.findType(qualifiedName);
            } catch (JavaModelException e) {
            }
	    }
	    return thisType;
	}
	
	public RttiLoader getRttiLoader() {
	    return loader;
	}

	public boolean isPrimitive() {
		return primitive;
	}

	public boolean isArray() {
		return arrayDepth > 0;
	}

	private boolean isTypeAvailable() {
		return getType() != null;
	}

	public boolean isInterface() {
		if (isTypeAvailable()) {
			try {
				return getType().isInterface();
			} catch (JavaModelException ignore) {
			}
		}
		return false;
	}

	public boolean isFinal() {
		if (isTypeAvailable()) {
			try {
				return Flags.isFinal(getType().getFlags());
			} catch (JavaModelException ignore) {
			}
		}
		return false;
	}
	
	public String getQualifiedName() {
		if (isArray()) {
			StringBuffer buffer = new StringBuffer(arrayItem.getQualifiedName());
			for (int i = 0; i < arrayDepth; i++) {
				buffer.append("[]");
			}
			return buffer.toString();
		}
		if (isPrimitive()) {
			return (String) WRAPPERS.get(qualifiedName);
		}
		return qualifiedName;
	}

	public String getShortName() {
		String qname = getQualifiedName();
		int pos = qname.lastIndexOf('.');
		if (pos != -1) {
			return qname.substring(pos + 1);
		} else {
			return qname;
		}
	}

	public String getWrapperName() {
		return qualifiedName;
	}
	
	public boolean equals(Object test) {
		if((test != null) && (test instanceof IRtti)) {
			IRtti testRtti = (IRtti) test;
			if (autoConvert && (isPrimitive() || testRtti.isPrimitive())) {
				if (getWrapperName().equals(testRtti.getWrapperName())) {
					return true;
				}
			}
			if (isArray()) {
				if (testRtti.isArray()) {
					IRtti thisArrayItem = getArrayItemClass();
					IRtti testArrayItem = testRtti.getArrayItemClass();
					return thisArrayItem.equals(testArrayItem);
				}
				return false;
			}
			String qname = testRtti.getQualifiedName();
			if (getQualifiedName().equals(qname)) {
				return true;
			}
		}
		return false;
	}

	public IRtti getArrayItemClass() {
		if (isArray()) {
			return arrayItem;
		} else {
			return null;
		}
	}

	public IRtti getSuperClass() {
		if (isTypeAvailable() && !isInterface()
				&& !getQualifiedName().equals("java.lang.Object")) {
			try {
				String superClassName = getType().getSuperclassName();
				if (superClassName == null) {
					superClassName = "java.lang.Object";
				}
				return loader.loadRtti(superClassName);
			} catch (Exception ignore) {
			}
		}
		return null;
	}

	public IRtti[] getInterfaces() {
		if (isTypeAvailable()) {
			try {
				String[] interfaces = getType().getSuperInterfaceNames();
				IRtti[] ret = new IRtti[interfaces.length];
				for (int i = 0; i < interfaces.length; i++) {
					ret[i] = loader.loadRtti(interfaces[i]);
				}
				return ret;
			} catch (Exception ignore) {
			}
		}
		return new IRtti[0];
	}

	private boolean isWideningPrimitiveConversion(IRtti testRtti) {
	    String thisQName = getQualifiedName();
	    String testQName = testRtti.getQualifiedName();
	    if(autoConvert) {
		    if(WRAPPERS.containsKey(thisQName)) {	
		    	thisQName = WRAPPERS.get(thisQName).toString();
		    }
		    if(WRAPPERS.containsKey(testQName)) {	
		    	testQName = WRAPPERS.get(testQName).toString();
		    }
	    }
	    if(testQName.equals("byte")) {
	        return "short,int,long,float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("short")) {
	        return "int,long,float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("char")) {
	        return "int,long,float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("int")) {
	        return "long,float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("long")) {
	        return "float,double".indexOf(thisQName) != -1;
	    } else if(testQName.equals("float")) {
	        return "double".indexOf(thisQName) != -1;
	    } else {
	        return false;
	    }
	}

	private boolean isWideningArrayConversion(IRtti testRtti) {
	    String thisQName = getQualifiedName();
	    if(thisQName.equals("java.lang.Object") ||
	            thisQName.equals("java.lang.Cloneable") ||
	            thisQName.equals("java.io.Serializable")) {
	        return true;
	    } else {
	        IRtti thisArray = getArrayItemClass();
	        IRtti testArray = testRtti.getArrayItemClass();
	        return (thisArray != null) && (testArray != null) &&
	        	thisArray.isAssignableFrom(testArray);
	    }
	}

	public boolean isAssignableFrom(IRtti testRtti) {
		if(testRtti == null) {
		    return true;
			//testRtti = loader.loadRtti("java.lang.Object", root);
		}
		if (equals(testRtti)) {
			return true;
		} else {
			if (isPrimitive() || testRtti.isPrimitive()) {
			    // widening primitive conversion
			    return isWideningPrimitiveConversion(testRtti);
			} else if(testRtti.isArray()) {
			    return isWideningArrayConversion(testRtti);
			} else {
				if (isInterface()) {
					IRtti[] superInterfaces = testRtti.getInterfaces();
					for (int i = 0; i < superInterfaces.length; i++) {
						if (equals(superInterfaces[i])) {
							return true;
						}
					}
				}
				if (!testRtti.isInterface()) {
					IRtti superClass = testRtti.getSuperClass();
					if (superClass == null) {
						return false;
					} else {
						return isAssignableFrom(superClass);
					}
				}
			}
		}

		return false;
	}

	public IRtti getField(String field, boolean staticAccess)
			throws RttiNoSuchFieldException {
		try {
		    if(field.equals("class") && staticAccess) {
		        return loader.loadRtti("java.lang.Class");
		    }
		    if (isArray() && "length".equals(field)) {
				return loader.loadRtti("int");
			}
			if (isTypeAvailable()) {
				IField value = getType().getField(field);
				if (value.exists() && isPublicMember(value)) {
					if (staticAccess) {
						int flags = value.getFlags();
						if (!Flags.isStatic(flags)) {
							throw new RttiNoSuchFieldException(
									getQualifiedName(), field);
						}
					}
					String typeSignature = Signature.toString(value
							.getTypeSignature());
					return loader.loadRtti(typeSignature);
				} else {
					IRtti[] interfaces = getInterfaces();
					for (int i = 0; i < interfaces.length; i++) {
						try {
							return interfaces[i].getField(field, staticAccess);
						} catch (RttiNoSuchFieldException ignore) {
						}
					}
					IRtti superClass = getSuperClass();
					if (superClass != null) {
						return superClass.getField(field, staticAccess);
					}
				}
			}
		} catch (Exception ignore) {
		}
		throw new RttiNoSuchFieldException(getQualifiedName(), field);
	}

	public IRtti invoke(String methodName, IRtti[] args)
			throws RttiNoSuchMethodException {
		try {
			IMethod method = getMethod(methodName, args);
			if (method != null) {
				String returnType = Signature.toString(method.getReturnType());
				return loader.loadRtti(returnType);
			}
		} catch (Exception ignore) {
		}
		throw new RttiNoSuchMethodException(
		        getQualifiedName(), methodName, args);
	}

	public boolean hasConstructor(IRtti[] args) {
		String shortName = getShortName();
		boolean ret = hasMethod(shortName, args);
		if(!ret) {
			if((args == null) || (args.length == 0)) {
				boolean hasDefault = true;
				try {
					IMethod[] methods = getType().getMethods();
					for (int i = 0; i < methods.length; i++) {
						if (isPublicMember(methods[i]) && 
						        (methods[i].getElementName().equals(shortName))) {
							hasDefault = false;
							break;
						}
					}
				} catch (JavaModelException ignore) {
				}
				ret = hasDefault;
			}
		}
		return ret;
	}

	public boolean hasMethod(String methodName, IRtti[] args) {
		IMethod method = getMethod(methodName, args);
		return method != null;
	}

	public boolean hasSetterMethod(String property, IRtti arg) {
		IMethod setter = getSetterMethod(property, arg);
		return setter != null;
	}

	public boolean hasGetterMethod(String property) {
		IMethod getter = getGetterMethod(property);
		return getter != null;
	}

	public IRtti getProperty(String property)
			throws RttiNoSuchPropertyException {
		IMethod getter = getGetterMethod(property);
		if (getter != null) {
			try {
				String propertyType = Signature
						.toString(getter.getReturnType());
				return loader.loadRtti(propertyType);
			} catch (Exception ignore) {
			}
		}
		throw new RttiNoSuchPropertyException(getQualifiedName(), property);
	}

	public IRttiMethod[] getConstructors() {
	    return getMethods(null, true);
	}

	public IRttiMethod[] getMethods(Pattern pattern) {
	    return getMethods(pattern, false);
	}

    private IRttiMethod[] getMethods(Pattern pattern, boolean isConstructor) {
		if(isTypeAvailable()) {
			try {
				IMethod[] methods = getType().getMethods();
				List list = new ArrayList();
				for(int i = 0; i < methods.length; i++) {
				    if(isPublicMember(methods[i])) {
				        String methodName = methods[i].getElementName();
				        if(isConstructor) {
				            if(!methodName.equals(getShortName())) {
				                continue;
				            }
				        } else {
				            if(methodName.equals(getShortName()) ||
				                    !pattern.matcher(methodName).matches()) {
				                continue;
				            }
				        }
						String[] argTypes = methods[i].getParameterTypes();
						IRtti[] rttiArgs = new IRtti[argTypes.length];
						for (int k = 0; k < argTypes.length; k++) {
							String resolvedName = Signature.toString(argTypes[k]);
							try {
								rttiArgs[k] = loader.loadRtti(resolvedName);
							} catch (RttiClassNotFoundException e) {
								rttiArgs[k] = null;
							}
						}
						String retType = methods[i].getReturnType();
						String resolvedRet = Signature.toString(retType);
						IRtti retRtti = null;
						try {
							retRtti = loader.loadRtti(resolvedRet);
						} catch(RttiClassNotFoundException ignore) {
						}
						list.add(new DefaultRttiMethod(this,
						        methods[i], methodName, rttiArgs, retRtti));
					}
				}
				Collections.sort(list);
				return (IRttiMethod[])list.toArray(new IRttiMethod[list.size()]);
			} catch (JavaModelException ignore) {
			}
		}
		return null;	
    }	
    
	public String toString() {
		return getQualifiedName();
	}
	
	public IRttiProperty[] getSetterProperties() {
		if(isTypeAvailable()) {
			try {
				IMethod[] methods = getType().getMethods();
				List list = new ArrayList();
				for(int i = 0; i < methods.length; i++) {
				    if(isPublicMember(methods[i])) {
					    String methodName = methods[i].getElementName();
						String propertyName = getSetterPropertyName(methodName);
						if(propertyName != null) {
						    String[] argTypes = methods[i].getParameterTypes();
							String retType = Signature.toString(methods[i].getReturnType());
							if("void".equals(retType) && (argTypes.length == 1)) {
							    IRtti rttiArg = null;
								String resolvedName = Signature.toString(argTypes[0]);
								try {
									rttiArg = loader.loadRtti(resolvedName);
								} catch (RttiClassNotFoundException e) {
								}
								list.add(new DefaultRttiProperty(
										this, propertyName, rttiArg));
							}
						}
				    }
				}
				IRtti parent = getSuperClass();
				if(parent != null) {
				    IRttiProperty[] props = parent.getSetterProperties();
				    if(props != null) {
					    for(int i = 0; i < props.length; i++) {
					        list.add(props[i]);
					    }
				    }
				}
				return (IRttiProperty[])list.toArray(new IRttiProperty[list.size()]);
			} catch (JavaModelException ignore) {
			}
		}
		return null;	
	}

    public Object getAdapter(Class adapter) {
        if(IStorage.class.equals(adapter)) {
            if(isTypeAvailable()) {
                IType type = getType();
                if(type != null) {
                    IResource resource = type.getResource();
                    if(resource != null) {
                        return resource;
                    }
                }
            }
        } else if(IRtti.class.equals(adapter)){
            return this;
        }
        return null;
    }

	//-------------------------------------------------------------------------------------------

	private String getSetterPropertyName(String methodName) {
	    if(methodName.startsWith("set")) {
	        String propertyName = methodName.substring(3);
	        if((propertyName.length() == 0) || 
	                Character.isLowerCase(propertyName.charAt(0))) {
	            // method name is "set".
	            return null;
	        }
	        if((propertyName.length() == 1) || 
	                (Character.isLowerCase(propertyName.charAt(1)))) {
	            char[] chars = propertyName.toCharArray();
	            chars[0] = Character.toLowerCase(chars[0]);
	            propertyName = new String(chars);
	        }
		    return propertyName;
	    }
	    return null;
	}
	
	private String getAccessorBaseName(String property) {
		char[] chars = property.toCharArray();
		chars[0] = Character.toUpperCase(chars[0]);
		return new String(chars);
	}

	private boolean isPublicMember(IMember member) {
		if(!isInterface()) {
		    try {
                int flags = member.getFlags();
                if(!Flags.isPublic(flags)) {
                    return false;
                }
            } catch (JavaModelException e) {
            }
		}
		return true;
	}
	
	private IMethod getGetterMethod(String property) {
		String methodBaseName = getAccessorBaseName(property);
		IMethod method = getMethod("get" + methodBaseName);
		if (method == null) {
			method = getMethod("is" + methodBaseName);
		}
	    return method;
	}

	private IMethod getSetterMethod(String property, IRtti test) {
		String setterName = "set" + getAccessorBaseName(property);
		IRtti[] arg = null;
		if(test != null) {
			arg = new IRtti[] { test };
		}
		IMethod method = getMethod(setterName, arg);
	    return method;
	}

	private IMethod getMethod(String methodName) {
		if (isTypeAvailable()) {
			IMethod method = getType().getMethod(methodName, new String[0]);
			if (!method.exists()) {
				DefaultRtti superClass = (DefaultRtti) getSuperClass();
				if (superClass != null) {
					return superClass.getMethod(methodName);
				}
				return null;
			} else if(isPublicMember(method)) {
			    return method;
			}
		}
		return null;
	}

	private IMethod getMethod(String methodName, IRtti[] args) {
		if ((args == null) || (args.length == 0)) {
			return getMethod(methodName);
		} else {
			if (isTypeAvailable()) {
				try {
					IMethod[] methods = getType().getMethods();
					for (int i = 0; i < methods.length; i++) {
						if (isPublicMember(methods[i]) &&
						        methods[i].getElementName().equals(methodName)) {
							String[] argTypes = methods[i].getParameterTypes();
							if (args.length == argTypes.length) {
								boolean match = true;
								for (int k = 0; k < argTypes.length; k++) {
									String resolvedName = Signature
											.toString(argTypes[k]);
									IRtti argRtti = null;
									try {
										argRtti = loader.loadRtti(resolvedName);
										if (!argRtti.isAssignableFrom(args[k])) {
											match = false;
											break;
										}
									} catch (RttiClassNotFoundException e) {
										match = false;
										break;
									}
								}
								if (match) {
									return methods[i];
								}
							}
						}
					}
				} catch (JavaModelException ignore) {
				}
				DefaultRtti superClass = (DefaultRtti) getSuperClass();
				if (superClass != null) {
					return superClass.getMethod(methodName, args);
				}
			}
			return null;
		}
	}

}