/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.parser;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.IProgressMonitor;
import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.Kijimuna;
import org.seasar.kijimuna.core.util.MarkerUtils;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class DocumentHandler extends DefaultHandler implements ConstCore {

	private IProject project;
	private IStorage storage;
	private IParseStopper stopper;
	private IProgressMonitor monitor;
	private Stack stack;
	private Locator locator;
	private Map dtdMap;
	private ElementFactory factory;
	private Element result;
	private String markerType;
	private int errorSeverity;
	private int warningSeverity;

	public DocumentHandler(ElementFactory factory) {
	    this(factory, null, MARKER_SEVERITY_IGNORE, MARKER_SEVERITY_IGNORE);
	}

	public DocumentHandler(ElementFactory factory, 
	        String markerType, int errorSeverity, int warningSeverity) {
		stack = new Stack();
		setDocumentLocator(locator);
		dtdMap = new HashMap();
		this.factory = factory;
		this.markerType = markerType;
		this.errorSeverity = errorSeverity;
		this.warningSeverity = warningSeverity;
	}

	public void setStorage(IProject project, IStorage storage) {
		this.project = project;
		this.storage = storage;
	}

	public void setProgressMonitor(IProgressMonitor monitor) {
		this.monitor = monitor;
	}

	public void setParseStopper(IParseStopper stopper) {
		this.stopper = stopper;
	}

	public void putDtdPath(String publicId, String path) {
		dtdMap.put(publicId, path);
	}
	
	public void setDocumentLocator(Locator locator) {
		this.locator = locator;
	}

	public InputSource resolveEntity(String publicId, String systemId)
			throws SAXException {
		if (publicId != null) {
			String dtdPath = (String) dtdMap.get(publicId);
			if (dtdPath != null) {
				try {
					return new InputSource(Kijimuna.getEntry(dtdPath)
							.openStream());
				} catch (IOException ignore) {
				}
			}
		}
		return null;
	}

	public void startDocument() throws SAXException {
		stopTest();
		if(factory == null) {
			factory = new ElementFactory();
		}
	}

	public void startElement(String namespaceURI, String localName,
			String qName, Attributes attributes) {
		stopTest();
		if (monitor != null) {
			monitor.worked(1);
		}
		int depth = stack.size() + 1;
		Map property = new HashMap();
		for (int i = 0; i < attributes.getLength(); i++) {
			property.put(attributes.getQName(i), attributes.getValue(i));
		}
		Element element = factory.createElement(project, storage, qName);
		element.setStartLocation(depth, locator.getLineNumber(), locator.getColumnNumber());
		element.setAttributes(property);
		if (depth == 1) {
		    result = element;
			element.setRootElement(element);
		} else {
			element.setRootElement(result);
			Element parent = (Element) stack.peek();
			element.setParent(parent);
		}
		stack.push(element);
	}

	public void characters(char[] buffer, int start, int length) {
		stopTest();
		if (monitor != null) {
			monitor.worked(1);
		}
		Element element = (Element)stack.peek();
		StringBuffer body = new StringBuffer();
		String old = element.getBody();
		if(old != null) {
			body.append(old);
		}
		body.append(new String(buffer, start, length));
		element.setBody(body.toString());
	}

	public void endElement(String namespaceURI, String localName, String qName) {
		stopTest();
		if (monitor != null) {
			monitor.worked(1);
		}
		Element element = (Element) stack.pop();
		element.setEndLocation(locator.getLineNumber(), locator.getColumnNumber());
	}

	public void ignorableWhitespace(char[] ch, int start, int length)
			throws SAXException {
		stopTest();
	}

	public void fatalError(SAXParseException exception) throws SAXException {
		Kijimuna.reportException(exception);
		error(exception);
	}

	public void error(SAXParseException exception) throws SAXException {
		if ((storage != null) && (storage instanceof IFile) && (markerType != null)) {
			MarkerUtils.createMarker(markerType, MARKER_SEVERITY_XML_ERROR,
					errorSeverity, (IFile)storage, exception.getLineNumber(), 
					"[XML]" + exception.getMessage());
		}
	}

	public void warning(SAXParseException exception) throws SAXException {
		if ((storage != null) && (storage instanceof IFile) && (markerType != null)) {
			MarkerUtils.createMarker(markerType, MARKER_SEVERITY_XML_WARNING, 
					warningSeverity, (IFile)storage, exception.getLineNumber(),
					"[XML]" + exception.getMessage());
		}
	}

	private void stopTest() throws StopParse {
		if (stopper != null) {
			if (stopper.stopTest(locator)) {
				throw new StopParse();
			}
		}
	}
	
	public Element getResult() {
	    return result;
	}

}