/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.dicon.validation;

import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.dicon.IComponentInfo;
import org.seasar.kijimuna.core.dicon.IComponentNotFound;
import org.seasar.kijimuna.core.dicon.ITooManyRegisted;
import org.seasar.kijimuna.core.dicon.IValidation;
import org.seasar.kijimuna.core.dicon.model.ComponentElement;
import org.seasar.kijimuna.core.parser.Element;
import org.seasar.kijimuna.core.rtti.IRtti;
import org.seasar.kijimuna.core.rtti.IRttiMethod;
import org.seasar.kijimuna.core.util.ModelUtils;
import org.seasar.kijimuna.core.util.StringUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class AutoConstructorInvoke implements IValidation, ConstCore {

    public void validation(Element element) {
        if(element instanceof ComponentElement) {
            ComponentElement component = (ComponentElement)element;
            String el = component.getExpression();
            if((component.getArgList().size() == 0) && StringUtils.noneValue(el)) {
                String autoBinding = component.getAutoBindingMode();
                if(autoBinding.equals(DICON_VAL_AUTO_BINDING_AUTO) || 
                        autoBinding.equals(DICON_VAL_AUTO_BINDING_CONSTRUCTOR)) {
                    autoAssembler(component);
                } else {
                    defaultAssembler(component);
                }
            }
        }
    }

	// ftHgRXgN^ōsAbZu
	private void defaultAssembler(ComponentElement component) {
	    IRtti rtti = (IRtti)component.getAdapter(IRtti.class);
		if(rtti != null) {
			if(!rtti.hasConstructor(new IRtti[0])) {
			    MarkerSetting.createDiconMarker(
			            "dicon.validation.AutoConstructorInvoke.1",
			            component, new Object[] { rtti.getQualifiedName() });
			}
		}
	}
    
	private void reportInjection(ComponentElement component,
	        IRtti[] suitableArgs, IRtti[] injectedArgs, String display) {
		for(int i = 0; i < injectedArgs.length; i++) {
            if(injectedArgs[i] instanceof IComponentNotFound) {
				MarkerSetting.createDiconMarker(
				        "dicon.validation.AutoConstructorInvoke.3",
				        component, new Object[] { display, new Integer(i + 1) });
            } else if(injectedArgs[i] instanceof ITooManyRegisted) {
				MarkerSetting.createDiconMarker(
				        "dicon.validation.AutoConstructorInvoke.5",
				        component, new Object[] { display, new Integer(i + 1) });
            } else {
			    MarkerSetting.createDiconMarker(
			            "dicon.validation.AutoConstructorInvoke.2",
				        component, new Object[] {
                        	display, new Integer(i + 1),
                        	ModelUtils.getInjectedElementName(injectedArgs[i]) });
            }
		}
	}
	
	private boolean checkCyclicReference(
	        ComponentElement component, IRtti injectedArg, int num, String display) {
	    if(DICON_VAL_INSTANCE_SINGLETON.equals(component.getInstanceMode())) {
            ComponentElement injectedElement = 
            	(ComponentElement)injectedArg.getAdapter(ComponentElement.class);
            if(component.equals(injectedElement)) {
			    MarkerSetting.createDiconMarker(
			            "dicon.validation.AutoConstructorInvoke.4",
				        component, new Object[] { display, new Integer(num + 1) });
                return true;
            }
	    }
	    return false;
	}
	
	// ɃRXgN^TAbZu
	private void autoAssembler(ComponentElement component) {
	    IComponentInfo info = 
	        (IComponentInfo)component.getAdapter(IComponentInfo.class);
	    if(info != null) {
	        IRttiMethod suitable = info.getAutoInjectedConstructor();
	        if(suitable != null) {
			    IRtti[] suitableArgs = suitable.getArgs();
	            IRtti[] injectedArgs = suitable.getValues();
	            if(injectedArgs != null) {
					String display = ModelUtils.getMethodDisplay(suitable, true);
					for(int i = 0; i < suitableArgs.length; i++) {
					    if(injectedArgs[i] != null) {
					        checkCyclicReference(component, injectedArgs[i], i, display);
					    }
				    }
				    reportInjection(component, suitableArgs, injectedArgs, display);
	            }
	        } else {
				defaultAssembler(component);
	        }
	    }
	}

}
