/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.dicon.model;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IStorage;
import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.dicon.IComponentInfo;
import org.seasar.kijimuna.core.dicon.IComponentKey;
import org.seasar.kijimuna.core.dicon.ITooManyRegisted;
import org.seasar.kijimuna.core.dicon.tools.ComponentKeyRtti;
import org.seasar.kijimuna.core.dicon.tools.ComponentKeyString;
import org.seasar.kijimuna.core.dicon.tools.ComponentNotFoundRtti;
import org.seasar.kijimuna.core.dicon.tools.ContainerRtti;
import org.seasar.kijimuna.core.dicon.tools.InjectedRtti;
import org.seasar.kijimuna.core.parser.Element;
import org.seasar.kijimuna.core.rtti.IRtti;
import org.seasar.kijimuna.core.rtti.IRttiMethod;
import org.seasar.kijimuna.core.rtti.IRttiProperty;
import org.seasar.kijimuna.core.rtti.RttiClassNotFoundException;
import org.seasar.kijimuna.core.util.ProjectUtils;
import org.seasar.kijimuna.core.util.StringUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc.)
 */
public class ContainerElement extends DiconElement
	implements ConstCore, Comparable {

    private static IRtti s2ContainerRtti;
    
	private Map componentDefMap = new HashMap();
	private Map componentKeyMap = new HashMap();
	private Set containerKeySet = new TreeSet();
	private IComponentInfo info;

	public ContainerElement(IProject project, IStorage storage) {
		super(project, storage, DICON_TAG_CONTAINER);
		IComponentKey key = createComponentKey(MODEL_NAME_CONTAINER);
		componentDefMap.put(key, this);
		containerKeySet.add(key);
		IRtti rtti = getS2ContainerRtti();
		if(rtti != null) {
			IComponentKey interfaceKey = createComponentKey(rtti);
			componentDefMap.put(interfaceKey, this);
			containerKeySet.add(interfaceKey);
		}
	}

	private IRtti getS2ContainerRtti() {
        if(s2ContainerRtti == null) {
            try {
				s2ContainerRtti = getRttiLoader().loadRtti(
						MODEL_INTERFACE_S2CONTAINER);
			} catch (RttiClassNotFoundException e) {
			}
        }
        return s2ContainerRtti;
	}

	public void setAttributes(Map properties) {
		super.setAttributes(properties);
		String namespace = getNamespace();
		if(StringUtils.existValue(namespace)) {
			IComponentKey key = createComponentKey(namespace);
			componentDefMap.put(key, this);
			containerKeySet.add(key);
		}
	}

	protected void addChild(Element child) {
		super.addChild(child);
		if (child instanceof ComponentElement) {
			registComponentElement((ComponentElement) child);
		}
	}

	public List getComponentList() {
		return getChildren(DICON_TAG_COMPONENT);
	}

	public List getIncludeList() {
		return getChildren(DICON_TAG_INCLUDE);
	}

	public String getNamespace() {
		return super.getAttribute(DICON_ATTR_NAMESPACE);
	}

	public String getPath() {
		IStorage storage = (IStorage)getAdapter(IStorage.class);
		if(storage != null) {
		    return ProjectUtils.getResourceLoaderPath(storage);
		} else {
		    return null;
		}
	}

	public IComponentKey createComponentKey(Object key) {
		String keyString = key.toString();
	    IComponentKey ret = (IComponentKey)componentKeyMap.get(keyString);
	    if(ret == null) {
	    	if(key instanceof IRtti) {
	    		ret = new ComponentKeyRtti((IRtti)key);
	    	} else {
		        ret = new ComponentKeyString(keyString);
	    	}
		    componentKeyMap.put(keyString, ret);
	    }
	    return ret;
	}
	
	public void registComponentElement(ComponentElement component) {
		String instance = component.getInstanceMode();
		if (!instance.equals(DICON_VAL_INSTANCE_OUTER)) {
		    IComponentInfo keyInfo = (IComponentInfo)
		    	component.getAdapter(IComponentInfo.class);
		    if(keyInfo != null) {
		        IComponentKey[] keys = keyInfo.getComponentKeys();
				for (int i = 0; i < keys.length; i++) {
					registComponentElementByKey(keys[i], component);
				}
		    }
		}
	}

	private void registComponentElementByKey(
	        IComponentKey key, ComponentElement component) {
		if (key != null) {
			DiconElement registed = (DiconElement)componentDefMap.get(key);
			if (registed == null) {
				componentDefMap.put(key, component);
			} else {
			    key.makeTooMany();
			    if (registed instanceof TooManyRegistrationHolder) {
					((TooManyRegistrationHolder)
					        registed).addComponentElement(component);
				} else {
				    IProject project = (IProject)getAdapter(IProject.class);
				    IStorage storage = (IStorage)getAdapter(IStorage.class);
					if((project != null) && (storage != null)) {
						TooManyRegistrationHolder holder = 	
						    new TooManyRegistrationHolder(project, storage, key);
						holder.addComponentElement(registed);
						holder.addComponentElement(component);
						componentDefMap.put(key, holder);
					}
				}
			}
		}
	}

	private DiconElement findDefinition(IComponentKey componentKey) {
	    DiconElement element = 
	    	(DiconElement)componentDefMap.get(componentKey);
		if (element != null) {
			return element;
		} else {
			for (Iterator it = getIncludeList().iterator(); it.hasNext();) {
				IncludeElement incl = (IncludeElement) it.next();
				ContainerElement childContainer = incl.getChildContainer();
				element = childContainer.findDefinition(componentKey);
				if (element != null) {
					return element;
				}
			}
		}
		return null;
	}
	
	public IRtti getComponent(IComponentKey componentKey) {
		DiconElement element = findDefinition(componentKey);
		if(element instanceof ContainerElement) {
			ContainerElement container = (ContainerElement)element;
			return new ContainerRtti(getS2ContainerRtti(), container, componentKey);
		} else if(element instanceof ComponentElement) {
		    IRtti rtti = (IRtti)element.getAdapter(IRtti.class);
		    if(rtti != null) {
		        if(rtti instanceof ITooManyRegisted) {
		            return rtti;
		        } else {
			    	ComponentElement component = (ComponentElement)element;
			        return new InjectedRtti(rtti, component, componentKey);
		        }
		    }
		}
		return new ComponentNotFoundRtti(componentKey);
	}

    public Object getAdapter(Class adapter) {
        if(IRtti.class.equals(adapter)) {
            return getS2ContainerRtti();
        } else if (IComponentInfo.class.equals(adapter)){
        	if(info == null) {
        		info = new IComponentInfo() {
                    public IComponentKey[] getComponentKeys() {
                        return (IComponentKey[])containerKeySet.toArray(
                                new IComponentKey[containerKeySet.size()]);
                    }
                    public IRttiMethod getAutoInjectedConstructor() {
                    	return null;
                    }
                    public IRttiProperty[] getAutoInjectedProperties() {
                    	return new IRttiProperty[0];
                    }
                }; 
        	}
        	return info; 
        }
        return super.getAdapter(adapter);
    }

	public boolean isOGNL() {
	    return false;
	}
	
	public String getDisplayName() {
	    StringBuffer buffer = new StringBuffer();
	    buffer.append(getPath());
	    String namespace = getNamespace();
	    if(StringUtils.existValue(namespace)) {
	        buffer.append("<").append(namespace).append(">");
	    }
	    return buffer.toString();
	}
	
    public int compareTo(Object test) {
        if(test instanceof ContainerElement) {
            ContainerElement container = (ContainerElement)test;
            return getPath().compareTo(container.getPath());
        } else {
            return 0;
        }
    }
}