/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.dicon;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.Kijimuna;
import org.seasar.kijimuna.core.project.AbstractNature;
import org.seasar.kijimuna.core.project.IProjectRecordable;
import org.seasar.kijimuna.core.rtti.IRttiCache;
import org.seasar.kijimuna.core.rtti.RttiLoader;
import org.seasar.kijimuna.core.util.MarkerUtils;
import org.seasar.kijimuna.core.util.ProjectUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc)
 */
public class DiconNature extends AbstractNature
	implements IProjectRecordable, ConstCore {

    private static final String[] BUILDERS = 
        new String[] {
            ID_PROCESSOR_DICON_BUILDER, 
            ID_PROCESSOR_DICON_VALIDATOR
    	};

	public static DiconNature getInstance(IProject project) {
        if(project != null) {
		    try {
	            IProjectNature nature = project.getNature(ID_NATURE_DICON);
	            if(nature instanceof DiconNature) {
	                return (DiconNature)nature;
	            }
	        } catch (CoreException e) {
	            Kijimuna.reportException(e);
	        }
        }
        return null;
	}

	//------------------------------------------------------------
	
	private ModelManager model = new ModelManager();
	private RttiLoader rootLoader;

	public void setProject(IProject project) {
		super.setProject(project);
		model.setProjectName(project.getName());
	}
	
	public IRttiCache getRttiCache() {
		return getRttiLoader().getRttiCache();
	}
	
	public ModelManager getModel() {
		return model;
	}
	
	public void configure() throws CoreException {
		ProjectUtils.addBuilders(getProject(), BUILDERS);
	}

	public void deconfigure() throws CoreException {
	    ProjectUtils.removeBuilders(getProject(), BUILDERS);
		MarkerUtils.deleteMarker(getProject(), ID_MARKER);
		Kijimuna.getProjectRecorder().cleanup(getProject(), null);
	}

	public RttiLoader getRttiLoader() {
		if(rootLoader == null) {
			rootLoader = new RttiLoader(getProject().getName(), true);
		}
		return rootLoader;
	}
   
    public void initProjectRecords(IProgressMonitor monitor) {
    	model.init(monitor);
    }

    public boolean restoreProjectRecords(
    		IPath recorderPath, IProgressMonitor monitor) {
        File file = recorderPath.append(getProject().getName()
        		).addFileExtension(RECORDER_EXT_MODEL).toFile();
        boolean success = false;
        if(file.exists()) {
	        try {
				ObjectInputStream stream = 
					new ObjectInputStream(new FileInputStream(file));
				Object obj = stream.readObject();
				if(obj instanceof ModelManager) {
					synchronized(model) {
						model = (ModelManager)obj;
						model.afterRestoring();
					}
					success = true;
				} else {
				    Kijimuna.reportInfo(Kijimuna.getResourceString("dicon.DiconNature.1"));
				}
			} catch (Exception e) {
				Kijimuna.reportException(e);
			}
        }
        return success;
    }
    
    public boolean saveProjectRecords(
    		IPath recorderPath, IProgressMonitor monitor) {
    	boolean success = false;
		if(model.isDirty()) {
	        File file = recorderPath.append(getProject().getName()
	        		).addFileExtension(RECORDER_EXT_MODEL).toFile();
	        try {
	        	ObjectOutputStream stream = new ObjectOutputStream(
	        			new FileOutputStream(file));
	        	model.prepareStoraging();
        		stream.writeObject(model);
	        	success = true;
			} catch (Exception e) {
				Kijimuna.reportException(e);
			}
		}
		return success;
    }
    
    public void customProcess(int type, IPath recorderPath, IProgressMonitor monitor) {
        if(type == RECORDER_VALIDATE) {
            model.validate(monitor);
        }
    }
}