/*
 * 
 * The Seasar Software License, Version 1.1
 *
 * Copyright (c) 2003-2004 The Seasar Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgement:  
 *    "This product includes software developed by the 
 *    Seasar Project (http://www.seasar.org/)."
 *    Alternately, this acknowledgement may appear in the software
 *    itself, if and wherever such third-party acknowledgements 
 *    normally appear.
 *
 * 4. Neither the name "The Seasar Project" nor the names of its
 *    contributors may be used to endour or promote products derived 
 *    from this software without specific prior written permission of 
 *    the Seasar Project.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE SEASAR PROJECT 
 * OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
 * INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS 
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF 
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.seasar.kijimuna.core.dicon;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.seasar.kijimuna.core.ConstCore;
import org.seasar.kijimuna.core.Kijimuna;
import org.seasar.kijimuna.core.dicon.model.ContainerElement;
import org.seasar.kijimuna.core.dicon.tools.DiconElementFactory;
import org.seasar.kijimuna.core.dicon.validation.MarkerSetting;
import org.seasar.kijimuna.core.parser.DocumentHandler;
import org.seasar.kijimuna.core.parser.DocumentParser;
import org.seasar.kijimuna.core.parser.Element;
import org.seasar.kijimuna.core.project.AbstractProcessor;
import org.seasar.kijimuna.core.project.IFileProcessor;
import org.seasar.kijimuna.core.project.ResourceVisitor;
import org.seasar.kijimuna.core.util.FileUtils;
import org.seasar.kijimuna.core.util.MarkerUtils;

/**
 * @author Masataka Kurihara (Gluegent, Inc)
 */
public class DiconBuilder extends AbstractProcessor
		implements ConstCore {

    private DocumentParser parser = new DocumentParser();

    public String getNatureID() {
        return ID_NATURE_DICON;
    }
    
	public IFileProcessor getFileBuilder() {
		return this;
	}

	public void handlePrepareFullProcess(IProject project, IProgressMonitor monitor) {
		if (monitor != null) {
			monitor.beginTask(project.getName(), IProgressMonitor.UNKNOWN);
		}
	    DiconNature nature = DiconNature.getInstance(project);
	    if(nature != null) {
	        nature.getModel().clearContainer();
	        nature.getRttiCache().clearRttiCache();
	    }
	}

    public void handleFileAdded(IFile addedFile, boolean fullBuild, IProgressMonitor monitor) {
        process(addedFile.getProject(), addedFile, monitor);
    }

    public void handleFileChanged(IFile changedFile, IProgressMonitor monitor) {
        removeContainer(changedFile, monitor);
        process(changedFile.getProject(), changedFile, monitor);
    }

    public void handleFileRemoved(IFile removedFile, IProgressMonitor monitor) {
		if (FileUtils.isJavaFile(removedFile)) {
		    DiconNature nature = DiconNature.getInstance(removedFile.getProject());
		    if(nature != null) {
		        nature.getRttiCache().removeRttiFromCache(removedFile);
		    }
		} else {
			removeContainer(removedFile, monitor);
		}
	}
	
    public void handleClassPassChanged(IProject project, IProgressMonitor monitor) {
	    DiconNature nature = DiconNature.getInstance(project);
	    if(nature != null) {
	        nature.getRttiCache().clearRttiCache();
	    }
    }
    
    public void processProject(IProject project, IProgressMonitor monitor) {
        if(project != null) {
            try {
                project.accept(new ResourceVisitor(getNatureID(), this, monitor));
            } catch (CoreException e) {
                Kijimuna.reportException(e);
            }
        }
    }

    public void process(IProject project, IStorage storage, IProgressMonitor monitor) {
		String fileExt = storage.getFullPath().getFileExtension();
		if (!EXT_DICON.equalsIgnoreCase(fileExt)) {
			return;
		}
		if (monitor != null) {
			monitor.beginTask(storage.getName(), IProgressMonitor.UNKNOWN);
		}
		if(storage instanceof IFile) {
		    MarkerUtils.deleteMarker((IFile)storage, ID_MARKER);
		}
	    DiconNature nature = DiconNature.getInstance(project);
		if(nature != null) {
		    if (monitor != null) {
			    monitor.subTask(
			            Kijimuna.getResourceString("dicon.DiconBuilder.1",
			            new Object[] { storage.getName() }));
			}
		    int errorSeverity = MarkerSetting.getDiconMarkerPreference(
		            project, MARKER_CATEGORY_XML_ERROR, false);
		    int warningSeverity = MarkerSetting.getDiconMarkerPreference(
		            project, MARKER_CATEGORY_XML_WARNING, false);
			DocumentHandler handler = new DocumentHandler(new DiconElementFactory(),
			        ID_MARKER_DICONXML, errorSeverity, warningSeverity);
			handler.putDtdPath(PUBLIC_ID_DICON, "/components.dtd");
			Element result = parser.parse(project, storage, monitor, handler);
			if (result != null) {
				if(result instanceof ContainerElement) {
				    ContainerElement container = (ContainerElement)result;
					nature.getModel().addContainerAndRelatedFile(container, null);
				}
			}
		}
		if (monitor != null) {
			monitor.done();
		}
	}

	public void handleFinish(IProject project, IProgressMonitor monitor) {
	    DiconNature nature = DiconNature.getInstance(project);
		if(nature != null) {
			nature.getModel().fireRecordChanged();
		}
	}
    
	private void removeContainer(IFile file, IProgressMonitor monitor) {
		if (FileUtils.isDiconFile(file)) {
			if (monitor != null) {
				monitor.beginTask(file.getName(), IProgressMonitor.UNKNOWN);
			}
		    DiconNature nature = DiconNature.getInstance(file.getProject());
		    if(nature != null) {
		        nature.getModel().removeContainer(file);
		    }
		}
	}
	
}