package org.seasar.dao.meta;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.seasar.dao.DaoMetaData;
import org.seasar.dao.SqlCommand;
import org.seasar.dao.command.SelectCommand;
import org.seasar.dao.command.UpdateCommand;
import org.seasar.extension.jdbc.ResultSetHandler;
import org.seasar.extension.jdbc.impl.BeanListResultSetHandler;
import org.seasar.extension.jdbc.impl.BeanResultSetHandler;
import org.seasar.extension.jdbc.impl.ObjectResultSetHandler;
import org.seasar.framework.beans.BeanDesc;
import org.seasar.framework.beans.MethodNotFoundRuntimeException;
import org.seasar.framework.beans.factory.BeanDescFactory;
import org.seasar.framework.util.ClassUtil;
import org.seasar.framework.util.FieldUtil;
import org.seasar.framework.util.MethodUtil;
import org.seasar.framework.util.ResourceUtil;
import org.seasar.framework.util.StringUtil;
import org.seasar.framework.util.TextUtil;

/**
 * @author higa
 *  
 */
public class DaoMetaDataImpl implements DaoMetaData {

	private static final String[] INSERT_NAMES = new String[] { "insert",
			"create", "add" };

	private static final String[] UPDATE_NAMES = new String[] { "update",
			"modify", "store" };

	private static final String[] DELETE_NAMES = new String[] { "delete",
			"remove" };

	private Class daoClass_;

	private Class beanClass_;

	private Map sqlCommands_ = new HashMap();

	public DaoMetaDataImpl(Class daoClass) {
		daoClass_ = daoClass;
		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(daoClass);
		Field beanField = beanDesc.getField(BEAN_KEY);
		beanClass_ = (Class) FieldUtil.get(beanField, null);
		setup(beanDesc);
	}

	private void setup(BeanDesc beanDesc) {
		String[] names = beanDesc.getMethodNames();
		for (int i = 0; i < names.length; ++i) {
			Method[] methods = beanDesc.getMethods(names[i]);
			if (methods.length == 1 && MethodUtil.isAbstract(methods[0])) {
				setupMethod(beanDesc, methods[0]);
			}
		}
	}

	private void setupMethod(BeanDesc beanDesc, Method method) {
		String path = daoClass_.getName().replace('.', '/') + "_"
				+ method.getName() + ".sql";
		if (ResourceUtil.isExist(path)) {
			String sql = TextUtil.readText(path);
			setupMethodByManual(beanDesc, method, sql);
		}
	}

	private void setupMethodByManual(BeanDesc beanDesc, Method method,
			String sql) {

		if (isSelect(method)) {
			setupSelectMethodByManual(beanDesc, method, sql);
		} else {
			setupUpdateMethodByManual(beanDesc, method, sql);
		}
	}

	private void setupSelectMethodByManual(BeanDesc beanDesc, Method method,
			String sql) {

		ResultSetHandler rsh = null;
		if (method.getReturnType().isAssignableFrom(List.class)) {
			rsh = new BeanListResultSetHandler(beanClass_);
		} else if (method.getReturnType().isAssignableFrom(beanClass_)) {
			rsh = new BeanResultSetHandler(beanClass_);
		} else {
			rsh = new ObjectResultSetHandler();
		}
		SelectCommand cmd = new SelectCommand();
		cmd.setResultSetHandler(rsh);
		cmd.setSql(sql);
		cmd.setArgNames(getArgNames(beanDesc, method.getName()));
		sqlCommands_.put(method.getName(), cmd);
	}

	private void setupUpdateMethodByManual(BeanDesc beanDesc, Method method,
			String sql) {

		UpdateCommand cmd = new UpdateCommand();
		cmd.setSql(sql);
		String[] argNames = getArgNames(beanDesc, method.getName());
		if (argNames.length == 0 && method.getParameterTypes().length == 1
				&& method.getParameterTypes()[0].isAssignableFrom(beanClass_)) {
			argNames = new String[] { StringUtil.decapitalize(ClassUtil
					.getShortClassName(beanClass_)) };
		}
		cmd.setArgNames(argNames);
		sqlCommands_.put(method.getName(), cmd);
	}

	private boolean isSelect(Method method) {
		if (method.getReturnType() == Void.TYPE) {
			return false;
		}
		if (method.getReturnType() == int.class) {
			if (isInsert(method.getName())) {
				return false;
			}
			if (isUpdate(method.getName())) {
				return false;
			}
			if (isDelete(method.getName())) {
				return false;
			}
			return true;
		} else {
			return true;
		}
	}

	private boolean isInsert(String methodName) {
		for (int i = 0; i < INSERT_NAMES.length; ++i) {
			if (methodName.startsWith(INSERT_NAMES[i])) {
				return true;
			}
		}
		return false;
	}

	private boolean isUpdate(String methodName) {
		for (int i = 0; i < UPDATE_NAMES.length; ++i) {
			if (methodName.startsWith(UPDATE_NAMES[i])) {
				return true;
			}
		}
		return false;
	}

	private boolean isDelete(String methodName) {
		for (int i = 0; i < DELETE_NAMES.length; ++i) {
			if (methodName.startsWith(DELETE_NAMES[i])) {
				return true;
			}
		}
		return false;
	}

	private String[] getArgNames(BeanDesc beanDesc, String methodName) {
		String argsKey = methodName + ARGS_KEY_SUFFIX;
		if (beanDesc.hasField(argsKey)) {
			Field argNamesField = beanDesc.getField(argsKey);
			String argNames = (String) FieldUtil.get(argNamesField, null);
			return StringUtil.split(argNames, " ,");
		} else {
			return new String[0];
		}
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#getBeanClass()
	 */
	public Class getBeanClass() {
		return daoClass_;
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#getSqlCommand(java.lang.String)
	 */
	public SqlCommand getSqlCommand(String methodName)
			throws MethodNotFoundRuntimeException {

		SqlCommand cmd = (SqlCommand) sqlCommands_.get(methodName);
		if (cmd == null) {
			throw new MethodNotFoundRuntimeException(daoClass_, methodName,
					null);
		}
		return cmd;
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#hasSqlCommand(java.lang.String)
	 */
	public boolean hasSqlCommand(String methodName) {
		return sqlCommands_.containsKey(methodName);
	}
}