package org.maachang.rimdb.util;

/**
 * 数字Key数字Value.
 * 
 * @version 2014/06/29
 * @author masahito suzuki
 * @since rimdb-1.00
 */
public final class NNKeyValue {
	private static final int DEF_LENGTH = 32;
	private static final int MIN_LENGTH = 4;

	private NNChild[] list;
	private int mask;
	private int length;
	private int limit;
	private int base;

	/**
	 * コンストラクタ.
	 */
	public NNKeyValue() {
		this(DEF_LENGTH);
	}

	/**
	 * コンストラクタ.
	 * 
	 * @param size
	 *            初期サイズを設定します.この条件は２の倍数である必要があります.
	 */
	public NNKeyValue(int size) {
		if (size <= 0) {
			size = MIN_LENGTH;
		}
		size = ConvertUtil.bitMask(size);
		list = new NNChild[size];
		length = 0;
		mask = size - 1;
		limit = size;
		base = size;
	}

	/**
	 * クリア.
	 */
	public final void clear() {
		list = new NNChild[base];
		length = 0;
		mask = base - 1;
		limit = base;
	}

	/**
	 * 数値追加.
	 * 
	 * @param b
	 *            対象の数値を設定します.
	 * @param o
	 *            対象の要素を設定します.
	 * @return NNChild 現在追加された子要素が返却されます.
	 */
	public final NNChild put(final int b, final int o) {
		int h;
		if (length + 1 == limit) {
			int nLen = limit << 1;
			int msk = nLen - 1;
			NNChild[] nList = new NNChild[nLen];
			NNChild n, t;
			for (int i = 0; i < limit; i++) {
				n = list[i];
				while (n != null) {
					if (nList[(h = n.b & msk)] == null) {
						t = n.n;
						n.n = null;
					} else {
						t = n.n;
						n.n = nList[h];
					}
					nList[h] = n;
					n = t;
				}
			}
			list = nList;
			limit = nLen;
			mask = msk;
		}

		NNChild nn = list[(h = b & mask)];
		if (nn == null) {
			list[h] = new NNChild(b, o);
			length++;
			return list[h];
		}

		while (nn.n != null) {
			if (nn.b == b) {
				nn.o = o;
				return nn;
			}
			nn = nn.n;
		}
		if (nn.b == b) {
			nn.o = o;
			return nn;
		}
		nn.n = new NNChild(b, o);
		length++;
		return nn.n;
	}

	/**
	 * 情報が存在するかチェック.
	 * 
	 * @param b
	 *            対象の数値を設定します.
	 * @return boolean [true]の場合、数値は存在します.
	 */
	public final boolean containsKey(final int b) {
		NNChild n = list[b & mask];
		;
		while (n != null) {
			if (n.b == b) {
				return true;
			}
			n = n.n;
		}
		return false;
	}

	/**
	 * 情報を取得.
	 * 
	 * @param b
	 *            対象の数値を設定します.
	 * @return Integer 要素情報が返却されます.
	 */
	public final Integer get(final int b) {
		NNChild n = list[b & mask];
		while (n != null) {
			if (n.b == b) {
				return n.o;
			}
			n = n.n;
		}
		return null;
	}

	/**
	 * 子要素を取得.
	 * 
	 * @param b
	 *            対象の数値を設定します.
	 * @return NNChild 要素情報が返却されます.
	 */
	public final NNChild getChild(final int b) {
		NNChild n = list[b & mask];
		while (n != null) {
			if (n.b == b) {
				return n;
			}
			n = n.n;
		}
		return null;
	}

	/**
	 * 削除処理.
	 * 
	 * @param b
	 *            対象の数値を設定します.
	 */
	public final void remove(final int b) {
		NNChild n = list[b & mask];
		NNChild bf = null;
		while (n != null) {
			if (n.b == b) {
				if (bf == null) {
					if (n.n == null) {
						list[b & mask] = null;
					} else {
						list[b & mask] = n.n;
					}
				} else {
					if (n.n == null) {
						bf.n = null;
					} else {
						bf.n = n.n;
					}
				}
				length--;
				break;
			}
			bf = n;
			n = n.n;
		}
	}

	/**
	 * サイズを取得.
	 * 
	 * @return int サイズが返却されます.
	 */
	public final int size() {
		return length;
	}

	/**
	 * 数値配列として返却.
	 * 
	 * @return int[] 数値配列として返却します.
	 */
	public final int[] keyArray() {
		if (length <= 0) {
			return null;
		}
		NNChild n;
		int cnt = 0;
		int[] ret = new int[length];
		for (int i = 0; i < limit; i++) {
			if (list[i] != null) {
				n = list[i];
				while (n != null) {
					ret[cnt++] = n.b;
					n = n.n;
				}
			}
		}
		return ret;
	}

	/** 位置読み込み条件. **/
	private int pos = 0;
	private NNChild cPos = null;

	/**
	 * 読み込み中の位置をリセット.
	 * 
	 * @return NNKeyValue このオブジェクトが返却されます.
	 */
	public final NNKeyValue reset() {
		pos = 0;
		cPos = null;
		return this;
	}

	/**
	 * 次の読み込み条件を取得.
	 * 
	 * @return boolean [true]が返却された場合、情報は存在します.
	 */
	public final boolean hasNext() {
		if (cPos != null) {
			if ((cPos = cPos.n) != null) {
				return true;
			}
		}
		while (pos < limit) {
			if ((cPos = list[pos++]) != null) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 現在の読み込み位置の情報を取得.
	 * 
	 * @return int 現在の読み込み位置の内容が返却されます. [-1]の場合は、情報は存在しません.
	 */
	public final int next() {
		if (cPos == null) {
			return -1;
		}
		return cPos.b;
	}

	/*
	 * 現在の読み込み位置の要素情報を取得.
	 * 
	 * @return Integer 現在の読み込み位置の内容が返却されます.
	 */
	public final Integer nextValue() {
		if (cPos == null) {
			return null;
		}
		return cPos.o;
	}

	/*
	 * 現在の読み込み位置の子要素を取得.
	 * 
	 * @return NNChild 現在の読み込み位置の子要素が返却されます.
	 */
	public final NNChild nextChild() {
		return cPos;
	}
}
