package org.maachang.crypto ;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Crypto ユーティリティー群.
 *
 * @version 2012/08/12
 * @author  masahito suzuki
 * @since   Crypto 1.00
 */
class CryptoUtils {
    private CryptoUtils() {}
    
    /**
     * UTF8文字列を、通常バイナリに変換.
     */
    public static final String convertUtf8ToBinary( String n )
        throws Exception {
        if( n == null || n.length() <= 0 ) {
            throw new IllegalArgumentException( "対象の文字列が設定されていません" ) ;
        }
        int len = n.length() ;
        CryptoUtils_BinaryBuffer buf = new CryptoUtils_BinaryBuffer( n.length() * 3 ) ;
        char c ;
        for( int i = len-1 ; i >= 0 ; i -- ) {
            c = n.charAt( i ) ;
            if (c < 128) {
                buf.write( (int)c ) ;
            }
            else if ((c > 127) && (c < 2048)) {
                buf.write( (int)( (c & 63) | 128 ) ) ;
                buf.write( (int)( (c >> 6) | 192 ) ) ;
            }
            else {
                buf.write( (int)( (c & 63) | 128 ) ) ;
                buf.write( (int)( ((c >> 6) & 63) | 128 ) ) ;
                buf.write( (int)( (c >> 12) | 224 ) ) ;
            }
        }
        byte[] lst = buf.getBinary() ;
        buf = null ;
        len = lst.length ;
        StringBuilder ret = new StringBuilder( len ) ;
        for( int i = len-1 ; i >= 0 ; i -- ) {
            ret.append( (char)( lst[ i ] & 0x000000ff ) ) ;
        }
        lst = null ;
        return ret.toString() ;
    }
    
    /**
     * 通常バイナリをUTF8文字列に変換.
     */
    public static final String convertBinaryToUTF8( String n )
        throws Exception {
        if( n == null || n.length() <= 0 ) {
            throw new IllegalArgumentException( "対象の文字列が設定されていません" ) ;
        }
        int len = n.length() ;
        StringBuilder buf = new StringBuilder( len ) ;
        int c ;
        for( int i = 0 ; i < len ; i ++ ) {
            c = (int)( n.charAt( i ) & 0x000000ff ) ;
            if( c < 128 ) {
                buf.append( (char)c ) ;
            }
            else if( c > 191 && c < 224 ) {
                buf.append( (char)( ((c & 31) << 6) |
                    (((int)( n.charAt( i+1 ) & 0x000000ff ) & 255) & 63) ) ) ;
                i += 1 ;
            }
            else {
                buf.append( (char)( ((c & 15) << 12) |
                    ((((int)( n.charAt( i+1 ) & 0x000000ff ) & 255) & 63) << 6) |
                    (((int)( n.charAt( i+2 ) & 0x000000ff ) & 255) & 63) ) ) ;
                i += 2 ;
            }
        }
        return buf.toString() ;
    }
    
    /**
     * バイナリ検索.
     */
    public static final int binaryIndexOf( byte[] binary,byte[] data,int off )
        throws Exception {
        return binaryIndexOf( binary,data,off,0 ) ;
    }
    
    /**
     * バイナリ検索.
     */
    public static final int binaryIndexOf( byte[] binary,byte[] data,int off,int len )
        throws Exception {
        if( binary == null || binary.length <= 0 ||
            data == null || data.length <= 0 ) {
            return -1 ;
        }
        int dataLen = data.length ;
        if( len <= 0 || len >= binary.length ) {
            len = binary.length ;
        }
        int ret = -1 ;
        for( int i = off ; i < len ; i ++ ) {
            if( binary[ i ] == data[ 0 ] && i+dataLen <= len ) {
                ret = i ;
                for( int j = 1 ; j < dataLen ; j ++ ) {
                    if( binary[ i+j ] != data[ j ] ) {
                        ret = -1 ;
                        break ;
                    }
                }
                if( ret != -1 ) {
                    return ret ;
                }
            }
        }
        return -1 ;
    }
    
}

/** Base64. **/
class CryptoUtils_Base64 {
    private static final char REMAINDER_ENC = '=';
    private static final char[] ENC_CD = { 'A', 'B', 'C', 'D', 'E', 'F', 'G',
        'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T',
        'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f', 'g',
        'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
        'u', 'v', 'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6',
        '7', '8', '9', '+', '/' };
    private static final int[] DEC_CD = { 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x0000003e, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x0000003f,
        0x00000034, 0x00000035, 0x00000036, 0x00000037, 0x00000038,
        0x00000039, 0x0000003a, 0x0000003b, 0x0000003c, 0x0000003d,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x00000040, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x00000000, 0x00000001, 0x00000002, 0x00000003,
        0x00000004, 0x00000005, 0x00000006, 0x00000007, 0x00000008,
        0x00000009, 0x0000000a, 0x0000000b, 0x0000000c, 0x0000000d,
        0x0000000e,
        0x0000000f, 0x00000010, 0x00000011, 0x00000012, 0x00000013,
        0x00000014, 0x00000015, 0x00000016, 0x00000017, 0x00000018,
        0x00000019, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x0000001a, 0x0000001b, 0x0000001c, 0x0000001d,
        0x0000001e, 0x0000001f, 0x00000020, 0x00000021, 0x00000022,
        0x00000023, 0x00000024, 0x00000025, 0x00000026, 0x00000027,
        0x00000028,
        0x00000029, 0x0000002a, 0x0000002b, 0x0000002c, 0x0000002d,
        0x0000002e, 0x0000002f, 0x00000030, 0x00000031, 0x00000032,
        0x00000033, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff, 0x7fffffff,
        0x7fffffff };
    
    /**
     * コンストラクタ.
     */
    private CryptoUtils_Base64() {
    }
    
    /**
     * エンコード処理.
     */
    public static final String encode(byte[] binary)
        throws IllegalArgumentException {
        int i, j, k;
        int len;
        int allLen;
        int etc;
        String ret = null;
        char[] ary = null;
        if (binary == null || (allLen = binary.length) <= 0) {
            throw new IllegalArgumentException("引数は不正です");
        }
        etc = allLen % 3;
        len = allLen / 3;
        ary = new char[(len * 4) + ((etc != 0) ? 4 : 0)];
        for (i = 0, j = 0, k = 0; i < len; i++, j += 3, k += 4) {
            ary[k] = CryptoUtils_Base64.ENC_CD[(int) ((binary[j] & 0x000000fc) >> 2)];
            ary[k + 1] = CryptoUtils_Base64.ENC_CD[(int) (((binary[j] & 0x00000003) << 4) | ((binary[j + 1] & 0x000000f0) >> 4))];
            ary[k + 2] = CryptoUtils_Base64.ENC_CD[(int) (((binary[j + 1] & 0x0000000f) << 2) | ((binary[j + 2] & 0x000000c0) >> 6))];
            ary[k + 3] = CryptoUtils_Base64.ENC_CD[(int) (binary[j + 2] & 0x0000003f)];
        }
        switch (etc) {
        case 1:
            j = len * 3;
            k = len * 4;
            ary[k] = CryptoUtils_Base64.ENC_CD[(int) ((binary[j] & 0x000000fc) >> 2)];
            ary[k + 1] = CryptoUtils_Base64.ENC_CD[(int) ((binary[j] & 0x00000003) << 4)];
            ary[k + 2] = CryptoUtils_Base64.REMAINDER_ENC;
            ary[k + 3] = CryptoUtils_Base64.REMAINDER_ENC;
            break;
        case 2:
            j = len * 3;
            k = len * 4;
            ary[k] = CryptoUtils_Base64.ENC_CD[(int) ((binary[j] & 0x000000fc) >> 2)];
            ary[k + 1] = CryptoUtils_Base64.ENC_CD[(int) (((binary[j] & 0x00000003) << 4) | ((binary[j + 1] & 0x000000f0) >> 4))];
            ary[k + 2] = CryptoUtils_Base64.ENC_CD[(int) (((binary[j + 1] & 0x0000000f) << 2))];
            ary[k + 3] = CryptoUtils_Base64.REMAINDER_ENC;
            break;
        }
        ret = new String(ary, 0, ary.length);
        ary = null;
        return ret;
    }
    
    /**
     * デコード処理.
     */
    public static final byte[] decode(String base64)
            throws IllegalArgumentException {
        int i, j, k;
        int len;
        int allLen;
        int etc;
        byte[] ret = null;
        if (base64 == null || (allLen = base64.length()) <= 0) {
            throw new IllegalArgumentException("引数は不正です");
        }
        for (i = allLen - 1, etc = 0; i >= 0; i--) {
            if (base64.charAt(i) == CryptoUtils_Base64.REMAINDER_ENC) {
                etc++;
            } else {
                break;
            }
        }
        len = allLen / 4;
        ret = new byte[(len * 3) - etc];
        len -= 1;
        for (i = 0, j = 0, k = 0; i < len; i++, j += 4, k += 3) {
            ret[k] = (byte) (((CryptoUtils_Base64.DEC_CD[base64.charAt(j)] & 0x0000003f) << 2) | ((CryptoUtils_Base64.DEC_CD[base64
                    .charAt(j + 1)] & 0x00000030) >> 4));
            ret[k + 1] = (byte) (((CryptoUtils_Base64.DEC_CD[base64.charAt(j + 1)] & 0x0000000f) << 4) | ((CryptoUtils_Base64.DEC_CD[base64
                    .charAt(j + 2)] & 0x0000003c) >> 2));
            ret[k + 2] = (byte) (((CryptoUtils_Base64.DEC_CD[base64.charAt(j + 2)] & 0x00000003) << 6) | (CryptoUtils_Base64.DEC_CD[base64
                    .charAt(j + 3)] & 0x0000003f));
        }
        switch (etc) {
        case 0:
            j = len * 4;
            k = len * 3;
            ret[k] = (byte) (((CryptoUtils_Base64.DEC_CD[base64.charAt(j)] & 0x0000003f) << 2) | ((CryptoUtils_Base64.DEC_CD[base64
                    .charAt(j + 1)] & 0x00000030) >> 4));
            ret[k + 1] = (byte) (((CryptoUtils_Base64.DEC_CD[base64.charAt(j + 1)] & 0x0000000f) << 4) | ((CryptoUtils_Base64.DEC_CD[base64
                    .charAt(j + 2)] & 0x0000003c) >> 2));
            ret[k + 2] = (byte) (((CryptoUtils_Base64.DEC_CD[base64.charAt(j + 2)] & 0x00000003) << 6) | (CryptoUtils_Base64.DEC_CD[base64
                    .charAt(j + 3)] & 0x0000003f));
            break;
        case 1:
            j = len * 4;
            k = len * 3;
            ret[k] = (byte) (((CryptoUtils_Base64.DEC_CD[base64.charAt(j)] & 0x0000003f) << 2) | ((CryptoUtils_Base64.DEC_CD[base64
                    .charAt(j + 1)] & 0x00000030) >> 4));
            ret[k + 1] = (byte) (((CryptoUtils_Base64.DEC_CD[base64.charAt(j + 1)] & 0x0000000f) << 4) | ((CryptoUtils_Base64.DEC_CD[base64
                    .charAt(j + 2)] & 0x0000003c) >> 2));
            break;
        case 2:
            j = len * 4;
            k = len * 3;
            ret[k] = (byte) (((CryptoUtils_Base64.DEC_CD[base64.charAt(j)] & 0x0000003f) << 2) | ((CryptoUtils_Base64.DEC_CD[base64
                    .charAt(j + 1)] & 0x00000030) >> 4));
            break;
        }
        return ret;
    }
}

/** BinaryBuffer. **/
class CryptoUtils_BinaryBuffer {
    private static final int BUFFER = 1024 ;
    private byte[] binary = null ;
    private int pos = 0 ;
    
    public CryptoUtils_BinaryBuffer() {
        this( -1 ) ;
    }
    
    public CryptoUtils_BinaryBuffer( int size ) {
        if( size <= 0 ) {
            size = BUFFER ;
        }
        this.binary = new byte[ size ] ;
        this.pos = 0 ;
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    public void destroy() {
        this.binary = null ;
        this.pos = 0 ;
    }
    
    public int position( int pos ) {
        if( pos <= -1 ) {
            return this.pos ;
        }
        if( this.binary.length < pos ) {
            addLimit( pos - this.binary.length ) ;
            this.pos = pos ;
        }
        else {
            this.pos = pos ;
        }
        return this.pos ;
    }
    
    public int position() {
        return pos ;
    }
    
    public void write( int b ) throws IOException {
        addLimit( 1 ) ;
        binary[ pos ] = ( byte )( b & 0x000000ff ) ;
        pos ++ ;
    }
    
    public void write( byte[] bin ) throws IOException {
        if( bin == null || bin.length <= 0 ) {
            return ;
        }
        write( bin,-1 ) ;
    }
    
    public void write( byte[] bin,int len ) throws IOException {
        if( bin == null || bin.length <= 0 ) {
            return ;
        }
        if( len <= 0 || bin.length <= len ) {
            len = bin.length ;
        }
        addLimit( len ) ;
        int off = this.pos ;
        System.arraycopy( bin,0,this.binary,off,len ) ;
        this.pos += len ;
    }
    
    public byte getByte( int no ) {
        return binary[ no ] ;
    }
    
    public byte[] getRawBinary() {
        return binary ;
    }
    
    public byte[] getBinary() {
        return getBinary( this.pos ) ;
    }
    
    public byte[] getBinary( int length ) {
        if( length > binary.length ) {
            return this.binary ;
        }
        byte[] ret = new byte[ length ] ;
        System.arraycopy( this.binary,0,ret,0,length ) ;
        return ret ;
    }
    
    private void addLimit( int len ) {
        if( binary.length <= pos + len ) {
            byte[] t = binary ;
            int iLen = ( t.length * 2 ) + len ;
            binary = new byte[ iLen ] ;
            System.arraycopy( t,0,binary,0,t.length ) ;
            t = null ;
        }
    }
}

/** JSON解析. **/
class CryptoUtils_DecodeJSON {
    private static final int TYPE_ARRAY = 0 ;
    private static final int TYPE_MAP = 1 ;
    private static final String NEW_DATE = "Date" ;
    private static final int NEW_DATE_LEN = NEW_DATE.length() ;
    
    private CryptoUtils_DecodeJSON() {}
    
    /**
     * JSON形式から、オブジェクト変換.
     */
    public static final Object execution( String json )
        throws Exception {
        if( json == null ) {
            return null ;
        }
        int[] n = new int[1] ;
        json = json.trim() ;
        // コメントを除外.
        json = cutComment( json ).trim() ;
        if( json.length() == 0 ) {
            return json ;
        }
        while( true ) {
            // token解析が必要な場合.
            if( json.startsWith( "[" ) || json.startsWith( "{" ) ) {
                // JSON形式をToken化.
                List<Object> list = analysisJsonToken( json ) ;
                // Token解析処理.
                if( "[".equals( list.get( 0 ) ) ) {
                    // List解析.
                    return createJsonInfo( n,list,TYPE_ARRAY,0,list.size() ) ;
                }
                else {
                    // Map解析.
                    return createJsonInfo( n,list,TYPE_MAP,0,list.size() ) ;
                }
            }
            else if( json.startsWith( "(" ) && json.endsWith( ")" ) ) {
                json = json.substring( 1,json.length()-1 ).trim() ;
                continue ;
            }
            break ;
        }
        return decJsonValue( n,0,json ) ;
    }
    
    /** [decodeJSON]１つの要素を変換. **/
    private static final Object decJsonValue( int[] n,int no,String json )
        throws Exception {
        int len ;
        if( ( len = json.length() ) <= 0 ) {
            return json ;
        }
        // 文字列コーテーション区切り.
        if( ( json.startsWith( "\"" ) && json.endsWith( "\"" ) ) ||
            ( json.startsWith( "\'" ) && json.endsWith( "\'" ) ) ) {
            return json.substring( 1,len-1 ) ;
        }
        // NULL文字.
        else if( "null".equals( json ) ) {
            return null ;
        }
        // BOOLEAN true.
        else if( "true".equals( json ) ) {
            return Boolean.TRUE ;
        }
        // BOOLEAN false.
        else if( "false".equals( json ) ) {
            return Boolean.FALSE ;
        }
        // 数値.
        if( isNumber( n,json ) ) {
            if( n[0] == 0x00000004 ) {
                return Double.parseDouble( json ) ;
            }
            return Long.parseLong( json ) ;
        }
        // new Date.
        else if( json.startsWith( "new" ) ) {
            int x = indexByEquals( json,NEW_DATE,0 ) ;
            if( x != -1 ) {
                int b = x + NEW_DATE_LEN ;
                x = indexParAndCote( json,'(',')',b ) ;
                b = indexByEquals( json,"(",b ) ;
                if( x == -1 ) {
                    throw new IOException( "JSON解析に失敗(" + json + "):No:" + no ) ;
                }
                String s = json.substring( b+1,x ).trim() ;
                if( s.length() <= 0 ) {
                    return new Date() ;
                }
                if( ( s.startsWith( "'" ) && s.endsWith( "'" ) ) ||
                    ( s.startsWith( "\"" ) && s.endsWith( "\"" ) ) ) {
                    s = s.substring( 1,s.length()-1 ).trim() ;
                }
                return new Date( Long.parseLong( s ) ) ;
            }
        }
        // その他.
        throw new IOException( "JSON解析に失敗(" + json + "):No:" + no ) ;
    }
    
    /** JSON_Token_解析処理 **/
    private static final List<Object> analysisJsonToken( String json ) throws Exception {
        List<Object> ret = new ArrayList<Object>() ;
        int s = -1 ;
        char c ;
        int cote = -1 ;
        int bef = -1 ;
        int len = json.length() ;
        // Token解析.
        for( int i = 0 ; i < len ; i ++ ) {
            c = json.charAt( i ) ;
            // コーテーション内.
            if( cote != -1 ) {
                // コーテーションの終端.
                if( bef != '\\' && cote == c ) {
                    ret.add( json.substring( s-1,i+1 ) ) ;
                    cote = -1 ;
                    s = i+1 ;
                }
            }
            // コーテーション開始.
            else if( bef != '\\' && ( c == '\'' || c == '\"' ) ) {
                cote = c ;
                if( s != -1 && s != i && bef != ' ' && bef != '　' && bef != '\t' && bef != '\n' && bef != '\r' ) {
                    ret.add( json.substring( s,i+1 ) ) ;
                }
                s = i+1 ;
                bef = -1 ;
            }
            // ワード区切り.
            else if( c == '[' || c == ']' ||
                    c == '{' || c == '}' ||
                    c == '(' || c == ')' ||
                    c == ':' || c == ';' ||
                    c == ',' || ( c == '.' && ( bef < '0' || bef > '9' ) ) ) {
                if( s != -1 && s != i && bef != ' ' &&  bef != '　'  && bef != '\t' && bef != '\n' && bef != '\r' ) {
                    ret.add( json.substring( s,i ) ) ;
                }
                ret.add( new String( new char[]{c} ) ) ;
                s = i+1 ;
            }
            // 連続空間区切り.
            else if( c == ' ' || c == '　' || c == '\t' || c == '\n' || c == '\r' ) {
                if( s != -1 && s != i && bef != ' ' &&  bef != '　' && bef != '\t' && bef != '\n' && bef != '\r' ) {
                    ret.add( json.substring( s,i ) ) ;
                }
                s = -1 ;
            }
            // その他文字列.
            else if( s == -1 ) {
                s = i ;
            }
            bef = c ;
        }
        return ret ;
    }
    
    /** Json-Token解析. **/
    private static final Object createJsonInfo( int[] n,List<Object> token,int type,int no,int len )
        throws Exception {
        String value ;
        StringBuilder before = null ;
        // List.
        if( type == TYPE_ARRAY ) {
            List ret = new ArrayList() ;
            int flg = 0 ;
            for( int i = no+1 ; i < len ; i ++ ) {
                value = (String)token.get( i ) ;
                if( ",".equals( value ) || "]".equals( value ) ) {
                    if( "]".equals( value ) ) {
                        if( flg == 1 ) {
                            if( before != null ) {
                                ret.add( decJsonValue( n,i,before.toString() ) ) ;
                            }
                        }
                        n[ 0 ] = i ;
                        return ret ;
                    }
                    else {
                        if( flg == 1 ) {
                            if( before == null ) {
                                ret.add( null ) ;
                            }
                            else {
                                ret.add( decJsonValue( n,i,before.toString() ) ) ;
                            }
                        }
                    }
                    before = null ;
                    flg = 0 ;
                }
                else if( "[".equals( value ) ) {
                    ret.add( createJsonInfo( n,token,0,i,len ) ) ;
                    i = n[ 0 ] ;
                    before = null ;
                    flg = 0 ;
                }
                else if( "{".equals( value ) ) {
                    ret.add( createJsonInfo( n,token,1,i,len ) ) ;
                    i = n[ 0 ] ;
                    before = null ;
                    flg = 0 ;
                }
                else {
                    if( before == null ) {
                        before = new StringBuilder() ;
                        before.append( value ) ;
                    }
                    else {
                        before.append( " " ).append( value ) ;
                    }
                    flg = 1 ;
                }
            }
            n[ 0 ] = len-1 ;
            return ret ;
        }
        // map.
        else if( type == TYPE_MAP ) {
            Map ret = new HashMap() ;
            String key = null ;
            for( int i = no+1 ; i < len ; i ++ ) {
                value = (String)token.get( i ) ;
                if( ":".equals( value ) ) {
                    if( key == null ) {
                        throw new IOException( "Map形式が不正です(No:" + i + ")" ) ;
                    }
                }
                else if( ",".equals( value ) || "}".equals( value ) ) {
                    if( "}".equals( value ) ) {
                        if( key != null ) {
                            if( before == null ) {
                                ret.put( key,null ) ;
                            }
                            else {
                                ret.put( key,decJsonValue( n,i,before.toString() ) ) ;
                            }
                        }
                        n[ 0 ] = i ;
                        return ret ;
                    }
                    else {
                        if( key == null ) {
                            if( before == null ) {
                                continue ;
                            }
                            throw new IOException( "Map形式が不正です(No:" + i + ")" ) ;
                        }
                        if( before == null ) {
                            ret.put( key,null ) ;
                        }
                        else {
                            ret.put( key,decJsonValue( n,i,before.toString() ) ) ;
                        }
                        before = null ;
                        key = null ;
                    }
                }
                else if( "[".equals( value ) ) {
                    if( key == null ) {
                        throw new IOException( "Map形式が不正です(No:" + i + ")" ) ;
                    }
                    ret.put( key,createJsonInfo( n,token,0,i,len ) ) ;
                    i = n[ 0 ] ;
                    key = null ;
                    before = null ;
                }
                else if( "{".equals( value ) ) {
                    if( key == null ) {
                        throw new IOException( "Map形式が不正です(No:" + i + ")" ) ;
                    }
                    ret.put( key,createJsonInfo( n,token,1,i,len ) ) ;
                    i = n[ 0 ] ;
                    key = null ;
                    before = null ;
                }
                else if( key == null ) {
                    key = value ;
                    if( ( key.startsWith( "'" ) && key.endsWith( "'" ) ) ||
                        ( key.startsWith( "\"" ) && key.endsWith( "\"" ) ) ) {
                        key = key.substring( 1,key.length()-1 ).trim() ;
                    }
                }
                else {
                    if( before == null ) {
                        before = new StringBuilder() ;
                        before.append( value ) ;
                    }
                    else {
                        before.append( " " ).append( value ) ;
                    }
                }
            }
            n[ 0 ] = len-1 ;
            return ret ;
        }
        // その他.
        throw new IOException( "JSON解析に失敗(unknown)" ) ;
    }
    
    /** コメントをカット. **/
    private static final String cutComment( String script ) throws Exception {
        if( script == null || script.length() <= 0 ) {
            return "" ;
        }
        StringBuilder buf = new StringBuilder() ;
        int len = script.length() ;
        int cote = -1 ;
        int commentType = -1 ;
        int bef = -1 ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( i != 0 ) {
                bef = script.charAt( i-1 ) ;
            }
            char c = script.charAt(i) ;
            // コメント内の処理.
            if( commentType != -1 ) {
                switch( commentType ) {
                    case 1 : // １行コメント.
                        if( c == '\n' ) {
                            buf.append( c ) ;
                            commentType = -1 ;
                        }
                        break ;
                    case 2 : // 複数行コメント.
                        if( c == '\n' ) {
                            buf.append( c ) ;
                        }
                        else if( len > i + 1 && c == '*' && script.charAt( i+1 ) == '/' ) {
                            i ++ ;
                            commentType = -1 ;
                        }
                        break ;
                }
                continue ;
            }
            // シングル／ダブルコーテーション内の処理.
            if( cote != -1 ) {
                if( c == cote && ( char )bef != '\\' ) {
                    cote = -1 ;
                }
                buf.append( c ) ;
                continue ;
            }
            // それ以外の処理.
            if( c == '/' ) {
                if( len <= i + 1 ) {
                    buf.append( c ) ;
                    continue ;
                }
                char c2 = script.charAt( i+1 ) ;
                if( c2 == '*' ) {
                    commentType = 2 ;
                    continue ;
                }
                else if( c2 == '/' ) {
                    commentType = 1 ;
                    continue ;
                }
            }
            // コーテーション開始.
            else if( ( c == '\'' || c == '\"' ) && ( char )bef != '\\' ) {
                cote = ( int )( c & 0x0000ffff ) ;
            }
            buf.append( c ) ;
        }
        return buf.toString() ;
    }
    
    /**
     * 指定内容が数値かチェック.
     */
    public static final boolean isNumber( int[] dot,Object o ) {
        if( dot != null && dot.length > 0 ) {
            dot[ 0 ] = 0x00000000 ;
        }
        if( o instanceof Number ) {
            if( dot != null && dot.length > 0 ) {
                if( o instanceof Double || o instanceof BigDecimal ) {
                    dot[ 0 ] = 0x00000001 ;
                }
                else if( o instanceof Float ) {
                    dot[ 0 ] = 0x00000002 ;
                }
            }
            return true ;
        }
        if( dot != null && dot.length > 0 ) {
            dot[ 0 ] = 0x00000003 ;
        }
        return isNumberByString( dot,o ) ;
    }
    
    /** 指定引数が文字列の時の数値チェック処理. **/
    private static final boolean isNumberByString( int[] dot,Object o ) {
        String num = null ;
        if( o instanceof String ) {
            num = ( String )o ;
        }
        else {
            return false ;
        }
        if( num == null || ( num = num.trim() ).length() <= 0 ) {
            return false ;
        }
        int start = 0 ;
        if( num.startsWith( "-" ) ) {
            start = 1 ;
        }
        boolean dt = false ;
        int len = num.length() ;
        if( start < len ) {
            for( int i = start ; i < len ; i ++ ) {
                char c = num.charAt( i ) ;
                if( c == '.' ) {
                    if( dt ) {
                        return false ;
                    }
                    if( dot != null && dot.length > 0 ) {
                        dot[ 0 ] = 0x00000004 ;
                    }
                    dt = true ;
                }
                else if( ( c >= '0' && c <= '9' ) == false ) {
                    return false ;
                }
            }
        }
        else {
            return false ;
        }
        return true ;
    }
    
    /**
     * 文法の内、コーテーションを除く連続文字列を検知.
     */
    public static final int indexByEquals( String base,String cc,int off ) {
        int len = base.length() ;
        int cote = -1 ;
        char[] ck = cc.toCharArray() ;
        int cLen = ck.length ;
        char bef = 0 ;
        boolean yenFlag = false ;
        for( int i = off ; i < len ; i ++ ) {
            char c = base.charAt( i ) ;
            if( cote != -1 ) {
                if( bef != '\\' && c == cote ) {
                    yenFlag = false ;
                    cote = -1 ;
                }
                else if( c == '\\' && bef == '\\' ) {
                    yenFlag = true ;
                }
                else {
                    yenFlag = false ;
                }
            }
            else {
                if( bef != '\\' && ( c == '\'' || c == '\"' ) ) {
                    cote = c ;
                }
                else if( c == ck[ 0 ] ) {
                    boolean res = true ;
                    for( int j = 1 ; j < cLen ; j ++ ) {
                        if( i + j >= len || ck[ j ] != base.charAt( i + j ) ) {
                            res = false ;
                            break ;
                        }
                    }
                    if( res == true ) {
                        return i ;
                    }
                }
            }
            if( yenFlag ) {
                yenFlag = false ;
                bef = 0 ;
            }
            else {
                bef = c ;
            }
        }
        return -1 ;
    }
    
    /**
     * 指定カッコの終端を検知.
     */
    public static final int indexParAndCote( String base,char st,char ed,int off ) throws Exception {
        int len = base.length() ;
        int par = -1 ;
        int cote = -1 ;
        int idx = 0 ;
        int bef = 0 ;
        boolean yenFlag = false ;
        for( int i = off ; i < len ; i ++ ) {
            char c = base.charAt( i ) ;
            if( par != -1 || cote != -1 ) {
                if( cote != -1 ) {
                    if( bef != '\\' && c == cote ) {
                        yenFlag = false ;
                        cote = -1 ;
                    }
                    else if( c == '\\' && bef == '\\' ) {
                        yenFlag = true ;
                    }
                    else {
                        yenFlag = false ;
                    }
                }
                else if( par != -1 ) {
                    if( c == ed ) {
                        idx -- ;
                        if( idx <= 0 ) {
                            return i ;
                        }
                    }
                    else if( c == st ) {
                        idx ++ ;
                    }
                }
            }
            else {
                if( c == '\'' || c == '\"' ) {
                    cote = c ;
                }
                else if( c == st ) {
                    idx = 1 ;
                    par = c ;
                }
                else if( c == ed ) {
                    return i ;
                }
            }
            if( yenFlag ) {
                yenFlag = false ;
                bef = 0 ;
            }
            else {
                bef = c ;
            }
        }
        return -1 ;
    }
}
