package org.maachang.rimdb.table ;

import org.maachang.rimdb.index.IndexUtil;
import org.maachang.rimdb.table.array.ColumnArray;

/**
 * カラムテーブル.
 * 
 * @version 2014/07/07
 * @author  masahito suzuki
 * @since   rimdb-1.00
 */
@SuppressWarnings("unchecked")
public final class Columns {
    
    protected final ColumnArray[] list ;
    protected final int[] types ;
    protected final ColumnName[] columns ;
    protected final String[] columnNames ;
    protected final int length ;
    
    /**
     * コンストラクタ.
     * @param list 対象のカラムリストを設定します.
     * @param types 対象のカラムタイプ群を設定します.
     * @param columns 対象のカラム名リストを設定します.
     */
    public Columns( ColumnArray[] list,int[] types,ColumnName[] columns ) {
        final int len = list.length ;
        final String[] columnNames = new String[ len ] ;
        for( int i = 0 ; i < len ; i ++ ) {
            columnNames[ columns[ i ].no ] = columns[ i ].name ;
        }
        this.list = list ;
        this.columns = columns ;
        this.types = types ;
        this.columnNames = columnNames ;
        this.length = list.length ;
    }
    
    /**
     * 指定カラム名の項番を取得.
     * @param name 対象のカラム名を設定します.
     * @return int カラム項番が返却されます.
     */
    public final int getColumnNo( final String name ) {
        final int p = IndexUtil.search( columns,name ) ;
        if( p == -1 ) {
            return -1 ;
        }
        return columns[ p ].no ;
    }
    
    /**
     * カラム数を取得.
     * @return int カラム数が返却されます.
     */
    public final int length() {
        return length ;
    }
    
    /**
     * カラム名を取得.
     * @param no 対象のカラム項番を設定します.
     * @return String カラム名が返却されます.
     */
    public final String getColumnName( final int no ) {
        return columnNames[ no ] ;
    }
    
    /**
     * カラムタイプを取得.
     * @param no 対象のカラム項番を設定します.
     * @return int カラムタイプが返却されます.
     */
    public final int getType( final int no ) {
        return types[ no ] ;
    }
    
    /**
     * カラムタイプを取得.
     * @param name 対象のカラム名を設定します.
     * @return int カラムタイプが返却されます.
     *             [-1]の場合、取得に失敗しました.
     */
    public final int getType( final String name ) {
        final int p = getColumnNo( name ) ;
        if( p != -1 ) {
            return types[ p ] ;
        }
        return -1 ;
    }
    
    /**
     * カラムリストを取得.
     * @param no 対象のカラム項番を設定します.
     * @return ColumnArray カラムリストが返却されます.
     */
    public ColumnArray getColumnArray( final int no ) {
        return list[ no ] ;
    }
    
    /**
     * カラムリストを取得.
     * @param name 対象のカラム名を設定します.
     * @return ColumnArray カラムリストが返却されます.
     */
    public ColumnArray getColumnArray( final String name ) {
        final int p = getColumnNo( name ) ;
        if( p != -1 ) {
            return list[ p ] ;
        }
        return null ;
    }
    
}
