package org.maachang.rimdb.index.string ;

import org.maachang.rimdb.RimDbException;
import org.maachang.rimdb.util.OList;

/**
 * Like文の解析.
 * 
 * @version 2014/07/04
 * @author  masahito suzuki
 * @since   rimdb-1.00
 */
public final class LikeAnalyzer {
    
    private LikeAnalyzer() {}
    
    /**
     * Like文の解析.
     * Like検索は、[%,_]をサポート.
     * [%][*]は不明な複数文字が設定されている内容.
     * [_][?]は不明な１文字が設定されている内容.
     * @param like Like構文を設定します.
     * @return LikeParser Like構文がパースされた内容が返却されます.
     * @exception Exception 例外.
     */
    public static final LikeParser parser( final String like ) throws Exception {
        if( like == null || like.length() <= 0 ) {
            throw new RimDbException( "Likeパラメータが存在しません" ) ;
        }
        
        StringBuilder buf = null ;
        char c ;
        boolean yenFlag = false ;
        int type = -1 ;
        int pos = 0 ;
        final int len = like.length() ;
        final OList<Object> ret = new OList<Object>() ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            c = like.charAt( i ) ;
            
            // ターゲットポイント.
            if( !yenFlag && ( c == '%' || c == '*' || c == '_' || c == '?' ) ) {
                
                // データが存在する場合.
                if( buf != null ) {
                    if( ret.size() == 0 ) {
                        ret.add( LikeParser.POS_FIRST ) ;
                        // 初期化.
                        type = -1 ;
                        pos = 0 ;
                    }
                    ret.add( buf.toString() ) ;
                    buf = null ;
                }
                if( c == '%' || c == '*' ) {
                    if( type != 0 ) {
                        ret.add( LikeParser.POS_BETWEEN ) ;
                    }
                    type = 0 ; // 文字数無限.
                    pos = 0 ;
                }
                else if( c == '_' || c == '?' ) {
                    if( type == -1 ) {
                        type = 1 ; // 文字数指定.
                        pos = 1 ;
                    }
                    // 連続文字数指定.
                    else if( type == 1 ) {
                        pos ++ ;
                    }
                }
            }
            // 検索文字列.
            else {
                
                if( c == '\\' ) {
                    yenFlag = true ;
                    continue ;
                }
                if( buf == null ) {
                    if( type == 1 ) { // 文字数指定
                        ret.add( pos ) ;
                    }
                    buf = new StringBuilder() ;
                }
                buf.append( c ) ;
                
                // 初期化.
                type = -1 ;
                pos = 0 ;
            }
            // \\ではない.
            yenFlag = false ;
        }
        // 後処理.
        if( type == 0 ) { // 文字数無限.
            if( buf != null ) {
                ret.add( buf.toString() ) ;
                buf = null ;
            }
            if( (Integer )ret.get( ret.size() -1 ) != LikeParser.POS_BETWEEN ) {
                ret.add( LikeParser.POS_BETWEEN ) ;
            }
        }
        else if( type == 1 && pos > 0 ) { // 文字数指定.
            if( buf != null ) {
                ret.add( buf.toString() ) ;
                buf = null ;
            }
            ret.add( pos ) ;
        }
        else if( buf != null ) { // 検索文字が残っている.
            if( ret.size() == 0 ) {
                ret.add( LikeParser.POS_FIRST ) ;
            }
            ret.add( LikeParser.POS_LAST ) ;
            ret.add( buf.toString() ) ;
            buf = null ;
        }
        return new LikeParser( ret.getArray(),like ) ;
    }
    
    /**
     * マッチしているかチェック.
     * @param parser パーサーを設定します.
     * @param value 文字列を設定します.
     * @return boolean [true]の場合、マッチしています.
     * @exception Exception 例外.
     */
    public static final boolean match( final LikeParser parser,final String value )
        throws Exception {
        if( value == null || value.length() <= 0 ) {
            return false ;
        }
        final Object[] ps = parser.parser ;
        final int len = ps.length ;
        // [特殊条件]内容が１文字の場合は、％指定のみか、_文字数定義.
        if( len == 1 ) {
            final int n = (Integer)ps[ 0 ] ;
            if( n == LikeParser.POS_BETWEEN ) {
                return true ;
            }
            // 長さ一致.
            return value.length() == n ;
        }
        // [特殊条件]パース条件が3文字の場合.
        if( len == 3 ) {
            // Like構文ではなく、完全一致条件の場合.
            if( ps[ 0 ] instanceof Integer &&
                ps[ 1 ] instanceof Integer &&
                ps[ 2 ] instanceof String &&
                ( Integer )ps[ 0 ] == LikeParser.POS_FIRST &&
                ( Integer )ps[ 1 ] == LikeParser.POS_LAST ) {
                return value.equals( ps[ 2 ] ) ;
            }
        }
        // 通常パース条件.
        int p = 0 ;
        int b = 0 ;
        int n = -99 ; // 定義不明.
        boolean last = false ;
        String bw = null ;
        String v ;
        for( int i = 0 ; i < len ; i ++ ) {
            // 条件指定.
            if( ps[ i ] instanceof Integer ) {
                // 最後方一致条件の場合.
                if( (Integer)ps[ i ] == LikeParser.POS_LAST ) {
                    last = true ;
                }
                // 通常条件の場合.
                else {
                    n = (Integer)ps[ i ] ;
                }
            }
            // 文字指定.
            else {
                v = (String)ps[ i ] ;
                // 文字指定の場合.
                if( n > 0 ) {
                    if( bw == null ) {
                        if( value.length() < b + n + v.length() ||
                            !v.equals( value.substring( b + n,b + n + v.length() ) ) ) {
                            return false ;
                        }
                    }
                    else {
                        while( true ) {
                            if( value.length() < b + n + v.length() ||
                                !v.equals( value.substring( b + n,b + n + v.length() ) ) ) {
                                // 見つからない場合は、以前の部分検索条件が見つかるまで検索.
                                // 見つかった場合は、再度文字指定検索を再開.
                                if( ( p = value.indexOf( bw,b ) ) == -1 ) {
                                    return false ;
                                }
                                b = p + bw.length() ;
                            }
                            else {
                                break ;
                            }
                        }
                    }
                    // 最後方一致条件の場合は、現在の検索長と、文字列長が一致チェックで終了.
                    if( last ) {
                        return ( b + n + v.length() == value.length() ) ;
                    }
                    else {
                        b = b + n + v.length() ;
                    }
                    n = -99 ; // 定義不明.
                    bw = null ; // 前回のindexof条件をクリア.
                }
                // 条件指定の場合.
                // ただし最後方一致の場合は、そちらを優先させる.
                else if( !last ) {
                    switch( n ) {
                        case LikeParser.POS_FIRST : // 先頭一致.
                            if( !value.startsWith( v ) ) {
                                return false ;
                            }
                            b = v.length() ;
                            break ;
                        case LikeParser.POS_BETWEEN : // 次の文字列が一致.
                            if( ( p = value.indexOf( v,b ) ) == -1 ) {
                                return false ;
                            }
                            bw = v ; // 前回のindexof条件を保持.
                            b = p + v.length() ;
                            break ;
                        case -99 :
                            throw new RimDbException( "Like構文が不正です" ) ;
                    }
                }
                // 最後方一致条件の場合.
                if( last ) {
                    return value.endsWith( v ) ;
                }
            }
        }
        // 文字指定の場合.
        if( n > 0 ) {
            if( bw == null ) {
                if( value.length() != b + n ) {
                    return false ;
                }
            }
            else {
                while( true ) {
                    if( value.length() != b + n ) {
                        // 見つからない場合は、以前の部分検索条件が見つかるまで検索.
                        // 見つかった場合は、文字長一致検索を再開.
                        if( ( p = value.indexOf( bw,b ) ) == -1 ) {
                            return false ;
                        }
                        b = p + bw.length() ;
                    }
                    else {
                        break ;
                    }
                }
            }
        }
        return true ;
    }
    
}

