package org.maachang.mimdb.core;

import org.maachang.mimdb.server.ConnectionDefine;


/**
 * コネクションオブジェクト.
 * 
 * @version 2014/01/16
 * @author masahito suzuki
 * @since MasterInMemDB 1.02
 */
public abstract class MimdbConnection {
    
    /**
     * 組み込み用コネクションオブジェクトを生成.
     * @return MimdbConnection コネクションオブジェクトが返却されます.
     */
    public static final MimdbConnection create() {
        return new EmbeddedConnection() ;
    }
    
    /**
     * サーバー接続コネクションオブジェクトを生成.
     * @param addr 対象のIPアドレスを設定します.
     * @return MimdbConnection コネクションオブジェクトが返却されます.
     * @exception Exception 例外.
     */
    public static final MimdbConnection create( String addr )
        throws Exception {
        return create( -1,addr,ConnectionDefine.DEF_PORT ) ;
    }
    
    /**
     * サーバー接続コネクションオブジェクトを生成.
     * @param timeout コネクションタイムアウト値をミリ秒で設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @return MimdbConnection コネクションオブジェクトが返却されます.
     * @exception Exception 例外.
     */
    public static final MimdbConnection create( int timeout,String addr )
        throws Exception {
        return create( timeout,addr,ConnectionDefine.DEF_PORT ) ;
    }
    
    /**
     * サーバー接続コネクションオブジェクトを生成.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return MimdbConnection コネクションオブジェクトが返却されます.
     * @exception Exception 例外.
     */
    public static final MimdbConnection create( String addr,int port )
        throws Exception {
        return create( -1,addr,port ) ;
    }
    
    /**
     * サーバー接続コネクションオブジェクトを生成.
     * @param timeout コネクションタイムアウト値をミリ秒で設定します.
     * @param addr 対象のIPアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return MimdbConnection コネクションオブジェクトが返却されます.
     * @exception Exception 例外.
     */
    public static final MimdbConnection create( int timeout,String addr,int port )
        throws Exception {
        return new RemoteConnection( addr,port,timeout ) ;
    }
    
    /**
     * オブジェクトクローズ.
     */
    public abstract void close() ;
    
    /**
     * オブジェクトがクローズしているか取得.
     * @return boolean [true]の場合、クローズしています.
     */
    public abstract boolean isClose() ;
    
    /**
     * テーブルオブジェクトを取得
     * @param name 対象のテーブル名を設定します.
     * @return BaseTable テーブルオブジェクトが返却されます.
     * @exception Exception 例外.
     */
    public abstract BaseTable getTable( String name )
        throws Exception ;
    
    /**
     * ステートメントオブジェクトを取得.
     * @exception Exception 例外.
     */
    public abstract MimdbStatement getStatement()
        throws Exception ;
    
    /**
     * プリコンパイル済みのステートメントオブジェクトを取得.
     * @param sql 対象のSQL文を設定します.
     * @return MimdbPreparedStatement プリコンパイル済みオブジェクトが返却されます.
     * @exception Exception 例外.
     */
    public abstract MimdbPreparedStatement getPreparedStatement( String sql )
        throws Exception ;
    
    /**
     * タイムアウト値を設定.
     * @param timeout タイムアウト値を設定します.
     */
    public void setTimeout( int timeout ) {
    }
    
    /**
     * タイムアウト値を取得.
     * @return int タイムアウト値が返却されます.
     */
    public int getTimeout() {
        return 0 ;
    }
}

