package org.maachang.jni.io;

import java.io.IOException;
import java.io.InputStream;

/**
 * NativeInputStream.
 * 
 * @version 2010/06/04
 * @author  masahito suzuki
 * @since   SeabassNativeIO-1.0.0
 */
public class NativeInputStream extends InputStream {
    private static final int ACCESS_MODE = NativeIODefine.MODE_SEQ ;
    private static final int IO_OPT = NativeIODefine.OPT_READ ;
    private static final int TEMP_LEN = 4096 ;
    private long address = 0L ;
    private int position = 0 ;
    private int length = 0 ;
    private long handle = -1L ;
    private long fileSize = 0L ;
    private long allPos = 0L ;
    
    private NativeInputStream() {
        
    }
    
    /**
     * コンストラクタ.
     * @param name 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public NativeInputStream( String name ) throws Exception {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( NativeIODefine.useFile( name ) == false ) {
            throw new NotFileException( "指定ファイル[" + name + "]は存在しません" ) ;
        }
        name = NativeIODefine.fullPath( name ) ;
        this.handle = NativeIO.open( ACCESS_MODE,NativeIODefine.OPEN_ALWAYS,IO_OPT,NativeIODefine.nativeString( name ) ) ;
        if( this.handle == -1L ) {
            throw new IOException( "指定ファイル[" + name + "]のオープンに失敗しました" ) ;
        }
        this.fileSize = NativeIO.getLength( this.handle ) ;
        this.address = DirectMemoryIO.malloc( TEMP_LEN ) ;
    }
    
    protected void finalize() throws Exception {
        try {
            close() ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * ファイルクローズ.
     */
    public void close() {
        if( this.handle > -1L ) {
            NativeIO.close( this.handle ) ;
            this.handle = -1L ;
            this.position = 0 ;
            this.length = -1 ;
            this.fileSize = -1L ;
            this.allPos = -1L ;
            if ( this.address != 0L ) {
                DirectMemoryIO.free( this.address ) ;
                this.address = 0L ;
            }
        }
    }
    
    /**
     * 残り容量を取得.
     * @return int 残り容量が返されます.
     * @exception IOException I/O例外.
     */
    public int available() throws IOException {
        if( handle <= -1L ) {
            throw new IOException( "ファイルは既にクローズしています" ) ;
        }
        return ( int )( ( fileSize - allPos ) & 0x000000007fffffffL ) ;
    }
    
    /**
     * 読み込み処理.
     * @return int 読み込まれた情報が返されます.
     * @exception IOException I/O例外.
     */
    public int read() throws IOException {
        if( handle <= -1L ) {
            throw new IOException( "ファイルは既にクローズしています" ) ;
        }
        if( length <= -1 ) {
            return -1 ;
        }
        if( position >= length ) {
            length = NativeIO.sRead( handle,address,TEMP_LEN ) ;
            position = 0 ;
            if( length <= 0 ) {
                length = -1 ;
                return -1 ;
            }
        }
        int ret = ( int )( DirectMemoryIO.get( address,position ) & 0x000000ff ) ;
        position ++ ;
        allPos ++ ;
        return ret ;
    }
    
    /**
     * 読み込み処理.
     * @param b 読み込み対象のバイナリを設定します.
     * @return int 読み込まれた情報が返されます.
     * @exception IOException I/O例外.
     */
    public int read(byte b[]) throws IOException {
        if (b == null) {
            throw new NullPointerException( "バイナリが存在しません" );
        }
        return read(b, 0, b.length);
    }
    
    /**
     * 読み込み処理.
     * @param b 対象のバイナリを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param len 対象の長さを設定します.
     * @return int 読み込み長が返されます.
     * @exception IOException I/O例外.
     */
    public int read(byte b[], int off, int len) throws IOException {
        if( handle <= -1L ) {
            throw new IOException( "ファイルは既にクローズしています" ) ;
        }
        if( length <= -1 ) {
            return -1 ;
        }
        if (b == null) {
            throw new NullPointerException( "バイナリが存在しません" );
        } else if (off < 0 || len < 0 || len+off > b.length) {
            throw new IndexOutOfBoundsException( "指定範囲が不正です" );
        }
        int plen ;
        int cnt = 0 ;
        while( true ) {
            if( len <= 0 ) {
                return cnt ;
            }
            if( position >= length ) {
                position = 0 ;
                if( (length = NativeIO.sRead( handle,address,TEMP_LEN )) <= 0 ) {
                    length = -1 ;
                    return cnt ;
                }
                else if( len > length ) {
                    plen = length ;
                }
                else {
                    plen = len ;
                }
            }
            else {
                plen = length - position ;
                if( len < plen ) {
                    plen = len ;
                }
            }
            DirectMemoryIO.getBinary( address,position,b,off,plen ) ;
            position += plen ;
            off += plen ;
            len -= plen ;
            cnt += plen ;
        }
    }
    
    /**
     * NativeBufferで読み込み処理.
     * <p>※このメソッドと、通常のreadメソッドを併用しないでください</p>
     * @param buf 対象のNativeBufferを設定します.
     * @return int 読み込みサイズが返されます.
     */
    public int readBuffer( NativeBuffer buf ) throws IOException {
        if( buf == null || buf.isClear() ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( handle <= -1L ) {
            throw new IOException( "ファイルは既にクローズしています" ) ;
        }
        return NativeIO.sRead( handle,buf.getAddress(),buf.getLength() ) ;
    }
    
    /**
     * NativeBufferで読み込み処理.
     * <p>※このメソッドと、通常のreadメソッドを併用しないでください</p>
     * @param buf 対象のNativeBufferを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param len 対象の長さを設定します.
     * @return int 読み込みサイズが返されます.
     */
    public int readBuffer( NativeBuffer buf,int off,int len ) throws IOException {
        int bufLen ;
        if( buf == null || buf.isClear() || ( bufLen = buf.getLength() ) <= 0 || off < 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        len = ( len <= 0 ) ? bufLen - off : len ;
        if( len <= 0 || len > bufLen ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( handle == -1L ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        return NativeIO.sRead( handle,buf.getAddress()+off,len ) ;
    }
    
    /**
     * markサポート.
     * @return boolean [false]が返されます.
     */
    public boolean markSupported() {
        return false ;
    }
    
    /**
     * 現在のファイルサイズを取得.
     * @return long ファイルサイズが返されます.
     * @exception Exception 例外.
     */
    public long length() throws Exception {
        if( handle == -1L ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        return NativeIO.getLength( handle ) ;
    }
}
