package org.maachang.jni.io ;

/**
 * 状態チェックを無視するNativeBuffer.
 * <p>このバッファは、速度重視を考慮した、チェック処理なしで、
 *    Nativeメモリにアクセスするオブジェクトです.
 *    欠点として、ダイレクトにNativeメモリにアクセスするので、
 *    このオブジェクトによって操作した結果によって、処理異常が
 *    発生した場合、最悪JavaVMが異常停止してしまいます.</p>
 * 
 * @version 2010/06/04
 * @author  masahito suzuki
 * @since   SeabassNativeIO-1.0.0
 */
public abstract class AbstractFastNativeBuffer implements NativeBuffer {
    protected static final boolean BIG_ENDIAN = Unsafe.BIG_ENDIAN ;
    protected static final boolean UnsafeMode = Unsafe.UNSAFE_MODE ;
    //protected static final boolean UnsafeMode = false ;
    protected static final sun.misc.Unsafe unsafe = Unsafe.unsafe ;
    protected long address = 0L ;
    protected int length = 0 ;
    
    /**
     * メモリ先頭アドレスを取得.
     * @return long メモリの先頭アドレスが返されます.
     */
    public long getAddress() {
        return address ;
    }
    
    /**
     * メモリサイズを取得.
     * @return int メモリサイズが返されます.
     */
    public int getLength() {
        return length ;
    }
    
    /**
     * メモリクリアチェック.
     * @return boolean [true]の場合、メモリはクリアされています.
     */
    public boolean isClear() {
        return address == 0L ;
    }
    
    /**
     * byte情報を設定.
     * @param index 対象のインデックス位置を設定します.
     * @param value 設定対象の情報を設定します.
     */
    public void putByte( final int index,final byte value ) {
        if( UnsafeMode ) {
            unsafe.putByte( address+index,value ) ;
        }
        else {
            NativeIO.putByte( address+index,value ) ;
        }
    }
    
    /**
     * byte情報を取得.
     * @param index 対象のインデックス位置を設定します.
     * @return byte 情報が返されます.
     */
    public byte getByte( final int index ) {
        if( UnsafeMode ) {
            return unsafe.getByte( address+index ) ;
        }
        return NativeIO.getByte( address+index ) ;
    }
    
    /** binaryコピーしきい値. **/
    private static final int BINARY_COPY_TO_PUT_BYTE = DirectMemoryIO.BINARY_COPY_TO_PUT_BYTE ;
    
    /**
     * binary情報を設定.
     * @param index 対象のインデックス位置を設定します.
     * @param value 設定対象の情報を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 設定された長さが返されます.
     */
    public int putBinary( final int index,final byte[] value,final int offset,final int length ) {
        // binaryコピーしきい値範囲内の場合は、byte単位で処理.
        if( length <= BINARY_COPY_TO_PUT_BYTE ) {
            if( UnsafeMode ) {
                long j = address+index ;
                for( int i = offset ; i < length ; i ++,j ++ ) {
                    unsafe.putByte( j,value[ i ] ) ;
                }
            }
            else {
                long j = address+index ;
                for( int i = offset ; i < length ; i ++,j ++ ) {
                    NativeIO.putByte( j,value[ i ] ) ;
                }
            }
            return length ;
        }
        NativeIO.putBinary( address+index,value,offset,length ) ;
        return length ;
    }
    
    /**
     * binary情報を取得.
     * @param index 対象のインデックス位置を設定します.
     * @param value 取得対象の情報を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 設定された長さが返されます.
     */
    public int getBinary( final int index,final byte[] value,final int offset,final int length ) {
        // binaryコピーしきい値範囲内の場合は、byte単位で処理.
        if( length <= DirectMemoryIO.BINARY_COPY_TO_PUT_BYTE ) {
            if( UnsafeMode ) {
                long j = address+address ;
                for( int i = offset ; i < length ; i ++,j ++ ) {
                    value[ i ] = unsafe.getByte( j ) ;
                }
            }
            else {
                long j = address+address ;
                for( int i = offset ; i < length ; i ++,j ++ ) {
                    value[ i ] = NativeIO.getByte( j ) ;
                }
            }
            return length ;
        }
        NativeIO.getBinary( address+index,value,offset,length ) ;
        return length ;
    }
    
    /**
     * boolean設定.
     * @param index 対象のインデックス位置を設定します.
     * @param value 設定対象の情報を設定します.
     */
    public void putBoolean( final int index,final boolean value ) {
        if( UnsafeMode ) {
            unsafe.putByte( address+index,((value)?(byte)1:(byte)0) ) ;
        }
        else {
            NativeIO.putByte( address+index,((value)?(byte)1:(byte)0) ) ;
        }
    }
    
    /**
     * boolean取得.
     * @param index 対象のインデックス位置を設定します.
     * @return boolean 情報が返されます.
     */
    public boolean getBoolean( final int index ) {
        if( UnsafeMode ) {
            return ( unsafe.getByte( address+index ) == 0 ) ? false : true ;
        }
        return ( NativeIO.getByte( address+index ) == 0 ) ? false : true ;
    }
    
    /**
     * char設定.
     * @param index 対象のインデックス位置を設定します.
     * @param value 設定対象の情報を設定します.
     */
    public void putChar( final int index,char value ) {
        if( !BIG_ENDIAN ) {
            value = Unsafe.swap( value ) ;
        }
        if( UnsafeMode ) {
            unsafe.putChar( address+index,value ) ;
        }
        else {
            NativeIO.putChar( address+index,value ) ;
        }
    }
    
    /**
     * char取得.
     * @param index 対象のインデックス位置を設定します.
     * @return char 情報が返されます.
     */
    public char getChar( final int index ) {
        if( UnsafeMode ) {
            if( BIG_ENDIAN ) {
                return unsafe.getChar( address+index ) ;
            }
            return Unsafe.swap( unsafe.getChar( address+index ) ) ;
        }
        if( BIG_ENDIAN ) {
            return NativeIO.getChar( address+index ) ;
        }
        return Unsafe.swap( NativeIO.getChar( address+index ) ) ;
    }
    
    /**
     * short設定.
     * @param index 対象のインデックス位置を設定します.
     * @param value 設定対象の情報を設定します.
     */
    public void putShort( final int index,short value ) {
        if( !BIG_ENDIAN ) {
            value = Unsafe.swap( value ) ;
        }
        if( UnsafeMode ) {
            unsafe.putShort( address+index,value ) ;
        }
        else {
            NativeIO.putShort( address+index,value ) ;
        }
    }
    
    /**
     * short取得.
     * @param index 対象のインデックス位置を設定します.
     * @return short 情報が返されます.
     */
    public short getShort( final int index ) {
        if( UnsafeMode ) {
            if( BIG_ENDIAN ) {
                return unsafe.getShort( address+index ) ;
            }
            return Unsafe.swap( unsafe.getShort( address+index ) ) ;
        }
        if( BIG_ENDIAN ) {
            return NativeIO.getShort( address+index ) ;
        }
        return Unsafe.swap( NativeIO.getShort( address+index ) ) ;
    }
    
    /**
     * int設定.
     * @param index 対象のインデックス位置を設定します.
     * @param value 設定対象の情報を設定します.
     */
    public void putInt( final int index,int value ) {
        if( !BIG_ENDIAN ) {
            value = Unsafe.swap( value ) ;
        }
        if( UnsafeMode ) {
            unsafe.putInt( address+index,value ) ;
        }
        else {
            NativeIO.putInt( address+index,value ) ;
        }
    }
    
    /**
     * int取得.
     * @param index 対象のインデックス位置を設定します.
     * @return int 情報が返されます.
     */
    public int getInt( final int index ) {
        if( UnsafeMode ) {
            if( BIG_ENDIAN ) {
                return unsafe.getInt( address+index ) ;
            }
            return Unsafe.swap( unsafe.getInt( address+index ) ) ;
        }
        if( BIG_ENDIAN ) {
            return NativeIO.getInt( address+index ) ;
        }
        return Unsafe.swap( NativeIO.getInt( address+index ) ) ;
    }
    
    /**
     * long設定.
     * @param index 対象のインデックス位置を設定します.
     * @param value 設定対象の情報を設定します.
     */
    public void putLong( final int index,long value ) {
        if( !BIG_ENDIAN ) {
            value = Unsafe.swap( value ) ;
        }
        if( UnsafeMode ) {
            unsafe.putLong( address+index,value ) ;
        }
        else {
            NativeIO.putLong( address+index,value ) ;
        }
    }
    
    /**
     * long取得.
     * @param index 対象のインデックス位置を設定します.
     * @return long 情報が返されます.
     */
    public long getLong( final int index ) {
        if( UnsafeMode ) {
            if( BIG_ENDIAN ) {
                return unsafe.getLong( address+index ) ;
            }
            return Unsafe.swap( unsafe.getLong( address+index ) ) ;
        }
        if( BIG_ENDIAN ) {
            return NativeIO.getLong( address+index ) ;
        }
        return Unsafe.swap( NativeIO.getLong( address+index ) ) ;
    }
    
    /**
     * float設定.
     * @param index 対象のインデックス位置を設定します.
     * @param value 設定対象の情報を設定します.
     */
    public void putFloat( final int index,final float value ) {
        putInt( index,Float.floatToIntBits( value ) ) ;
    }
    
    /**
     * float取得.
     * @param index 対象のインデックス位置を設定します.
     * @return float 情報が返されます.
     */
    public float getFloat( final int index ) {
        return Float.intBitsToFloat( getInt( index ) ) ;
    }
    
    /**
     * double設定.
     * @param index 対象のインデックス位置を設定します.
     * @param value 設定対象の情報を設定します.
     */
    public void putDouble( final int index,final double value ) {
        putLong( index,Double.doubleToLongBits( value ) ) ;
    }
    
    /**
     * double取得.
     * @param index 対象のインデックス位置を設定します.
     * @return double 情報が返されます.
     */
    public double getDouble( final int index ) {
        return Double.longBitsToDouble( getLong( index ) ) ;
    }
    
    /**
     * IndexOf.
     * @param binary 検索対象バイナリを設定します.
     * @return int 操作位置が返されます.
     */
    public int indexOf( final byte[] binary ) {
        return indexOf( binary,0 ) ;
    }
    
    /**
     * IndexOf.
     * @param binary 検索対象バイナリを設定します.
     * @param index 対象の開始位置を設定します.
     * @return int 操作位置が返されます.
     */
    public int indexOf( final byte[] binary,final int index ) {
        int len ;
        if( binary == null || ( len = binary.length ) <= 0 ) {
            return -1 ;
        }
        else if( index < 0 || index + len > this.length ) {
            return -1 ;
        }
        int ret = NativeIO.indexOf( address+index,this.length-index,binary,len ) ;
        if( ret == -1 ) {
            return -1 ;
        }
        return ret + index ;
    }
    
    /**
     * lastIndexOf.
     * @param binary 操作対象のデータを設定します.
     * @return int 操作位置が返されます.
     */
    public int lastIndexOf( final byte[] binary ) {
        return lastIndexOf( binary,this.length - 1 ) ;
    }
    
    /**
     * lastIndexOf.
     * @param binary 操作対象のデータを設定します.
     * @param index 対象の開始位置を設定します.
     * @return int 操作位置が返されます.
     */
    public int lastIndexOf( final byte[] binary,final int index ) {
        int len ;
        if( binary == null || ( len = binary.length ) <= 0 ) {
            return -1 ;
        }
        else if( index < 0 || index + len > this.length ) {
            return -1 ;
        }
        return NativeIO.lastIndexOf( address,index,this.length,binary,len ) ;
    }
}

