﻿using System;

namespace Framework
{
    /// <summary>
    /// double の拡張メソッドです。
    /// </summary>
    public static class DoubleExtension
    {
        /// <summary>
        /// 指定したミリ秒数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Ms(this double target)
        {
            return TimeSpan.FromMilliseconds(target);
        }

        /// <summary>
        /// 指定した秒数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Second(this double target)
        {
            return TimeSpan.FromSeconds(target);
        }

        /// <summary>
        /// 指定した秒数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Seconds(this double target)
        {
            return TimeSpan.FromSeconds(target);
        }

        /// <summary>
        /// 指定した分数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Minute(this double target)
        {
            return TimeSpan.FromMinutes(target);
        }

        /// <summary>
        /// 指定した分数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Minutes(this double target)
        {
            return TimeSpan.FromMinutes(target);
        }

        /// <summary>
        /// 指定した時間数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Hour(this double target)
        {
            return TimeSpan.FromHours(target);
        }

        /// <summary>
        /// 指定した時間数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Hours(this double target)
        {
            return TimeSpan.FromHours(target);
        }

        /// <summary>
        /// 指定した日数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Day(this double target)
        {
            return TimeSpan.FromDays(target);
        }

        /// <summary>
        /// 指定した日数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Days(this double target)
        {
            return TimeSpan.FromDays(target);
        }

        /// <summary>
        /// 指定した週数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Week(this double target)
        {
            return TimeSpan.FromDays(target * 7);
        }

        /// <summary>
        /// 指定した週数を表す System.TimeSpan を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static TimeSpan Weeks(this double target)
        {
            return TimeSpan.FromDays(target * 7);
        }

        /// <summary>
        /// Convert.ToInt32(target)を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static int ToInt(this double target)
        {
            return Convert.ToInt32(target);
        }

        /// <summary>
        /// Math.Round(target)を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static double Round(this double target)
        {
            return Math.Round(target);
        }

        /// <summary>
        /// Math.Round(target, digits)を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="digits"></param>
        /// <returns></returns>
        public static double Round(this double target, int digits)
        {
            return Math.Round(target, digits);
        }

        /// <summary>
        /// Math.Floor(target)を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static double Floor(this double target)
        {
            return Math.Floor(target);
        }

        /// <summary>
        /// 有効桁数digits桁で切り捨てます。digitsに負の値を指定した場合は指定した桁の分が0になります。
        /// ただし最上位の1桁は0になりません。
        /// 1234.Floor(-2) ⇒ 1200, 1234.Floor(-5) ⇒ 1000, 0.004321.Floor(-2) ⇒ 0.0043, 0.004321.Floor(-10) ⇒ 0.004
        /// </summary>
        /// <param name="target"></param>
        /// <param name="digits">有効桁数</param>
        /// <returns></returns>
        public static double Floor(this double target, int digits)
        {
            if (digits > 0)
            {
                var e = Math.Pow(10d, digits);
                return Math.Floor(target * e) / e;
            }
            if (digits < 0)
            {
                //有効桁数が負の値の場合

                digits = Math.Abs(digits);

                if (target < 1)
                {
                    //targetが１より小さい場合

                    var str = target.ToString().Substring(2);
                    var len = str.Length;
                    var len2 = str.TrimStart('0').Length;

                    var pow = (digits >= len2) ? len -len2 + 1 : len - digits;
                    var e = Math.Pow(10d, pow);

                    return Math.Floor(target * e) / e;
                }
                else
                {
                    //targetが１より大きい場合

                    var e = 1d;
                    for (int i = 0; i < digits; i++)
                    {
                        if (e * 10d > target) break;
                        e *= 10d;
                    }
                    return Math.Floor(target / e) * e;
                }
            }
            return Math.Floor(target);
        }

        /// <summary>
        /// 基準値mの倍数に近い値に切り捨てます。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="m">基準値</param>
        /// <returns></returns>
        public static double FloorM(this double target, int m)
        {
            return (target / m).Floor() * m;
        }

        /// <summary>
        /// Math.Ceiling(target)を返します
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static double Ceiling(this double target)
        {
            return Math.Ceiling(target);
        }

        /// <summary>
        /// 有効桁数digits桁で切り上げます。digitsに負の値を指定した場合は指定した桁の分が0になり、その上位一桁を切り上げます。
        /// ただし最上位の1桁は0になりません。
        /// 1234.Floor(2) ⇒ 1300, 1234.Floor(5) ⇒ 2000
        /// </summary>
        /// <param name="target"></param>
        /// <param name="digits">有効桁数</param>
        /// <returns></returns>
        public static double Ceiling(this double target, int digits)
        {
            if (digits > 0)
            {
                var e = Math.Pow(10d, digits);
                return Math.Ceiling(target * e) / e;
            }
            if (digits < 0)
            {
                var e = 1d;
                for (int i = 0; i > digits; i--)
                {
                    if (e * 10d > target) break;
                    e *= 10d;
                }
                return Math.Ceiling(target / e) * e;
            }
            return Math.Ceiling(target);
        }

        /// <summary>
        /// 基準値mの倍数に近い値に切り上げる。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="m">基準値</param>
        /// <returns></returns>
        public static double CeilingM(this double target, int m)
        {
            return (target / m).Ceiling() * m;
        }

        /// <summary>
        /// targetとcompareの大きいほうを返します。
        /// どちらかがNaNの場合はNaNを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static double Max(this double target, double compare)
        {
            return Math.Max(target, compare);
        }

        /// <summary>
        /// targetとcompareの小さいほうを返します。
        /// どちらかがNaNの場合はNaNを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static double Min(this double target, double compare)
        {
            return Math.Min(target, compare);
        }

        /// <summary>
        /// targetとcompareの大きいほうを返します。
        /// どちらかがNaNの場合はNaNを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static double Max(this double target, params double[] compare)
        {   
            var val = target;
            foreach (double comp in compare)
            {
                val = Math.Max(val, comp);
            }

            return val;
        }

        /// <summary>
        /// targetとcompareの小さいほうを返します。
        /// どちらかがNaNの場合はNaNを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static double Min(this double target, params double[] compare)
        {
            var val = target;
            foreach (double comp in compare)
            {
                val = Math.Min(val, comp);
            }

            return val;
        }

        /// <summary>
        /// targetとcompareの大きいほうを返します。
        /// どちらかがNaNの場合はNaNを無視します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static double MaxIgnoreNaN(this double target, double compare)
        {
            if (target.IsNaN())
                return compare;

            if (compare.IsNaN())
                return target;

            return target.Max(compare);
        }

        /// <summary>
        /// targetとcompareの小さいほうを返します。
        /// どちらかがNaNの場合はNaNを無視します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static double MinIgnoreNaN(this double target, double compare)
        {
            if (target.IsNaN())
                return compare;

            if (compare.IsNaN())
                return target;

            return target.Min(compare);
        }

        /// <summary>
        /// targetとcompareの大きいほうを返します。
        /// どちらかがNaNの場合はNaNを無視します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static double MaxIgnoreNaN(this double target, params double[] compare)
        {
            var val = target;
            foreach (double comp in compare)
            {
                val = val.MaxIgnoreNaN(comp);
            }

            return val;
        }

        /// <summary>
        /// targetとcompareの小さいほうを返します。
        /// どちらかがNaNの場合はNaNを無視します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static double MinIgnoreNaN(this double target, params double[] compare)
        {
            var val = target;
            foreach (double comp in compare)
            {
                val = val.MinIgnoreNaN(comp);
            }

            return val;
        }

        /// <summary>
        /// targetがdouble.Nanの場合はtrueを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static bool IsNaN(this double target)
        {
            return double.IsNaN(target);
        }

        /// <summary>
        /// targetがdouble.Nanでない場合はtrueを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static bool IsNotNaN(this double target)
        {
            return double.IsNaN(target) == false;
        }

        /// <summary>
        /// targetをdecimalにコンバートします。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static decimal ToDecimal(this double target)
        {
            return Convert.ToDecimal(target);
        }

        /// <summary>
        /// targetのn乗を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="n">乗数</param>
        /// <returns></returns>
        public static double Pow(this double target, int n)
        {
            if (n == 0) return 1;
            if (n == 1) return target;
            if (n == 2) return target * target;

            return Math.Pow(target, n);
            //var ret = target;
            //for (int i = 1; i < n / 2; i++)
            //{
            //    ret = ret * target;
            //}
            //ret = ret * ret;

            //if (n % 2 == 1)
            //{
            //    ret = ret * target;
            //}

            //return ret;
        }

        /// <summary>
        /// targetのd乗を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="n">乗数</param>
        /// <returns></returns>
        public static double Pow(this double target, double d)
        {
            return Math.Pow(target, d);
        }

        /// <summary>
        /// 数値が-e～eの範囲に含まれるかどうかを返します。
        /// 数値が-eもしくはeと等しい場合もtrueが返されます。
        /// </summary>
        public static bool Within(this double target, double e)
        {
            return -e <= target && target <= e;
        }

        /// <summary>
        /// 数値がfrom～toの範囲に含まれるかどうかを返します。
        /// 数値がfromもしくはtoと等しい場合もtrueが返されます。
        /// </summary>
        public static bool Within(this double target, double from, double to)
        {
            return from <= target && target <= to;
        }

        /// <summary>
        /// targetがプラスの場合は1、ゼロの場合は0、マイナスの場合は-1を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static int Sign(this double target)
        {
            if (target > 0) return 1;
            if (target < 0) return -1;
            return 0;
        }

        /// <summary>
        /// target が double.Nan の場合は defaultValue を返します。
        /// そうでない場合は target を返します。
        /// </summary>
        public static double DefaultIfNan(this double target, double defaultValue)
        {
            return double.IsNaN(target) ? defaultValue : target;
        }

        /// <summary>
        /// targetの平方根を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static double Sqrt(this double target)
        {
            return Math.Sqrt(target);
        }

        /// <summary>
        ///     指定した値が小数を含むかどうかを返します。</summary>
        /// <param name="target">
        ///     検査対象となる値。</param>
        /// <returns>
        ///     小数を含む場合は true。それ以外は false。</returns>
        public static bool IsDecimal(this double target)
        {
            if (target - target.Floor() != 0)
            {
                return true;
            }

            return false;
        }
    }
}
