﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Framework
{
    /// <summary>
    /// decimal の拡張メソッドです。
    /// </summary>
    public static class DecimalExtension
    {
        /// <summary>
        /// doubleに変換します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static double ToDouble(this decimal target)
        {
            return Convert.ToDouble(target);
        }

        /// <summary>
        /// -1を掛けた値を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static decimal Negate(this decimal target)
        {
            return decimal.Negate(target);
        }

        /// <summary>
        /// targetのn乗を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="n"></param>
        /// <returns></returns>
        public static decimal Pow(this decimal target, int n)
        {

            if (n > 0)
            {
                var ret = target;
                for (int i = 0; i < n - 1; i++) ret = ret * target; 
                return ret;
            }
            else if (n < 0)
            {
                var ret = 1m;
                for (int i = n; i < 0; i++) ret = ret / target;
                return ret;
            }
            else
            {
                return 1;
            }
        }

        /// <summary>
        /// targetのd乗を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="n"></param>
        /// <returns></returns>
        public static decimal Pow(this decimal target, double d)
        {
            return (decimal)Math.Pow((double)target, d);
        }

        /// <summary>
        /// 末尾の'0'や'.'を除去した文字列にして返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static string ToPlaneString(this decimal target)
        {
            if (target == 0) return "0";

            var str = target.ToString();
            if (str.IndexOf('.') < 0) return str;

            return str.TrimEnd('0').TrimEnd('.');
        }

        /// <summary>
        /// Math.Round(target, digits)を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="digits"></param>
        /// <returns></returns>
        public static decimal RoundEx(this decimal target, int digits)
        {
            //return Math.Round(target, digits);
            decimal dCoef = 10m.Pow(digits);

            return target > 0 ? ((target * dCoef) + 0.5m).FloorEx() / dCoef :
                                ((target * dCoef) - 0.5m).CeilingEx() / dCoef;

        }

        /// <summary>
        /// Math.Ceiling(target)を返します
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static decimal CeilingEx(this decimal target)
        {
            return Math.Ceiling(target);
        }

        /// <summary>
        /// 有効桁数digits桁で切り上げます。digitsに負の値を指定した場合は指定した桁の分が0になり、その上位一桁を切り上げます。
        /// ただし最上位の1桁は0になりません。
        /// 1234.Floor(2) ⇒ 1300, 1234.Floor(5) ⇒ 2000
        /// </summary>
        /// <param name="target"></param>
        /// <param name="digits"></param>
        /// <returns></returns>
        public static decimal CeilingEx(this decimal target, int digits)
        {
            if (digits > 0)
            {
                var e = (decimal)Math.Pow(10d, digits);
                return Math.Ceiling(target * e) / e;
            }
            if (digits < 0)
            {
                var e = 1m;
                for (int i = 0; i > digits; i--)
                {
                    if (e * 10m > target) break;
                    e *= 10m;
                }
                return Math.Ceiling(target / e) * e;
            }
            return Math.Ceiling(target);
        }

        /// <summary>
        /// targetとcompareの大きいほうを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static decimal Max(this decimal target, decimal compare)
        {
            return target > compare ? target : compare;
        }

        /// <summary>
        /// targetとcompareの小さいほうを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static decimal Min(this decimal target, decimal compare)
        {
            return target < compare ? target : compare;
        }

        /// <summary>
        /// Math.Floor(target)を返します
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static decimal FloorEx(this decimal target)
        {
            return Math.Floor(target);
        }

        /// <summary>
        /// 有効桁数digits桁で切り捨てます。digitsに負の値を指定した場合は指定した桁の分が0になります。
        /// ただし最上位の1桁は0になりません。
        /// 1234.Floor(-2) ⇒ 1200, 1234.Floor(-5) ⇒ 1000, 0.004321.Floor(-2) ⇒ 0.0043, 0.004321.Floor(-10) ⇒ 0.004
        /// </summary>
        /// <param name="target"></param>
        /// <param name="digits"></param>
        /// <returns></returns>
        public static decimal FloorEx(this decimal target, int digits)
        {
            if (digits > 0)
            {
                var e = (decimal)Math.Pow(10d, digits);
                return Math.Floor(target * e) / e;
            }
            if (digits < 0)
            {
                //有効桁数が負の値の場合

                digits = Math.Abs(digits);

                if (target < 1)
                {
                    //targetが１より小さい場合

                    var str = target.ToString().Substring(2);
                    var len = str.Length;
                    var len2 = str.TrimStart('0').Length;

                    var pow = (digits >= len2) ? len - len2 + 1 : len - digits;
                    var e = (decimal)Math.Pow(10d, pow);

                    return Math.Floor(target * e) / e;
                }
                else
                {
                    //targetが１より大きい場合

                    var e = 1m;
                    for (int i = 0; i < digits; i++)
                    {
                        if (e * 10m > target) break;
                        e *= 10m;
                    }
                    return Math.Floor(target / e) * e;
                }
            }
            return Math.Floor(target);
        }

        /// <summary>
        /// 指定された角度のサインを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static decimal Sin(this decimal target)
        {
            return (decimal)Math.Sin((double)target);
        }

        /// <summary>
        /// 指定された角度のコサインを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static decimal Cos(this decimal target)
        {
            return (decimal)Math.Cos((double)target);
        }

        /// <summary>
        /// 指定された角度のタンジェントを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static decimal Tan(this decimal target)
        {
            return (decimal)Math.Tan((double)target);
        }

        /// <summary>
        /// サインが指定数となる角度を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static decimal Asin(this decimal target)
        {
            return (decimal)Math.Asin((double)target);
        }

        /// <summary>
        /// コサインが指定数となる角度を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static decimal Acos(this decimal target)
        {
            return (decimal)Math.Acos((double)target);
        }

        /// <summary>
        /// タンジェントが指定数となる角度を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static decimal Atan(this decimal target)
        {
            return (decimal)Math.Atan((double)target);
        }

        /// <summary>
        /// タンジェントが2つの指定された数の商である角度を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="compare"></param>
        /// <returns></returns>
        public static decimal Atan2(this decimal target, decimal target2)
        {
            return (decimal)Math.Atan2((double)target, (double)target2);
        }

        /// <summary>
        /// 負の1(-1)かどうかを判定します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static bool IsMinusOne(this decimal target)
        {
            return target == decimal.MinusOne;
        }

        /// <summary>
        /// 小数点以下の桁数を返します
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static int DecimalLength(this decimal target)
        {
            var strtarget = target.ToString();
            return strtarget.Substring(strtarget.IndexOf(".") + 1).Length;
        }

    }
}
