﻿using System;
using System.Collections.Specialized;
using System.Data;

namespace Framework.Data
{
    /// <summary>
    /// 画面とDB間の橋渡しをするための汎用的な入れ物。ベースは System.Collections.Specialized.NameValueCollection です。
    /// </summary>
    public class MappingData : System.Collections.Specialized.NameValueCollection
    {
        /// <summary>
        /// 空コンストラクタ
        /// </summary>
        public MappingData() { }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="data">元データ</param>
        public MappingData(NameValueCollection data) : base(data) { }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="data">元データ</param>
        public MappingData(MappingData data) : base((NameValueCollection)data) { }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="row">元データ</param>
        public MappingData(DataRow row)
        {
            this.Add(row);
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MappingData(string name, string value)
        {
            base.Add(name, value);
        }

        /// <summary>
        /// データを追加する
        /// </summary>
        /// <param name="data"></param>
        public void Add(MappingData data)
        {
            base.Add((NameValueCollection)data);
        }
        
        /// <summary>
        /// rowのカラム名をname、値をvalueとして追加する。
        /// </summary>
        /// <param name="row"></param>
        public void Add(DataRow row)
        {
            for (int i = 0; i < row.Table.Columns.Count; i++)
            {
                string val = string.Empty;
                DataColumn col = row.Table.Columns[i];
                if (row[i] != DBNull.Value)
                {
                    if (col.DataType == typeof(decimal))
                    {
                        val = ((decimal)row[i]).ToPlaneString();
                    }
                    else if (col.DataType == typeof(bool))
                    {
                        val = ((bool)row[i]) ? BoolString.True : BoolString.False;
                    }
                    else
                    {
                        val = row[i].ToString();
                    }
                }
                base.Add(col.ColumnName, val);
            }
        }

        /// <summary>
        /// 値をNameValueCollectionに格納して返します。
        /// </summary>
        /// <returns></returns>
        public NameValueCollection GetBaseData()
        {
            var baseData = new NameValueCollection();
            foreach (string key in this.AllKeys)
            {
                baseData.Add(key, this[key]);
            }
            return baseData;
        }

        /// <summary>
        /// ColumnNameがKeyと一致するカラムにValueをセットします。
        /// </summary>
        /// <param name="row"></param>
        public void CopyTo(DataRow row)
        {
            var tbl = row.Table;

            foreach (string key in this.Keys)
            {
                if (string.IsNullOrEmpty(key)) continue;

                if (tbl.Columns.Contains(key))
                {
                    string val = base[key];
                    var col = tbl.Columns[key];
                    if (string.IsNullOrEmpty(val))
                    {
                        if (tbl.Columns[key].AllowDBNull)
                        {
                            row[key] = DBNull.Value;
                        }
                        else
                        {
                            row[key] = string.Empty;
                        }
                    }
                    else
                    {
                        if (col.DataType == typeof(decimal))
                        {
                            row[key] = decimal.Parse(val);
                        }
                        else if (col.DataType == typeof(int))
                        {
                            row[key] = int.Parse(val);
                        }
                        else if (col.DataType == typeof(double))
                        {
                            row[key] = double.Parse(val);
                        }
                        else if (col.DataType == typeof(bool))
                        {
                            row[key] = (val == BoolString.True);
                        }
                        else
                        {
                            row[key] = val;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// keyにvalueが複数ある場合に、valueを最初の一つ目の値にする
        /// </summary>
        /// <param name="key"></param>
        public MappingData Uniq(string key)
        {
            var values = this.GetValues(key);
            if (values != null)
            {
                this[key] = values[0];
            }

            return this;
        }

        /// <summary>
        /// valueが複数ある場合に、valueを最初の一つ目の値にする
        /// </summary>
        /// <param name="key"></param>
        public MappingData Uniq()
        {
            foreach (var key in this.AllKeys)
            {
                this[key] = this.GetValues(key)[0];
            }

            return this;
        }

    }
}
