﻿using System.Text;

namespace Framework
{
    /// <summary>
    /// charの拡張メソッドです。
    /// </summary>
    public static class CharExtension
    {
        /// <summary>
        /// 指定回数だけ繰り返した文字列を生成します。
        /// </summary>
        /// <param name="times">繰り返す回数</param>
        /// <exception cref="ArgException">timesに負の数を指定した場合</exception>
        public static string Repeat(this char target, int times)
        {
            if (times < 0)
                throw ArgException.ExpectedGreaterThanEq(0).but("times", times);

            var buf = new StringBuilder(times / 2);
            buf.Append(target);
            var result = new StringBuilder(times);
            for (int i = times; i > 0; i >>= 1)
            {
                if ((i & 1) == 1)
                    result.Append(buf);
                buf.Append(buf);
            }
            return result.ToString();
        }

        /// <summary>
        /// 指定回数だけ繰り返した文字列を生成します。
        /// nullで起動した場合、timesにかかわらずnullを返します。
        /// </summary>
        /// <param name="times">繰り返す回数</param>
        /// <exception cref="ArgException">timesに負の数を指定した場合</exception>
        public static string Repeat(this char? target, int times)
        {
            if (target == null)
                return null;
            return Repeat(target.Value, times);
        }

        /// <summary>
        /// char.MinValue に等しい場合にtrueを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static bool IsMinValue(this char target)
        {
            return (target == char.MinValue);
        }

        /// <summary>
        /// char.MinValue に等しくない場合にtrueを返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static bool IsNotMinValue(this char target)
        {
            return (target != char.MinValue);
        }

        /// <summary>
        /// 全角文字の場合は、true を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static bool IsZenkaku(this char target)
        {
            var cnt = System.Text.Encoding.GetEncoding(932).GetByteCount(new[] { target });
            return (cnt == 2);
        }

        /// <summary>
        /// 半角文字の場合は、true を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        public static bool IsHankaku(this char target)
        {
            return target.IsZenkaku() == false;
        }
    }
}
