﻿using System;

namespace Framework
{
    /// <summary>
    /// メソッドに渡された引数の何れかが不正な場合に送出される例外クラスです。
    /// 標準のSystem.ArgumentExceptionよりも多くの情報を保持しています。
    /// </summary>
    public sealed class ArgException : Exception
    {
        internal readonly string argName;
        internal readonly object argVal;

        internal readonly ExpectedType type;
        internal readonly string expectedName;
        internal readonly object[] expecteds;

        internal readonly string message;

        /// <summary>
        /// 引数の名前と値を指定して、オブジェクトを生成します。
        /// このコンストラクタよりも便利なファクトリメソッドがありますので、
        /// 基本的にはそちらを使用してください。
        /// </summary>
        public ArgException(string name, object val)
        {
            argName = name;
            argVal = val;
            type = ExpectedType.None;
            expectedName = null;
            expecteds = new object[0];
            message = null;
        }

        /// <summary>
        /// メッセージを指定してオブジェクトを生成します。
        /// このコンストラクタよりも便利なファクトリメソッドがありますので、
        /// 基本的にはそちらを使用してください。
        /// </summary>
        public ArgException(string message)
        {
            argName = null;
            argVal = null;
            type = ExpectedType.None;
            expectedName = null;
            expecteds = new object[0];
            this.message = message;
        }

        internal ArgException(string name, object val, ExpectedType type, params object[] expecteds)
        {
            argName = name;
            argVal = val;
            this.type = type;
            this.expectedName = null;
            this.expecteds = expecteds;
            message = null;
        }

        internal ArgException(string name, object val, ExpectedType type, string expectedName, params object[] expecteds)
        {
            argName = name;
            argVal = val;
            this.type = type;
            this.expectedName = expectedName;
            this.expecteds = expecteds;
            message = null;
        }

        /// <summary>
        /// 引数がfromからtoの間にあることを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedBetween(object from, object to)
        {
            return new ExpectedBetweenBuilder(from, to);
        }

        /// <summary>
        /// 引数が指定したもののどれかであることを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedAny(params object[] expecteds)
        {
            return new ExpectedAnyBuilder(expecteds);
        }

        /// <summary>
        /// 引数が指定したものより大きいことを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedGreaterThan(IComparable expected)
        {
            return new ExpectedThresholdBuilder(ExpectedType.GreaterThan, expected);
        }

        /// <summary>
        /// 引数が指定したものより大きいことを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedGreaterThan(string expectedName, IComparable expected)
        {
            return new ExpectedThresholdBuilder(ExpectedType.GreaterThan, expectedName, expected);
        }

        /// <summary>
        /// 引数が指定したもの以上であることを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedGreaterThanEq(IComparable expected)
        {
            return new ExpectedThresholdBuilder(ExpectedType.GreaterThanEq, expected);
        }

        /// <summary>
        /// 引数が指定したもの以上であることを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedGreaterThanEq(string expectedName, IComparable expected)
        {
            return new ExpectedThresholdBuilder(ExpectedType.GreaterThanEq, expectedName, expected);
        }

        /// <summary>
        /// 引数が指定したものより小さいことを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedLessThan(IComparable expected)
        {
            return new ExpectedThresholdBuilder(ExpectedType.LessThan, expected);
        }

        /// <summary>
        /// 引数が指定したものより小さいことを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedLessThan(string expectedName, IComparable expected)
        {
            return new ExpectedThresholdBuilder(ExpectedType.LessThan, expectedName, expected);
        }

        /// <summary>
        /// 引数が指定したもの以下であることを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedLessThanEq(IComparable expected)
        {
            return new ExpectedThresholdBuilder(ExpectedType.LessThanEq, expected);
        }

        /// <summary>
        /// 引数が指定したもの以下であることを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedLessThanEq(string expectedName, IComparable expected)
        {
            return new ExpectedThresholdBuilder(ExpectedType.LessThanEq, expectedName, expected);
        }

        /// <summary>
        /// 引数がnullでないことを望んでいました。
        /// </summary>
        public static IUnexpectedNullBuilder UnexpectedNull()
        {
            return new UnexpectedNullBuilder();
        }

        /// <summary>
        /// 引数が指定された値ではないことを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder Unexpected(object unexpected)
        {
            return new UnexpectedBuilder(unexpected);
        }

        /// <summary>
        /// 引数が指定された変数名と等しくないことを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder UnexpectedEq(string unexpectedValName)
        {
            return new UnexpectedEqBuilder(unexpectedValName);
        }

        /// <summary>
        /// 引数が指定されたフォーマットであることを望んでいました。
        /// </summary>
        public static IArgExceptionBuilder ExpectedFormatIs(string format)
        {
            return new ExpectedFormatBuilder(format);
        }

        public override string Message
        {
            get
            {
                return message ?? type.ErrorMessage(argName, argVal, expectedName, expecteds);
            }
        }
    }
}
